package com.digiwin.athena.datamap.mechanism.convert.parsers.fieldValueType;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.datamap.mechanism.convert.CustomFuncResult;
import com.digiwin.athena.datamap.mechanism.convert.MechanismParserFactory;
import com.digiwin.athena.mechanism.common.MechanismEnum;
import com.digiwin.athena.mechanism.common.MechanismVariable;
import com.digiwin.athena.mechanism.widgets.config.MechanismConditionConfig;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @Description 字符串类型解析器
 * @Author liugl
 * @Date 2024/7/22 10:29
 **/
@Service
@Slf4j
public class StringParser implements FieldValueTypeParser {

    @Autowired
    MechanismParserFactory mechanismParserFactory;

    @Override
    public boolean accept(String dataType) {
        return MechanismEnum.FieldValueType.STRING.getCode().equalsIgnoreCase(dataType);
    }

    @Override
    public JSONArray parse(MechanismConditionConfig conditionConfig, JSONArray metaDataList) {
        // 处理全局类型的判断值，例如自定义值，机制函数，枚举值
        CustomFuncResult rightCondResult = mechanismParserFactory.getRightCondParserByType(conditionConfig.getRightCondType())
                .preParse(conditionConfig, metaDataList);
        if (!rightCondResult.getIsNeedNextJudge()) {
            return rightCondResult.getMetaDataList();
        }
        if (MechanismEnum.ConditionType.HIGH_GRADE.getCode().equalsIgnoreCase(conditionConfig.getType())) {
            return parseHighGrade(conditionConfig, metaDataList);
        }
        return parsePrimary(conditionConfig, metaDataList);
    }

    /**
     * 基础条件比较
     * @param conditionConfig
     * @param metaDataList
     * @return 返回结果
     */
    private JSONArray parsePrimary(MechanismConditionConfig conditionConfig, JSONArray metaDataList) {
        JSONArray result = new JSONArray();
        String rightToValue;
        try {
            MechanismVariable rightTo = conditionConfig.getRight().getTo();
            rightToValue = rightTo.getValue().toString();
        } catch (Exception e) {
            log.error("StringParser.parse 右侧条件判断值转换失败，入参：{}", conditionConfig);
            // 如果条件值不是String或者是空值则不比较，直接返回空数据
            return result;
        }
        String dataName = conditionConfig.getLeft().getData_name();
        List<Object> resultList = metaDataList.stream().filter(v -> {
            JSONObject metaDataObj = JSON.parseObject(JSON.toJSONString(v));
            String sourceValue = null;
            try {
                sourceValue = metaDataObj.getString(dataName);
            } catch (Exception e) {
                log.warn("StringParser.parse 表格数据转换失败，数据为：{}", v);
                return Boolean.FALSE;
            }
            // String类型只有一种判断方式 相等 equal
            return rightToValue.equalsIgnoreCase(sourceValue);
        }).collect(Collectors.toList());
        return JSONArray.parseArray(JSON.toJSONString(resultList));
    }

    /**
     * 高级条件 自定义函数比较
     *
     * @param conditionConfig
     * @param metaDataList
     * @return 返回结果
     */
    private JSONArray parseHighGrade(MechanismConditionConfig conditionConfig, JSONArray metaDataList) {
        JSONArray resultArray = new JSONArray();
        String functionValueType = conditionConfig.getLeft().getFunctionValueType();
        Object functionValue = conditionConfig.getLeft().getFunctionValue();
        Object value = conditionConfig.getRight().getTo().getFunctionValue();
        // 目前自定义函数输出类型只有数值和boolean类型
        if (MechanismEnum.FieldValueType.NUMERIC.getCode().equalsIgnoreCase(functionValueType)) {
            try {
                BigDecimal functionValueTemp = new BigDecimal(functionValue.toString());
                BigDecimal valueTemp = new BigDecimal(value.toString());
                if (functionValueTemp.compareTo(valueTemp) == 0) {
                    resultArray.addAll(metaDataList);
                    return resultArray;
                }
            } catch (Exception e) {
                log.error("StringParser.compareCondition 失败，e={}", e);
                return null;
            }
            return null;
        }
        if (ObjectUtils.nullSafeEquals(functionValue.toString(), value.toString())) {
            resultArray.addAll(metaDataList);
            return resultArray;
        }
        return null;
    }

    @Override
    public Boolean isBothEqual(Object source, Object targetLeft, Object targetRight, String opType) {
        try {
            String sourceStr = source.toString();
            String targetRightStr = targetRight.toString();
            return StringUtils.equalsIgnoreCase(sourceStr, targetRightStr);
        } catch (Exception e) {
            log.error("StringParser.isBothEqual 失败，e={}", e);
            return Boolean.FALSE;
        }
    }
}
