package com.digiwin.athena.datamap.mechanism.limithandler.parsers.limitCondition;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.datamap.mechanism.convert.parsers.fieldValueType.NumericParser;
import com.digiwin.athena.datamap.spi.AudcService;
import com.digiwin.athena.datamap.spi.DatamapEocService;
import com.digiwin.athena.mechanism.bo.LimitAbilityBo;
import com.digiwin.athena.mechanism.common.MechanismEnum;
import com.digiwin.athena.mechanism.dto.MechanismApiInfoDTO;
import com.digiwin.athena.mechanism.widgets.condition.PersonnelCondition;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

@Service
@Slf4j
public class DataMapPersonnelConditionParser implements LimitConditionParser {

    @Autowired
    DatamapEocService datamapEocService;

    @Autowired
    AudcService acduService;

    @Autowired
    NumericParser numericParser;

    private static final String EOC_DUTY_LIST_URL = "/api/eoc/v2/duty/list?pageSize=99999";

    private static final String EOC_PROFICIENCY_URL = "/api/v1/user/track/search";

    @Override
    public boolean accept(String type) {
        return MechanismEnum.LimitConditionType.PERSONNEL.getCode().equalsIgnoreCase(type);
    }

    @Override
    public Boolean parse(MechanismApiInfoDTO apiInfoDTO) {
        LimitAbilityBo bo = JSONObject.parseObject(JSON.toJSONString(apiInfoDTO.getAbility()), LimitAbilityBo.class);
        PersonnelCondition personnelCondition = bo.getPersonnelCondition();
        if (personnelCondition == null) {
            return Boolean.FALSE;
        }
        String performerType = personnelCondition.getPerformerType();
        if (MechanismEnum.PersonnelConditionType.PROFICIENCY.getCode().equalsIgnoreCase(performerType)) {
            return parserProficiency(bo);
        }
        if (MechanismEnum.PersonnelConditionType.DUTY.getCode().equalsIgnoreCase(performerType)) {
            return parserDuty(bo);
        }
        if (MechanismEnum.PersonnelConditionType.STAFF.getCode().equalsIgnoreCase(performerType)) {
            return parserStaff(bo);
        }
        return Boolean.FALSE;
    }

    /**
     * 处理熟练度
     *
     * @return
     */
    private Boolean parserProficiency(LimitAbilityBo bo) {
        // {'workType':'TASK','workCode':'maintenanceProjectApproval_DTD','operations':['OPEN'],'userId':userId,'tenantId':tenantId}
        Map<String, Object> profile = DWServiceContext.getContext().getProfile();
        Map<String, Object> params = new HashMap<>();
        params.put("workType", bo.getPersonnelCondition().getWorkType());
        params.put("workCode", bo.getPersonnelCondition().getWorkCode());
        params.put("operations", bo.getPersonnelCondition().getOperations());
        params.put("userId", profile.get("userId"));
        params.put("tenantId", profile.get("tenantId"));
        log.info("查询熟练度开始，入参为{},{}", EOC_PROFICIENCY_URL, params);
        Map proficiencyMap = acduService.requestAudc(EOC_PROFICIENCY_URL, "post", params);
        log.info("查询熟练度，结果为{}", proficiencyMap);
        if (!CollectionUtils.isEmpty(proficiencyMap) && Objects.equals(proficiencyMap.get("status").toString(), HttpStatus.OK.toString())) {
            JSONObject response = (JSONObject) proficiencyMap.get("response");
            BigDecimal count = response.getBigDecimal("count");
            PersonnelCondition condition = bo.getPersonnelCondition();
            return numericParser.isSatisfyCond(condition.getProficiencyType().name(), count, ObjectUtils.isEmpty(condition.getFromValue()) ? null : new BigDecimal(condition.getFromValue().toString()),
                    new BigDecimal(condition.getToValue().toString()));
        }
        return Boolean.FALSE;
    }

    /**
     * 处理职能
     *
     * @return
     */
    private Boolean parserDuty(LimitAbilityBo bo) {
        log.info("查询职能开始，入参为{}", EOC_DUTY_LIST_URL);
        Map dutyMap = datamapEocService.requestEoc(EOC_DUTY_LIST_URL, "get", new HashMap<>());
        log.info("查询职能结束，结果为{}", dutyMap);
        if (!CollectionUtils.isEmpty(dutyMap) && Objects.equals(dutyMap.get("code").toString(), HttpStatus.OK.toString())) {
            Map<String, Map<String, Object>> dataMap = (Map<String, Map<String, Object>>) dutyMap.get("data");
            if (!CollectionUtils.isEmpty(dataMap)) {
                List<Map<String, Object>> dataList = (List<Map<String, Object>>) dataMap.get("list");
                List<String> dutyIdList = dataList.stream().map(data -> data.get("id").toString()).collect(Collectors.toList());
                if (null == bo.getPersonnelCondition().getToValue() || bo.getPersonnelCondition().getToValue().toString().isEmpty()) {
                    return false;
                }
                List<String> targetDutyIds = handleList(bo.getPersonnelCondition().getToValue().toString());
                for (String each : targetDutyIds) {
                    if (dutyIdList.contains(each)) {
                        return true;
                    }
                }
            }
        }
        return Boolean.FALSE;
    }

    /**
     * 处理人员
     *
     * @return result
     */
    private Boolean parserStaff(LimitAbilityBo bo) {
        log.info("获取用户设置的人员:{}", bo.getPersonnelCondition().getToValue());
        if (null == bo.getPersonnelCondition().getToValue() || bo.getPersonnelCondition().getToValue().toString().isEmpty()) {
            return false;
        }
        List<String> staff = handleList(bo.getPersonnelCondition().getToValue().toString());
        Boolean result = staff.contains(DWServiceContext.getContext().getProfile().get("userId").toString());
        log.info("人员条件判断结果:{}", result);
        return result;
    }

    private static List<String> handleList(String performerValueStaff) {
        String[] emails = performerValueStaff.replace("[", "").replace("]", "").replace("\"", "").split(",");
        return Arrays.asList(emails);
    }
}
