package com.digiwin.athena.datamap.service;

import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.service.AllowAnonymous;
import com.digiwin.app.service.DWService;
import com.digiwin.app.service.restful.DWRequestMapping;
import com.digiwin.app.service.restful.DWRequestMethod;
import com.digiwin.app.service.restful.DWRestfulService;
import com.digiwin.athena.annotation.ScheduledCall;
import com.digiwin.athena.datamap.povo.CheckAppExpirePo;
import com.digiwin.athena.kg.activity.Activity;
import com.digiwin.athena.kg.domain.OpenWindowDefinition;
import com.digiwin.athena.kg.dto.AppRelationBatchPo;
import com.digiwin.athena.kg.domain.ApplicationCustomizedConfig;
import com.digiwin.athena.kg.dto.ApplicationRelationQueryDTO;
import com.digiwin.athena.domain.core.app.Application;
import com.digiwin.athena.domain.core.app.ApplicationRelation;
import com.digiwin.athena.domain.core.app.FunctionCtrl;
import org.springframework.web.bind.annotation.RequestBody;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 应用定义接口
 *
 * @javadoc
 */
@DWRestfulService
public interface IAppService extends DWService {
    /**
     * 根据应用code和类型查询应用下关联关系
     *
     * @param type 组件类型
     * @param code 组件code
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationByTypeAndCode", method = {DWRequestMethod.GET})
    ApplicationRelation getApplicationByTypeAndCode(String type, String code) throws DWBusinessException;

    /**
     * 根据应用code和类型查询应用下关联关系
     *
     * @param type 组件类型
     * @param code 组件code
     * @param lang 语言
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationByTypeAndCodeAndLang1", method = {DWRequestMethod.GET})
    ApplicationRelation getApplicationByTypeAndCodeAndLang1(String type, String code, String lang) throws DWBusinessException;

    /**
     * 根据组件类型，组件code模糊查询应用下关联关系
     *
     * @param queryDTO type 组件类型 codes 组件code集合 activityMap 任务信息
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationsMapByTypeAndCodesAndActivityMap", method = {DWRequestMethod.POST})
    Map<String, List<ApplicationRelation>> postApplicationsMapByTypeAndCodesAndActivityMap(@RequestBody ApplicationRelationQueryDTO queryDTO) throws DWBusinessException;

    /**
     * 根据组件类型，组件code模糊查询应用下关联关系
     *
     * @param queryDTO type  组件类型 codes 组件code集合
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationsMapByTypeAndCodes", method = {DWRequestMethod.POST})
    Map<String, List<ApplicationRelation>> postApplicationsMapByTypeAndCodes(@RequestBody ApplicationRelationQueryDTO queryDTO) throws DWBusinessException;

    /**
     * 根据组件类型，组件code查询应用下关联关系
     *
     * @param queryDTO type     组件类型  codes 组件code集合
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationByTypeAndCodeList", method = {DWRequestMethod.POST})
    List<ApplicationRelation> postApplicationByTypeAndCodeList(@RequestBody ApplicationRelationQueryDTO queryDTO) throws DWBusinessException;

    /**
     * 根据组件类型，组件code查询应用下关联关系
     *
     * @param queryDTO type     组件类型 subType  二级组件类型 codes 组件code集合
     * @return 应用下组件关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationByTypeAndSubTypeAndCodeList", method = {DWRequestMethod.POST})
    List<ApplicationRelation> postApplicationByTypeAndSubTypeAndCodeList(@RequestBody ApplicationRelationQueryDTO queryDTO) throws DWBusinessException;

    /**
     * 根据组件类型，组件code查询应用下组件关系
     *
     * @param type 组件类型
     * @param code 组件code集合
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationsByTypeAndCode", method = {DWRequestMethod.GET})
    List<ApplicationRelation> getApplicationsByTypeAndCode(String type, String code) throws DWBusinessException;

    /**
     * 根据应用code集合查询应用
     *
     * @param codes 组件code集合
     * @return 应用
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationsByCodes", method = {DWRequestMethod.POST})
    List<Application> postApplicationsByCodes(Collection<String> codes) throws DWBusinessException;

    /**
     * 根据应用code查询应用
     *
     * @param code 组件code
     * @return 应用
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationByCode", method = {DWRequestMethod.GET})
    Application getApplicationByCode(String code) throws DWBusinessException;

    /**
     * 根据应用code和租户id查询应用
     *
     * @param code     组件code
     * @param tenantId 租户id
     * @return
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationByCodeAndTenantId", method = {DWRequestMethod.GET})
    Application getApplicationByCodeAndTenantId(String code, String tenantId) throws DWBusinessException;

    /**
     * 根据业务组件code查询应用扩展信息
     *
     * @param code 组件code
     * @return 应用扩展信息
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationFunctionCtrlByBizCode", method = {DWRequestMethod.GET})
    FunctionCtrl getApplicationFunctionCtrlByBizCode(String code) throws DWBusinessException;

    /**
     * 根据应用code查询应用扩展信息
     *
     * @param code 应用code
     * @return 应用扩展信息
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationFunctionCtrlByAppCode", method = {DWRequestMethod.GET})
    FunctionCtrl getApplicationFunctionCtrlByAppCode(String code) throws DWBusinessException;

    // TODO liugla 2024/10/22 11:31 注释

    /**
     * ApplicationRecommendation表的作用
     *
     * @return
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationRecommendations", method = {DWRequestMethod.GET})
    List<Map> getApplicationRecommendations() throws DWBusinessException;

    /**
     * 根据组件类型，组件code查询应用下关联关系，并根据应用code分组
     *
     * @param queryDTO type  组件类型 codes 组件code lang  语言
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationRelationsMapByTypeAndCodesAndLang", method = {DWRequestMethod.POST})
    Map<String, List<ApplicationRelation>> postApplicationRelationsMapByTypeAndCodesAndLang(@RequestBody ApplicationRelationQueryDTO queryDTO) throws DWBusinessException;

    /**
     * 根据组件类型，组件code查询应用下关联关系集合
     *
     * @param type 组件类型
     * @param code 组件code
     * @param lang 语言
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationRelationsByTypeAndCodeAndLang", method = {DWRequestMethod.GET})
    List<ApplicationRelation> getApplicationRelationsByTypeAndCodeAndLang(String type, String code, String lang) throws DWBusinessException;

    /**
     * 根据组件类型，组件code查询应用下关联关系
     *
     * @param type 组件类型
     * @param code 组件code
     * @param lang 语言
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationByTypeAndCodeAndLang", method = {DWRequestMethod.GET})
    ApplicationRelation getApplicationByTypeAndCodeAndLang(String type, String code, String lang) throws DWBusinessException;

    /**
     * 根据组件code查询应用下关联关系
     *
     * @param queryDTO codes     组件code localeStr 语言
     * @return 应用下关联关系 {组件code, 应用下关联关系}
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationRelationsMapByApplicationAndLang", method = {DWRequestMethod.POST})
    Map<String, ApplicationRelation> postApplicationRelationsMapByApplicationAndLang(@RequestBody ApplicationRelationQueryDTO queryDTO) throws DWBusinessException;

    /**
     * 根据组件code查询应用下关联关系
     *
     * @param code      组件code
     * @param localeStr 语言
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationRelationByApplicationAndLang", method = {DWRequestMethod.GET})
    ApplicationRelation getApplicationRelationByApplicationAndLang(String code, String localeStr) throws DWBusinessException;

    /**
     * 根据租户id获取应用许可密钥
     *
     * @param tenantId 租户id
     * @return 许可密钥加密串
     * @throws Exception
     */
    @DWRequestMapping(path = "/app/licenseKey", method = {DWRequestMethod.GET})
    @AllowAnonymous
    Object getLicenseKey(String tenantId) throws Exception;

    /**
     * 根据租户id获取应用许可密钥
     *
     * @param tenantId    租户id
     * @param requestFrom 数据来源
     * @param key         校验许可的md值
     * @param timestamp   有效时间
     * @return 许可密钥加密串
     * @throws Exception
     */
    @DWRequestMapping(path = "/app/licenseKeyByTenantIdAndRequestFromAndKeyAndTime", method = {DWRequestMethod.GET})
    @AllowAnonymous
    Object getLicenseKeyByTenantIdAndRequestFromAndKeyAndTime(String tenantId, String requestFrom, String key, long timestamp) throws Exception;

    /**
     * 设置当前用户许可
     *
     * @return 操作结果
     * @throws Exception
     */
    @DWRequestMapping(path = "/app/setLicenseKey", method = {DWRequestMethod.POST})
    Object postSetLicenseKey() throws Exception;

    /**
     * 给指定用户设置许可
     *
     * @param tenantId 租户id
     * @return 操作结果
     * @throws Exception
     */
    @DWRequestMapping(path = "/app/setLicenseKeyByTenantId", method = {DWRequestMethod.POST})
    @AllowAnonymous
    Object postSetLicenseKeyByTenantId(String tenantId) throws Exception;

    /**
     * 根据应用code获取应用
     *
     * @param code 应用code
     * @return 应用
     * @throws Exception
     */
    @DWRequestMapping(path = "/app", method = {DWRequestMethod.GET})
    Object get(String code) throws Exception;

    /**
     * 根据应用code获取应用（内涵侦测规则和基础参数）
     *
     * @param applicationCodeList 应用code
     * @return 应用
     * @throws Exception
     */
    @DWRequestMapping(path = "/app/list", method = {DWRequestMethod.POST})
    Object postList(Set<String> applicationCodeList) throws Exception;

    /**
     * 根据应用名称和标签查询应用集合
     *
     * @param queryDTO applicationName 应用名称 tags 标签
     * @return 应用集合
     * @throws Exception
     */
    @DWRequestMapping(path = "/app/byNameAndTags", method = {DWRequestMethod.POST})
    Object postByNameAndTags(@RequestBody ApplicationRelationQueryDTO queryDTO) throws Exception;


    /**
     * 根据应用code获取应用
     *
     * @param code 应用code
     * @return 应用
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationInfo", method = {DWRequestMethod.GET})
    Application getApplicationInfo(String code) throws DWBusinessException;

    /**
     * 根据应用code从应用组件关系表获取任务
     *
     * @param appCode 应用code
     * @return 任务code
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/taskCode", method = {DWRequestMethod.GET})
    List<String> getTaskCode(String appCode) throws DWBusinessException;

    /**
     * 根据应用code从应用组件关系表获取机制
     *
     * @param appCode 应用code
     * @return 机制code
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/mechanismCode", method = {DWRequestMethod.GET})
    List<String> getMechanismCode(String appCode) throws DWBusinessException;

    /**
     * 根据应用code从应用组件关系表获取侦测
     *
     * @param appCode 应用code
     * @return 侦测code
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/monitorRuleCode", method = {DWRequestMethod.GET})
    List<String> getMonitorRuleCode(String appCode) throws DWBusinessException;

    /**
     * 根据组件code和类型获取应用组件关系
     *
     * @param applicationRelationQueryDTO 查询参数，组件code和类型
     * @return 应用下关联关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationRelationsByCodesAndType", method = {DWRequestMethod.POST})
    List<ApplicationRelation> postApplicationRelationsByCodesAndType(@RequestBody ApplicationRelationQueryDTO applicationRelationQueryDTO) throws DWBusinessException;

    /**
     * 根据任务code获取组件应用关系
     *
     * @param code 任务code
     * @return 应用组件关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/appCodeByTaskCode", method = {DWRequestMethod.GET})
    List<ApplicationRelation> getAppCodeByTaskCode(String code) throws DWBusinessException;

    /**
     * 根据机制code获取应用组件关系
     *
     * @param code 机制code
     * @return 应用组件关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/appCodeByMechanismCode", method = {DWRequestMethod.GET})
    List<ApplicationRelation> getAppCodeByMechanismCode(String code) throws DWBusinessException;

    /**
     * 根据侦测code查询应用组件关系，没有数据再查找Application中直接配置再paramDescription中的数据
     *
     * @param code 侦测code
     * @return 应用组件关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/appCodeByMonitorRuleCode", method = {DWRequestMethod.GET})
    List<ApplicationRelation> getAppCodeByMonitorRuleCode(String code) throws DWBusinessException;

    /**
     * 根据应用code和租户版本获取应用下的组件关系
     *
     * @param queryDTO codes      应用code tenantVersion 租户版本
     * @return 返回应用组件关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationRelationByCodesAndTenantVersion", method = {DWRequestMethod.POST})
    List<ApplicationRelation> postApplicationRelationByCodesAndTenantVersion(@RequestBody ApplicationRelationQueryDTO queryDTO) throws DWBusinessException;

    /**
     * 根据应用code和租户版本获取应用下的组件关系
     *
     * @param queryDTO appCodes 应用code  codes 组件code
     * @return 返回应用组件关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationRelationByAppCodesAndCodesAndTypeAndSubType", method = {DWRequestMethod.POST})
    List<ApplicationRelation> postApplicationRelationByAppCodesAndCodesAndTypeAndSubType(@RequestBody ApplicationRelationQueryDTO queryDTO) throws DWBusinessException;

    /**
     * 根据应用code获取应用下的组件关系
     *
     * @param appCodes 应用code
     * @return 应用组件关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationRelation", method = {DWRequestMethod.POST})
    List<ApplicationRelation> postApplicationRelation(List<String> appCodes) throws DWBusinessException;

    /**
     * 根据应用code获取流程（现已废弃）
     *
     * @param appCode 应用code
     * @return flowCode
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/flowCode", method = {DWRequestMethod.GET})
    List<String> getFlowCode(String appCode) throws DWBusinessException;

    /**
     * 根据flow code获取应用组件关系数据（现已废弃）
     *
     * @param code 组件flow code
     * @return 应用组件关系数据
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/appCodeByFlowCode", method = {DWRequestMethod.GET})
    List<ApplicationRelation> getAppCodeByFlowCode(String code) throws DWBusinessException;

    /**
     * 根据应用code获取参数变量，数据来源ApplicationRelation
     *
     * @param appCode 应用code
     * @return 返回参数变量
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/variableCode", method = {DWRequestMethod.GET})
    List<String> getVariableCode(String appCode) throws DWBusinessException;

    /**
     *这个方法不要把@AllowAnonymous去除，租户初始化时获取系统有哪些应用code的方法，如果想要有版本信息，请另写方法
     */
    @AllowAnonymous
    @ScheduledCall
    @DWRequestMapping(path = "/app/allAppCode", method = {DWRequestMethod.GET})
    List<String> getAllAppCode() throws DWBusinessException;

    /**
     * 获取非公共的应用
     *
     * @return 应用code
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/notCommonAppCode", method = {DWRequestMethod.GET})
    List<String> getNotCommonAppCode() throws DWBusinessException;



    /**
     * 根据组件code和类型查询应用组件关系
     *
     * @param type 组件类型
     * @param code 组件code
     * @return 应用下关联关系
     * @throws Exception
     */
    @DWRequestMapping(path = "/app/appCodeByTypeAndCode", method = {DWRequestMethod.GET})
    List<ApplicationRelation> getAppCodeByTypeAndCode(String type, String code) throws Exception;

    /**
     * 根据应用code和组件类型获取范式数据
     *
     * @param type    组件类型
     * @param appCode 应用code
     * @return 应用下组件数据
     * @throws Exception
     */
    @DWRequestMapping(path = "/app/paradigmByTypeAndAppCode", method = {DWRequestMethod.GET})
    List<ApplicationRelation> getParadigmByTypeAndAppCode(String type, String appCode) throws Exception;

    /**
     * 通过参数的开窗条件列名称查找开窗信息
     *
     * @param openWin 开窗条件列名称
     * @return 开窗信息
     * @throws Exception
     */
    @DWRequestMapping(path = "/app/openWinInfo", method = {DWRequestMethod.POST})
    Map<String, Map<String, OpenWindowDefinition>> postOpenWinInfo(Map<String, Map<String, String>> openWin) throws Exception;

    /**
     * 根据应用获取应用code
     *
     * @param application 应用
     * @return 应用code
     * @throws Exception
     */
    @DWRequestMapping(path = "/app/appCodeByApplication", method = {DWRequestMethod.GET})
    String getAppCodeByApplication(String application) throws Exception;

    /**
     * 根据应用code和类型获取应用下关联关系
     *
     * @param type    组件类型
     * @param appCode 应用code
     * @return 返回应用组件关系
     * @throws Exception
     */
    @AllowAnonymous
    @ScheduledCall
    @DWRequestMapping(path = "/app/applicationRelationByTypeAndAppCode", method = {DWRequestMethod.GET})
    List<ApplicationRelation> getApplicationRelationByTypeAndAppCode(String type, String appCode) throws Exception;

    /**
     * 根据应用code，类型和租户id获取应用下关联关系
     *
     * @param type    组件类型
     * @param appCode 应用code
     * @param version 租户版本
     * @return 返回应用下关联关系
     * @throws Exception
     */
    @DWRequestMapping(path = "/app/applicationRelationByTypeAndAppCodeAndVersion", method = {DWRequestMethod.GET})
    List<ApplicationRelation> getApplicationRelationByTypeAndAppCodeAndVersion(String type, String appCode, String version) throws Exception;

    /**
     * 根据应用code，类型批量获取应用下关联关系
     *
     * @param queryDTO type     组件类型 appCodes 应用code
     * @return 应用下关联关系
     * @throws Exception
     */
    @DWRequestMapping(path = "/app/applicationRelationByTypeAndAppCodes", method = {DWRequestMethod.POST})
    List<ApplicationRelation> postApplicationRelationByTypeAndAppCodes(@RequestBody ApplicationRelationQueryDTO queryDTO) throws Exception;

    /**
     * 根据应用code，类型批量获取应用下关联关系
     *
     * @param queryDTO type     组件类型  appCodes 应用code
     * @return 返回应用下关联关系
     * @throws Exception
     */
    @DWRequestMapping(path = "/app/queryApplicationRelationByTypeAndAppCode", method = {DWRequestMethod.POST})
    List<ApplicationRelation> postQueryApplicationRelationByTypeAndAppCode(@RequestBody ApplicationRelationQueryDTO queryDTO) throws Exception;

    /**
     * 根据应用code，组件类型和租户id获取应用组件关系
     *
     * @param tenantId 租户id
     * @param type     组件类型
     * @param appCode  应用code
     * @return 应用下租户关系
     */
    @DWRequestMapping(path = "/app/applicationRelationByTenantIdAndTypeAndAppCode", method = {DWRequestMethod.GET})
    List<ApplicationRelation> getApplicationRelationByTenantIdAndTypeAndAppCode(String tenantId, String type, String appCode) throws DWBusinessException;

    /**
     * 根据应用code + 组件code + 类型更新应用下关联关系
     *
     * @param request 参数请求
     * @return 操作的数据条数
     * @throws Exception
     */
    @DWRequestMapping(path = "/app/appRelationUpdate", method = {DWRequestMethod.POST})
    Object postAppRelationUpdate(AppRelationBatchPo request) throws Exception;

    /**
     * 根据类型获取应用组件关系
     *
     * @param type 组件类型
     * @return 应用组件关系
     */
    @DWRequestMapping(path = "/app/applicationRelationByType", method = {DWRequestMethod.GET})
    List<ApplicationRelation> getApplicationRelationByType(String type) throws Exception;
// TODO liugla 2024/10/22 14:40 ApplicationCustomizedConfig 表的作用
    /**
     * @return
     */
    @DWRequestMapping(path = "/app/configByApplicationCustomized", method = {DWRequestMethod.POST})
    List<ApplicationCustomizedConfig> postConfigByApplicationCustomized() throws DWBusinessException;

    /**
     * 查询应用客制插件配置-分组时加租户ID维度
     * @return
     */
    @DWRequestMapping(path = "/app/configByGroupApplicationCustomized", method = {DWRequestMethod.POST})
    List<ApplicationCustomizedConfig> postConfigByGroupApplicationCustomized() throws DWBusinessException;

    /**
     * 获取所有的应用
     *
     * @return 应用
     */
    @DWRequestMapping(path = "/app/allApplications", method = {DWRequestMethod.POST})
    @AllowAnonymous
    List<Application> postAllApplications();

    /**
     * 根据组件code和类型获取应用组件关系
     *
     * @param tenantId 租户id
     * @param type     组件类型
     * @param code     组件code
     * @return 组件应用关系
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationRelationByTypeAndCode", method = {DWRequestMethod.POST})
    @AllowAnonymous
    Object postApplicationRelationByTypeAndCode(String tenantId, String type, String code) throws DWBusinessException;

    /**
     * 保存应用关联关系数据
     * @param request 保存数据
     * @return
     * @throws DWBusinessException
     */
    @DWRequestMapping(path = "/app/applicationRelationByDDSM", method = {DWRequestMethod.POST})
    Object postApplicationRelationByDDSM(AppRelationBatchPo request) throws DWBusinessException;


    /**
     * 该方法给敏捷数据使用
     * 根据应用code列表查询应用名称信息
     *
     * @param codes 应用code列表
     * @param appType  应用类型
     */
    @DWRequestMapping(path = "/app/queryApplicationsNameByCodeList", method = {DWRequestMethod.POST})
    List<Map<String, String>> postQueryApplicationsNameByCodeList(List<String> codes, Integer appType) throws DWBusinessException;

    /**
     * 根据机制code获取应用
     *
     *
     * @param mechanismCode 机制code
     * @return 应用
     * @throws DWBusinessException exception
     */
    @DWRequestMapping(path = "/app/appByMechanismCode", method = {DWRequestMethod.GET})
    Application getAppByMechanismCode(String mechanismCode) throws DWBusinessException;

    /**
     * 根据作业id去查询应用
     * @param activityId 作业id
     * @return 应用
     * @throws DWBusinessException exception
     */
    @DWRequestMapping(path = "/app/appByActivityId", method = {DWRequestMethod.GET})
    Application getAppByActivityId(String activityId) throws DWBusinessException;

    @DWRequestMapping(path = "/app/checkAppExpire", method = {DWRequestMethod.POST})
    Object postCheckAppExpire(CheckAppExpirePo po) throws DWBusinessException;
}