package com.digiwin.athena.datamap.service.impl;

import com.alibaba.fastjson.JSONPath;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.container.exceptions.DWException;
import com.digiwin.athena.kmservice.aspect.MyExceptionHandler;
import com.digiwin.athena.kmservice.constants.KnowledgeGraphDb;
import com.digiwin.athena.kmservice.action.execution.ProductNameResolver;
import com.digiwin.athena.datamap.povo.ActivityInputData;
import com.digiwin.athena.datamap.povo.ViewRequest;
import com.digiwin.athena.repository.neo4j.ActivityRepository;
import com.digiwin.athena.datamap.service.IActivityService;
import com.digiwin.athena.datamap.service.IViewService;
import com.digiwin.athena.datamap.service.inner.DataMapPickService;
import com.digiwin.athena.datamap.spi.IKnowledgeGraphService;
import com.digiwin.athena.datamap.spi.DataMapKgService;
import com.digiwin.athena.domain.core.Activity;
import com.digiwin.athena.domain.core.Task;
import com.digiwin.athena.domain.definition.actions.DataEntityMetadataDTO;
import com.digiwin.athena.kg.monitorRule.ConditionOperatorTypeEnum;
import com.digiwin.athena.kg.monitorRule.DynamicCondition;
import com.digiwin.athena.kg.monitorRule.DynamicConditionTypeEnum;
import com.digiwin.athena.dto.ActivityBaseInfoVO;
import com.digiwin.athena.dto.ActivityVisibleConfig;
import com.digiwin.athena.kmservice.utils.ServiceUtils;
import com.mongodb.client.model.Filters;
import lombok.extern.slf4j.Slf4j;
import org.bson.conversions.Bson;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;

@Slf4j
@MyExceptionHandler
@Service
public class DataMapActivityService implements IActivityService {

    private static final String ACTIVITY = "activity";

    private static final String PRODUCT_CODE = "productCode";

    @Autowired
    DataMapTaskService dataMapTaskService;

    @Autowired
    private DataMapPickService dataPickService;

    @Autowired
    private ActivityRepository activityRepository;

    @Autowired
    private ProductNameResolver productNameResolver;

    @Autowired
    private IKnowledgeGraphService knowledgeGraphService;

    @Autowired
    private DataMapKgService dataMapKgService;

    @Autowired
    private IViewService viewService;

    @Override
    public Object postSaveActivity(Activity activity) {
        dataPickService.save(activity, Activity.class);
        return activity;
    }

    @Override
    public Map<String, Object> getInputData(String activityId) throws DWBusinessException {
        Task task = dataPickService.findByCode(activityId, Task.class);
        if (null != task && null != task.getInputData()) {
            Map<String, Object> result = new HashMap<>();
            result.put(ACTIVITY, activityId);
            result.put("inputData", task.getInputData());
            return result;
        }
        return null;
    }

    @Deprecated
    @Override
    public Map<String, Object> getInputData(String taskId, String activityId) throws DWBusinessException {
        String tenantId = ServiceUtils.getTenantId();
        String tenantVersion = dataPickService.tenantVersion();
        // TODO inputData 在 neo4j 里的应该没用了
        Optional<ActivityInputData> inputData = activityRepository.getActivityInputData(tenantId, taskId, activityId, tenantVersion);
        if (inputData.isPresent()) {
            ActivityInputData data = inputData.get();
            data.getInputData().setField(new ArrayList<>(data.getFields()));
            Map<String, Object> result = new HashMap<>();
            result.put(ACTIVITY, activityId);
            result.put("inputData", DataEntityMetadataDTO.convertFromDataEntity(data.getInputData()));
            return result;
        }
        return null;
    }

    @Override
    public List<ActivityBaseInfoVO> postActivityByTask(String taskCode, String appCode, String locale, Map<String, Object> paramCodeAndValue) throws Exception {
        ViewRequest request = new ViewRequest();
        request.setProjectCode(taskCode);
        request.setLocale(locale);
        // 根据项目code获取任务数据
        List<ActivityBaseInfoVO> tasks = viewService.postTasksByProject(request);
        if (null == tasks || tasks.isEmpty()) {
            return Collections.emptyList();
        }
        // 利用应用参数值、产品、条件筛选获取的任务信息
        String tenantId = ServiceUtils.getTenantId();
        List<ActivityBaseInfoVO> filterResult = filterActivityInfo(appCode, tasks, paramCodeAndValue, tenantId);
        setConfig(taskCode, tenantId, filterResult);
        return filterResult;
    }

    @Override
    public List<ActivityBaseInfoVO> getAllApproveActivityByApp(String applicationCode, String locale) throws DWException {
        ServiceUtils.setLocale(locale);
        List<String> codes = dataPickService.getCodeByTypeAndAppCode(ACTIVITY, applicationCode);
        if (CollectionUtils.isEmpty(codes)) {
            return Collections.emptyList();
        }
        Map<String, Object> q = new HashMap<>();
        q.put("code", codes);
        q.put("type", "approve");
        ViewRequest viewRequest = new ViewRequest();
        viewRequest.getQuery().putAll(q);
        return viewService.postTaskListApprove(viewRequest);
    }

    private List<ActivityBaseInfoVO> filterActivityInfo(String applicationCode, List<ActivityBaseInfoVO> approveActivityList, Map<String, Object> paramCodeAndValue, String tenantId) throws Exception {
        List<ActivityBaseInfoVO> result = new ArrayList<>();
        // 获取当前租户对应的敏/稳态产品名称
        List<String> productList = productNameResolver.getCloudProductNamesWithRetryAfterEocSet(tenantId, 3);
        for (ActivityBaseInfoVO activityBaseInfo : approveActivityList) {
            Object activityCode = JSONPath.eval(activityBaseInfo, "activityId");
            ActivityVisibleConfig activityVisibleConfig;
            // 如果有产品限定的条件就用限定条件，如果没有就用productCode为空的通用条件
            Bson bsonHasProduct = Filters.and(
                    Filters.eq("activityCode", activityCode),
                    Filters.eq("appCode", applicationCode),
                    Filters.in(PRODUCT_CODE, productList));
            activityVisibleConfig = knowledgeGraphService.findOne(bsonHasProduct, ActivityVisibleConfig.class, "activityVisibleConfig", KnowledgeGraphDb.SYSTEM);
            if (activityVisibleConfig == null) {
                Bson bsonHasNoProduct = Filters.and(
                        Filters.eq("activityCode", activityCode),
                        Filters.eq("appCode", applicationCode),
                        Filters.or(
                                Filters.eq(PRODUCT_CODE, ""),
                                Filters.eq(PRODUCT_CODE, null)));
                activityVisibleConfig = knowledgeGraphService.findOne(bsonHasNoProduct, ActivityVisibleConfig.class, "activityVisibleConfig", KnowledgeGraphDb.SYSTEM);
            }
            if (conditionResult(activityVisibleConfig, paramCodeAndValue)) {
                result.add(activityBaseInfo);
            }
        }
        return result;
    }

    /*
     * 利用已有参数值对条件进行判断,未设条件默认显示，设定条件后符合条件的显示
     */
    private Boolean conditionResult(ActivityVisibleConfig activityVisibleConfig, Map<String, Object> paramCodeAndValue) throws DWBusinessException {
        if (activityVisibleConfig == null) {
            return true;
        }
        DynamicCondition dynamicCondition = activityVisibleConfig.getAcitvityVisibleCondition();
        if (DynamicConditionTypeEnum.SINGLE.toString().equals(dynamicCondition.getType())) {
            return judgeExpression(dynamicCondition, paramCodeAndValue);
        }
        if (DynamicConditionTypeEnum.AND_GROUP.toString().equals(dynamicCondition.getType())) {
            return judgeAndGroup(paramCodeAndValue, dynamicCondition);
        }
        if (DynamicConditionTypeEnum.OR_GROUP.toString().equals(dynamicCondition.getType())) {
            return judgeOrGroup(paramCodeAndValue, dynamicCondition);
        }
        return true;
    }

    private Boolean judgeAndGroup(Map<String, Object> paramCodeAndValue, DynamicCondition dynamicCondition) throws DWBusinessException {
        List<DynamicCondition> conditions = dynamicCondition.getItems();
        for (DynamicCondition cd : conditions) {
            if (!judgeExpression(cd, paramCodeAndValue)) {
                return false;
            }
        }
        return true;
    }

    private Boolean judgeOrGroup(Map<String, Object> paramCodeAndValue, DynamicCondition dynamicCondition) throws DWBusinessException {
        List<DynamicCondition> conditions = dynamicCondition.getItems();
        for (DynamicCondition cd : conditions) {
            if (judgeExpression(cd, paramCodeAndValue)) {
                return true;
            }
        }
        return false;
    }

    // 处理条件中的IN、GREATER_THAN、LESS_THAN三种运算
    private Boolean judgeExpression(DynamicCondition dc, Map<String, Object> paramCodeAndValue) throws DWBusinessException {
        boolean result = false;
        if (paramCodeAndValue == null || paramCodeAndValue.isEmpty()) {
            // TODO 该接口感觉也应该挪到 datamap，因为也是运行时
            paramCodeAndValue = (Map<String, Object>) dataMapKgService.getVariable(dc.getLeft(), ServiceUtils.getTenantId());
        }
        if (paramCodeAndValue == null || paramCodeAndValue.isEmpty()) {
            return true;
        }

        Object variable = paramCodeAndValue.get(dc.getLeft());
        String op = dc.getOp();
        String[] value = dc.getRight().split(","); // 多个条件以逗号为分隔
        if (ConditionOperatorTypeEnum.IN.toString().equals(op)) {
            if (Arrays.asList(value).contains(String.valueOf(variable))) {
                result = true;
            }
        }
        try {
            if (ConditionOperatorTypeEnum.GREATER_THAN.toString().equals(op)) {
                double douValue = variable != null ? Double.parseDouble(String.valueOf(variable)) : 0;
                if (douValue > Double.parseDouble(value[0])) {
                    result = true;
                }
            }
            if (ConditionOperatorTypeEnum.LESS_THAN.toString().equals(op)) {
                double douValue = variable != null ? Double.parseDouble(String.valueOf(variable)) : 0;
                if (douValue < Double.parseDouble(value[0])) {
                    result = true;
                }
            }
        } catch (Exception e) {
            log.error("条件数值比较出现异常, 请检查配置条件值类型！");
            throw e;
        }
        return result;
    }

    @SuppressWarnings("unchecked")
    private void setConfig(String taskCode, String tenantId, List<ActivityBaseInfoVO> filterResult) throws DWException {
        if (CollectionUtils.isEmpty(filterResult)) {
            return;
        }
        for (ActivityBaseInfoVO item : filterResult) {
            Map<String, Object> dm = getTaskInfoFromDataMap(taskCode, item.getActivityId(), tenantId);
            item.setConfig((Map<String, Object>) dm.get("config"));
        }
    }

    private Map<String, Object> getTaskInfoFromDataMap(String taskId, String activityId, String tenantId) throws DWException {
        ViewRequest request = new ViewRequest();
        request.setProjectCode(taskId);
        request.setTaskCode(activityId);
        return viewService.postTask(request);
    }
}
