package com.digiwin.athena.datamap.spi;

import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.datamap.service.inner.DataMapKgDataPickService;
import com.digiwin.athena.kmservice.constants.KnowledgeGraphDb;
import com.digiwin.athena.datamap.service.inner.DataMapPickService;
import com.digiwin.athena.domain.common.ITenantObject;
import com.digiwin.athena.domain.component.bo.TenantAppComponentListBO;
import com.digiwin.athena.domain.core.app.Application;
import com.digiwin.athena.domain.core.app.ApplicationRelation;
import com.digiwin.athena.domain.core.app.TenantAppRelation;
import com.digiwin.athena.domain.neo4j.TenantEntity;
import com.digiwin.athena.dto.BasicQuery;
import com.digiwin.athena.kmservice.utils.MergeUtil;
import com.digiwin.athena.kmservice.utils.ServiceUtils;
import com.mongodb.client.FindIterable;
import com.mongodb.client.model.Filters;
import com.mongodb.client.result.DeleteResult;
import com.mongodb.client.result.UpdateResult;
import org.apache.commons.collections.CollectionUtils;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.convert.MongoConverter;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import java.util.*;

@Service
public class DbKnowledgeGraphService  implements IKnowledgeGraphService {

    @Autowired
    @Qualifier("kgSystem")
    private MongoTemplate mongoTemplateSys;
    @Autowired
    @Qualifier("kgTenant")
    private MongoTemplate mongoTemplateTenant;
    @Autowired
    @Qualifier("mappingMongoConverterSystem")
    MongoConverter converter;
    @Autowired
    @Qualifier("dataMapSystem")
    private MongoTemplate mongoTemplate;

    @Autowired
    DataMapPickService dataPickService;

    @Autowired
    DataMapKgDataPickService dataMapKgDataPickService;

    public static List<Class<?>> tenantClasses = Collections.singletonList(TenantAppComponentListBO.class);

    public static List<Class<?>> sysClasses = Arrays.asList(Application.class, ApplicationRelation.class, TenantEntity.class, TenantAppRelation.class);

    @Override
    public <T> T findOne(Bson filter, Class<T> entityClass, String collectionName, KnowledgeGraphDb db) {
        return findOne(filter, null, entityClass, collectionName, db);
    }

    @Override
    public <T> T findOne(Bson filter, Bson projection, Class<T> entityClass, String collectionName, KnowledgeGraphDb db) {
        FindIterable<Document> findIterable = mongoTemplate.getMongoDbFactory().getDb(db.getValue()).getCollection(collectionName).find(filter);
        if (projection != null) {
            findIterable.projection(projection);
        }
        findIterable.limit(1);
        Document document = findIterable.first();
        if (document == null) {
            return null;
        }
        return converter.read(entityClass, document);
    }

    @Override
    public <T extends ITenantObject> T findOneWithTenant(Bson filter, Class<T> entityClass, String collectionName, KnowledgeGraphDb db) throws DWBusinessException {
        String tenantId = ServiceUtils.getTenantId();
        String tenantVersion = dataPickService.tenantVersion();
        List<String> tenantIds = new ArrayList<>();
        tenantIds.add(tenantId);
        if (!"SYSTEM".equals(tenantId)) {
            tenantIds.add("SYSTEM");
            // 兼容未设定租户的数据，预设为系统级数据
            tenantIds.add(null);
        }
        Bson combinedFilter = Filters.and(
                filter,
                Filters.in("tenantId", tenantIds),
                Filters.eq("version", tenantVersion));
        FindIterable<Document> findIterable = mongoTemplate.getMongoDbFactory().getDb(db.getValue()).getCollection(collectionName).find(combinedFilter, Document.class);
        int i = 0;
        T first = null;
        for (Document document : findIterable) {
            T t = converter.read(entityClass, document);
            if (i++ == 0) {
                first = t;
            }
            if (!"SYSTEM".equals(t.getTenantId()) && t.getTenantId() != null) {
                // 优先返回租户级的配置
                return t;
            }
        }
        // 否则返回第一个非租户级配置
        return first;
    }

    @Override
    public <T> List<T> find(Bson filter, Class<T> entityClass, String collectionName, KnowledgeGraphDb db) {
        return find(filter, null, entityClass, collectionName, db);
    }

    @Override
    public <T> List<T> find(Bson filter, Bson projection, Class<T> entityClass, String collectionName, KnowledgeGraphDb db) {
        FindIterable<Document> findIterable = mongoTemplate.getMongoDbFactory().getDb(db.getValue()).getCollection(collectionName).find(filter, Document.class);
        if (projection != null) {
            findIterable.projection(projection);
        }
        List<T> list = new ArrayList<>();
        for (Document t : findIterable) {
            list.add(converter.read(entityClass, t));
        }
        return list;
    }

    @Override
    public UpdateResult upsert(Bson filter, Bson update, String collectionName, KnowledgeGraphDb db) {
        return mongoTemplate.getMongoDbFactory().getDb(db.getValue()).getCollection(collectionName).updateOne(filter, update);
    }

    @Override
    public DeleteResult remove(Bson filter, String collectionName, KnowledgeGraphDb db) {
        return mongoTemplate.getMongoDbFactory().getDb(db.getValue()).getCollection(collectionName).deleteMany(filter);
    }



    public List<Application> tenantAppRelations(List<String> appCodes) throws DWBusinessException {
        String tenantId = ServiceUtils.getTenantId();
        String tenantVersion = dataPickService.tenantVersion();
        List<Application> relations = mongoTemplateSys.find(Query.query(Criteria.where("tenantId").in(tenantId, "SYSTEM", null).and("code").in(appCodes).and("version").is(tenantVersion)), Application.class);
        relations = MergeUtil.excludeSameCode(relations, Application.class, null);
        return relations;
    }

    public Application tenantApp(String appCode) throws DWBusinessException {
        String tenantId = ServiceUtils.getTenantId();
        String tenantVersion = dataPickService.tenantVersion();
        List<Application> applications = mongoTemplateSys.find(Query.query(Criteria.where("tenantId").in(tenantId, "SYSTEM", null).and("code").is(appCode).and("version").is(tenantVersion)), Application.class);
        applications = MergeUtil.excludeSameCode(applications, Application.class, null);
        return CollectionUtils.isNotEmpty(applications) ? applications.get(0) : null;
    }


    public List<ApplicationRelation> applicationRelations(List<String> appCodes,String type,String subType) throws DWBusinessException {
        List<ApplicationRelation> result = new ArrayList<>();
        List<Application> tenantAppRelations = tenantAppRelations(appCodes);
        Map<String,Application> taMaps = new HashMap<>();
        tenantAppRelations.forEach(t->{
            taMaps.put(t.getCode(),t);
        });
        String tenantId = ServiceUtils.getTenantId();
        String tenantVersion = dataPickService.tenantVersion();
        Query query = new Query();
        query.addCriteria(Criteria.where("appCode").in(appCodes).and("version").is(tenantVersion).and("type").is(type));
        if(null!=subType){
            query.addCriteria(Criteria.where("subType").is(subType));
        }
        List<ApplicationRelation> relations = mongoTemplateSys.find(query,ApplicationRelation.class);
        for(ApplicationRelation relation:relations){
            Application tenantAppRelation1 = taMaps.get(relation.getAppCode());
            if(null==tenantAppRelation1){result.add(relation);continue;}
            int queryLevel = 0;
            if("base_entry".equals(subType) || "statement".equals(subType)){
                queryLevel = 2;
            }else{
                if(Boolean.TRUE.equals(tenantAppRelation1.getIndividualAll())){
                    queryLevel =1;
                }
            }
            switch (queryLevel){
                case 0:
                    if("SYSTEM".equals(relation.getTenantId()) || StringUtils.isEmpty(relation.getTenantId()) || (tenantId.equals(relation.getTenantId()) && relation.getSourceLevel()==null)){
                        result.add(relation);
                    }
                    break;
                case 1:
                    if(tenantId.equals(relation.getTenantId())){
                        result.add(relation);
                    }
                    break;
                case 2:
                    result.add(relation);
                    break;
            }

        }

        result = MergeUtil.excludeSameCode(result, ApplicationRelation.class,null);

        return result;
    }

    @Override
    public List<ApplicationRelation> relations(String app, String type) throws DWBusinessException {
        List<ApplicationRelation> relations = applicationRelations(Arrays.asList(app),type,null);
        return relations;
    }

    @Override
    public List<ApplicationRelation> relations(List<String> apps, String type) throws DWBusinessException {
        if(CollectionUtils.isEmpty(apps)){return null;}
        List<ApplicationRelation> relations = applicationRelations(apps,type,null);
        return relations;
    }

    @Override
    public ApplicationRelation relation(String code, String type) throws DWBusinessException {
        Map<String, Object> query = new HashMap<>();
        query.put("code", code);
        if (null != type) {
            query.put("type", type);
        }
        return dataMapKgDataPickService.findOne(BasicQuery.of(query, null), ApplicationRelation.class);
    }


    @Override
    public List<Application> applications(List<String> appCodes) throws DWBusinessException {
        return dataMapKgDataPickService.findByCodes(appCodes, Application.class);
    }

    @Override
    public Application application(String app) throws DWBusinessException {
        return dataMapKgDataPickService.findByCode(app, Application.class);
    }

    @Deprecated
    @Override
    public <T> List<T> find(BasicQuery basicQuery, Class<T> c, KnowledgeGraphDb db, String col) {
        Query query = dataPickService.buildQuery(basicQuery, c);
        MongoTemplate template = mongoTemplateSys;
        if (KnowledgeGraphDb.TENANT.equals(db)) {
            template = mongoTemplateTenant;
        }
        return template.find(query, c,col);
    }

    @Deprecated
    @Override
    public <T> T findOne(BasicQuery basicQuery, Class<T> c, KnowledgeGraphDb db, String col) {
        T t = null;
        List<T> ts = find(basicQuery, c, db, col);
        if (CollectionUtils.isNotEmpty(ts)) {
            t = ts.get(0);
        }
        return t;
    }
}
