package com.digiwin.athena.datamap.dao;

import com.alibaba.druid.util.StringUtils;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.kg.ComponentConstants;
import com.digiwin.athena.datamap.service.inner.DataMapKgDataPickService;
import com.digiwin.athena.domain.component.ComponentStructuredList;
import com.digiwin.athena.domain.component.bo.ComponentStructuredListType;
import com.digiwin.athena.domain.core.app.Application;
import com.digiwin.athena.kmservice.utils.I18nUtils;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

@Repository
public class DatamapComponentStructuredListDAO {

    @Autowired
    @Qualifier("kgSystem")
    private MongoTemplate mongoTemplateSystem;

    @Autowired
    @Qualifier("kgDataPickService")
    DataMapKgDataPickService dataPickService;

    /**
     * 根据应用code,范式code从系统级组件清单中获取范式组件
     * @param appCode 应用code
     * @param paradigm 范式code
     * @param version 版本号
     * @return
     * @throws DWBusinessException
     */
    public ComponentStructuredList findBaseFunctionComponentStructuredList(String appCode, String paradigm, String version, String tenantId) throws DWBusinessException {
        Criteria criteria = Criteria.where(ComponentConstants.APP_CODE).is(appCode)
                .and(ComponentConstants.PARADIGM).is(paradigm)
                .and(ComponentConstants.TYPE).is(ComponentStructuredListType.BaseFunction)
                .and(ComponentConstants.VERSION).is(version)
                .and(ComponentConstants.TENANT_ID).is(tenantId);
        ComponentStructuredList result = mongoTemplateSystem.findOne(Query.query(criteria), ComponentStructuredList.class, "componentStructuredList");
        if (result != null) {
            return result;
        } else {
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.noAvailableComponent",paradigm,appCode));
        }
    }

    /**
     * 根据应用code,范式code从组件清单中获取范式组件
     * @param appCode 应用code
     * @param paradigms 范式code集合
     * @param version 版本号
     * @return
     * @throws DWBusinessException
     */
    public List<ComponentStructuredList> findBaseFunctionComponentStructuredList(String appCode, List<String> paradigms, String version, String tenantId) throws DWBusinessException {
        Criteria criteria = Criteria.where(ComponentConstants.APP_CODE).is(appCode)
                .and(ComponentConstants.PARADIGM).in(paradigms)
                .and(ComponentConstants.VERSION).is(version)
                .and(ComponentConstants.TYPE).is(ComponentStructuredListType.BaseFunction)
                .and(ComponentConstants.TENANT_ID).is(tenantId);
        List<ComponentStructuredList> result = mongoTemplateSystem.find(Query.query(criteria), ComponentStructuredList.class, "componentStructuredList");
        if (result != null) {
            return result;
        } else {
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.noAvailableComponent",paradigms,appCode));
        }
    }

    /**
     * 根据应用code从组件清单中获取范式组件
     * @param appCode 应用code
     * @param version 版本号
     * @return
     * @throws DWBusinessException
     */
    public List<ComponentStructuredList> findBaseFunctionComponentStructuredList(String appCode,String version, String tenantId) throws DWBusinessException {
        Criteria criteria = Criteria.where(ComponentConstants.APP_CODE).is(appCode)
                .and(ComponentConstants.VERSION).is(version)
                .and(ComponentConstants.TYPE).is(ComponentStructuredListType.BaseFunction)
                .and(ComponentConstants.TENANT_ID).is(tenantId);
        List<ComponentStructuredList> result = mongoTemplateSystem.find(Query.query(criteria), ComponentStructuredList.class, "componentStructuredList");
        return result;
    }

    /**
     * 根据机制code从应用组件清单表中获取系统级机制组件数据
     * @param mechanismCode 机制code
     * @param version 租户版本
     * @return
     * @throws DWBusinessException
     */
    public ComponentStructuredList findMechanismComponentStructuredList(String mechanismCode,String version, String tenantId) throws DWBusinessException {
        Criteria criteria = Criteria.where(ComponentConstants.MECHANISM_CODE).is(mechanismCode)
                .and(ComponentConstants.VERSION).is(version)
                .and(ComponentConstants.TYPE).is(ComponentStructuredListType.Mechanism);
        ComponentStructuredList result = dataPickService.findOneByCondition(criteria, tenantId, ComponentStructuredList.class, "componentStructuredList");
        if (result != null) {
            return result;
        } else {
            throw new DWBusinessException(I18nUtils.getValue("knowledgegraph.noAvailableComponent3",mechanismCode));
        }
    }

    /**
     * 根据机制code集合从应用组件清单表中获取系统级机制组件数据
     * @param mechanismCode 机制code集合
     * @param version 租户版本
     * @return
     * @throws DWBusinessException
     */
    public List<ComponentStructuredList> findMechanismComponentStructuredList(List<String> mechanismCode,String version, String tenantId) throws DWBusinessException {
        Criteria criteria = Criteria.where(ComponentConstants.MECHANISM_CODE).in(mechanismCode)
                .and(ComponentConstants.VERSION).is(version)
                .and(ComponentConstants.TYPE).is(ComponentStructuredListType.Mechanism);
        List<ComponentStructuredList> result = dataPickService.find(criteria, tenantId, ComponentStructuredList.class, "componentStructuredList");
        if (CollectionUtils.isNotEmpty(result)) {
            return result;
        } else {
            return new ArrayList<>();
        }
    }

    /**
     * 根据应用code从应用组件清单中获取系统级机制组件code
     * @param appCode 应用code
     * @param version 租户版本
     * @return
     * @throws DWBusinessException
     */
    public List<String> findAllMechanismCodes(String appCode, String version, String tenantId) throws DWBusinessException {
        Criteria criteria = Criteria.where(ComponentConstants.APP_CODE).is(appCode).and(ComponentConstants.VERSION).is(version);
        List<String> mechanismCodes = new ArrayList<>();
        List<ComponentStructuredList> result = dataPickService.find(criteria, tenantId, ComponentStructuredList.class, "componentStructuredList");
        result = dataPickService.excludeWithSameCode(result);
        if (result != null && !result.isEmpty()) {
            result.forEach(list -> {
                String mechanismCode = list.getMechanismCode();
                if (!StringUtils.isEmpty(mechanismCode))
                    mechanismCodes.add(mechanismCode);
            });
        }
        return mechanismCodes;
    }

    /**
     * 增加租户条件
     * @param criteria 查询条件
     * @param appCode 应用code
     * @param tenantId 租户id
     * @throws DWBusinessException 业务异常
     */
    private void addTenantCondition(Criteria criteria, String appCode, String tenantId) throws DWBusinessException {
        Application application = dataPickService.tenantAppRelation(appCode, tenantId);
        boolean individualAll = Optional.ofNullable(application).map(Application::getIndividualAll).orElse(false);
        criteria.and(ComponentConstants.TENANT_ID).is(individualAll ? tenantId : "SYSTEM");
    }
}
