package com.digiwin.athena.datamap.mechanism.activitybuilder;

import com.digiwin.app.container.exceptions.DWBusinessException;

import com.digiwin.athena.datamap.mechanism.support.MechanismParseContext;

import com.digiwin.athena.datamap.service.inner.DataMapPickService;
import com.digiwin.athena.kmservice.utils.ServiceUtils;
import com.digiwin.athena.domain.common.Constants;
import com.digiwin.athena.domain.core.*;
import com.digiwin.athena.mechanism.widgets.ActivityWidget;
import com.digiwin.athena.mechanism.widgets.SourceWidget;
import com.digiwin.athena.mechanism.widgets.activity.TaskInitiateActivityWidget;
import io.seata.common.util.CollectionUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.core.annotation.Order;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * 执行零件-发起任务
 * 在指定的任务中新增不同group的起止状态配置
 */
@Slf4j
@Service
@Order(1)
public class TaskInitiateBuildService implements ActivityBuildService {

    @Autowired
    private DataMapPickService dataPickService;

    @Autowired
    @Qualifier("dataMapSystem")
    private MongoTemplate mongoTemplate;

    @Override
    public boolean accept(ActivityWidget ma) {
        return "taskInitiate".equalsIgnoreCase(ma.getType());
    }

    /**
     *  发起任务不是通过Activity实现，所以此处会返回一个空的list
     * @return
     */
    @Override
    public List<Activity> build(ActivityWidget ma, MechanismParseContext context) {
        TaskInitiateActivityWidget convert = dataPickService.convert(ma, TaskInitiateActivityWidget.class);
        SourceWidget sourceWidget = context.getBo().getCheckSource();
        if ("taskSource".equalsIgnoreCase(sourceWidget.getType())) {
            this.taskInitiateWhenTaskSource(convert, context);
        } else if ("projectSource".equalsIgnoreCase(sourceWidget.getType())) {
            this.taskInitiateWhenProjectSource(convert, context);
        } else if ("scheduleSource".equalsIgnoreCase(sourceWidget.getType())) {
            this.taskInitiateWhenScheduleSource(convert, context);
        }

        return new ArrayList<>();
    }

    private void taskInitiateWhenTaskSource(TaskInitiateActivityWidget convert, MechanismParseContext context) {
        try {
            String dataCode = context.getSourceWidget().getTarget();
            // 找到数据源任务的from状态
            Task sourceTask = this.dataPickService.findByCode(dataCode, Task.class);
            if (sourceTask == null) {
                throw new RuntimeException("taskInitiate error, not found sourceTask:" + dataCode);
            }
            List<StateMap> stateMaps = sourceTask.getStateMaps();
            StateMap sourceStateMap = stateMaps.get(0); // 选取第一个，当有多个的时候也不确定是哪个状态要发起，先选第一个

            this.taskInitiateWithNewFrom(sourceStateMap.getInput(), convert, context);

        } catch (DWBusinessException e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException("taskInitiate error:" + e.getMessage());
        }
    }

    private void taskInitiateWhenProjectSource(TaskInitiateActivityWidget convert, MechanismParseContext context) {
        try {
            String dataCode = context.getSourceWidget().getTarget();
            Project sourceProject = this.dataPickService.findBetter(dataCode, Project.class);
            if (sourceProject == null) {
                throw new RuntimeException("taskInitiate error, not found sourceProject:" + dataCode);
            }
            DataState init = sourceProject.getInit();
            String code = init.getCode();

            this.taskInitiateWithNewFrom(code, convert, context);

        } catch (DWBusinessException e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException("taskInitiate error:" + e.getMessage());
        }

    }

    private void taskInitiateWhenScheduleSource(TaskInitiateActivityWidget convert, MechanismParseContext context) {
        try {
            // 当是驱动的能力的时候， 是使用驱动编译时新建的那个project 中 init属性下的code 作为新任务的from
            String startState = context.getStartState();
            this.taskInitiateWithNewFrom(startState, convert, context);

        } catch (DWBusinessException e) {
            log.error(e.getMessage(), e);
            throw new RuntimeException("taskInitiate error:" + e.getMessage());
        }
    }

    private void taskInitiateWithNewFrom(String newFrom, TaskInitiateActivityWidget convert, MechanismParseContext context) throws DWBusinessException {
        String version = dataPickService.tenantVersion();
        String tenantId = ServiceUtils.getTenantId();
        Query queryTenant = Query.query(Criteria.where("code").is(convert.getCode()).and("tenantId").is(tenantId));
        Task targetTenantTask = dataPickService.tenantTemplate().findOne(queryTenant, Task.class, dataPickService.tenantCol(Task.class));
        if (targetTenantTask == null) {
            Task targetSystemTask = mongoTemplate.findOne(Query.query(Criteria.where("code").is(convert.getCode())
                    .and("tenantId").is(Constants.defaultTenantId).and("version").lte(version)), Task.class);
            if (targetSystemTask == null) {
                throw new RuntimeException("taskInitiate error, not found targetSystemTask:" + convert.getCode());
            }
            List<String> sysFrom = targetSystemTask.getFrom();
            List<StateMap> sysStateMaps = targetSystemTask.getStateMaps();
            // 构造新的stateMap，并设置group，新的stateMap的input节点为检查点的input
            StateMap targetStateMap = new StateMap();
            targetStateMap.setInput(newFrom);
            targetStateMap.setGroup(context.getBindingContext().getPluginId());
            targetStateMap.setOutput(sysStateMaps.get(0).getOutput());
            sysStateMaps.add(targetStateMap);
            sysFrom.add(targetStateMap.getInput());

            Task task1 = new Task();
            task1.setCode(targetSystemTask.getCode());
            task1.setFrom(sysFrom);
            task1.setTo(targetSystemTask.getTo());
            task1.setStateMaps(sysStateMaps);
            task1.setVersion("1.0");
            task1.setTenantId(tenantId);
            task1.setPluginId(context.getBindingContext().getPluginId());
            task1.setMilestone(targetSystemTask.getMilestone());

            this.dataPickService.save(task1, Task.class);
        } else {

            List<String> from = targetTenantTask.getFrom();
            List<StateMap> stateMaps1 = targetTenantTask.getStateMaps();
            if (CollectionUtils.isEmpty(stateMaps1)) { // 如果租户配置中没有stateMap的信息
                Task targetSystemTask = mongoTemplate.findOne(Query.query(Criteria.where("code").is(convert.getCode())
                        .and("tenantId").is(Constants.defaultTenantId).and("version").lte(version)), Task.class);
                if (targetSystemTask == null) {
                    throw new RuntimeException("taskInitiate error, not found targetSystemTask:" + convert.getCode());
                }
                from = targetSystemTask.getFrom();
                stateMaps1 = targetSystemTask.getStateMaps();
            }

            Optional<StateMap> any = stateMaps1.stream().filter(e -> StringUtils.equals(e.getGroup(), context.getBindingContext().getPluginId())).findAny();
            if (any.isPresent()) { // 如果之前有过此机制的,更新下from
                StateMap stateMap = any.get();
                if (newFrom != stateMap.getInput()) {
                    stateMap.setInput(newFrom);
                    from.remove(newFrom);
                    from.add(newFrom);
                }
            } else {
                StateMap targetStateMap = new StateMap();
                targetStateMap.setInput(newFrom);
                targetStateMap.setGroup(context.getBindingContext().getPluginId());
                targetStateMap.setOutput(stateMaps1.get(0).getOutput());
                stateMaps1.add(targetStateMap);
                from.add(newFrom);
            }

            targetTenantTask.setFrom(from);
            targetTenantTask.setStateMaps(stateMaps1);
            targetTenantTask.setPluginId(context.getBindingContext().getPluginId());

            // this.mongoTemplate.save(targetTenantTask);
            dataPickService.saveTenantObject(targetTenantTask, Task.class);

        }
    }

}
