package com.digiwin.athena.datamap.mechanism.convert;

import com.digiwin.athena.datamap.mechanism.component.AbstractAbilityComponentHandler;
import com.digiwin.athena.datamap.mechanism.convert.parsers.customFunction.CustomFuncParser;
import com.digiwin.athena.datamap.mechanism.convert.parsers.fieldValueType.FieldValueTypeParser;
import com.digiwin.athena.datamap.mechanism.convert.parsers.hooks.HooksParser;
import com.digiwin.athena.datamap.mechanism.convert.parsers.judgeBasis.JudgeBasisParser;
import com.digiwin.athena.datamap.mechanism.convert.parsers.rightCond.RightCondTypeParser;
import com.digiwin.athena.datamap.mechanism.limithandler.parsers.limitAction.LimitActionParser;
import com.digiwin.athena.datamap.mechanism.limithandler.parsers.limitCondition.LimitConditionParser;
import com.digiwin.athena.domain.component.ComponentTypeEnum;
import com.digiwin.athena.mechanism.dto.MechanismComponentDTO;
import com.digiwin.athena.mechanism.widgets.config.MechanismConditionConfig;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.List;
import java.util.Optional;

/**
 * @Description 与机制相关的策略工厂
 * @Author liugl
 * @Date 2024/6/28 8:55
 **/
@Service
@Slf4j
public class MechanismParserFactory {

    @Autowired
    List<JudgeBasisParser> judgeBasisParserList;

    @Autowired
    List<HooksParser> hooksParserList;

    @Autowired
    List<FieldValueTypeParser> fieldValueTypeParserList;

    @Autowired
    List<CustomFuncParser> customFuncParserList;

    @Autowired
    List<RightCondTypeParser> rightCondParserList;

    @Autowired
    List<LimitActionParser> limitActionParserList;

    @Autowired
    List<LimitConditionParser> limitConditionParserList;

    @Autowired
    List<AbstractAbilityComponentHandler> abilityComponentHandlerList;

    public JudgeBasisParser getJudgeParserByType(String type) {
        if (CollectionUtils.isEmpty(judgeBasisParserList)) {
            return null;
        }
        if (ObjectUtils.isEmpty(type)) {
            return null;
        }
        Optional<JudgeBasisParser> judgeParserOptional = judgeBasisParserList.stream().filter(v -> v.accept(type)).findFirst();
        if (judgeParserOptional.isPresent()) {
            return judgeParserOptional.get();
        }
        return null;
    }

    /**
     * 根据配置获取hook解析器
     *
     * @param config 配置信息
     * @return 返回HooksParser
     */
    public HooksParser getHooksParserByConfig(MechanismConditionConfig config) {
        if (CollectionUtils.isEmpty(hooksParserList)) {
            return null;
        }
        if (ObjectUtils.isEmpty(config)) {
            return null;
        }
        Optional<HooksParser> hooksParserOptional = hooksParserList.stream().filter(v -> v.accept(config)).findFirst();
        if (hooksParserOptional.isPresent()) {
            return hooksParserOptional.get();
        }
        return null;
    }

    /**
     * 根据配置获取字段类型解析器
     *
     * @param config 配置信息
     * @return 返回FieldLogicParser
     */
    public FieldValueTypeParser getFieldTypeParserByConfig(String config) {
        if (CollectionUtils.isEmpty(fieldValueTypeParserList)) {
            return null;
        }
        if (ObjectUtils.isEmpty(config)) {
            return null;
        }
        Optional<FieldValueTypeParser> hooksParserOptional = fieldValueTypeParserList.stream().filter(v -> v.accept(config)).findFirst();
        if (hooksParserOptional.isPresent()) {
            return hooksParserOptional.get();
        }
        return null;
    }

    /**
     * 根据函数方法名获取自定义函数
     *
     * @param func 函数方法名
     * @return 返回 CustomFuncParser
     */
    public CustomFuncParser getCustomFuncParserByName(String func) {
        if (CollectionUtils.isEmpty(customFuncParserList)) {
            return null;
        }
        if (ObjectUtils.isEmpty(func)) {
            return null;
        }
        Optional<CustomFuncParser> hooksParserOptional = customFuncParserList.stream().filter(v -> v.accept(func)).findFirst();
        if (hooksParserOptional.isPresent()) {
            return hooksParserOptional.get();
        }
        return null;
    }

    /**
     * 根据类型获取右侧条件类型 例如：自定义值，自定义函数，任务字段，枚举值，机制参数
     *
     * @param type 函数方法名
     * @return 返回 CustomFuncParser
     */
    public RightCondTypeParser getRightCondParserByType(String type) {
        if (CollectionUtils.isEmpty(rightCondParserList)) {
            return null;
        }
        if (ObjectUtils.isEmpty(type)) {
            return null;
        }
        Optional<RightCondTypeParser> hooksParserOptional = rightCondParserList.stream().filter(v -> v.accept(type)).findFirst();
        if (hooksParserOptional.isPresent()) {
            return hooksParserOptional.get();
        }
        return null;
    }

    /**
     * 根据类型获取机制能力处理类
     *
     * @param type 函数方法名
     * @return 返回 AbstractAbilityComponentHandler
     */
    public AbstractAbilityComponentHandler getAbilityHandlerByType(String type) {
        if (CollectionUtils.isEmpty(abilityComponentHandlerList)) {
            return null;
        }
        if (ObjectUtils.isEmpty(type)) {
            return null;
        }
        MechanismComponentDTO dto = new MechanismComponentDTO();
        dto.setType(type);
        if (ComponentTypeEnum.MechanismLimit.name().equalsIgnoreCase(type)) {
            dto.setMechanismVersion("2.0");
        }
        Optional<AbstractAbilityComponentHandler> hooksParserOptional = abilityComponentHandlerList.stream()
                .filter(v -> v.accept(dto)).findFirst();
        if (hooksParserOptional.isPresent()) {
            return hooksParserOptional.get();
        }
        return null;
    }

    /**
     * 根据类型获取机制能力处理类
     *
     * @param type 函数方法名
     * @return 返回 LimitActionParser
     */
    public LimitActionParser getLimitActionParserByType(String type) {
        if (CollectionUtils.isEmpty(limitActionParserList)) {
            return null;
        }
        if (ObjectUtils.isEmpty(type)) {
            return null;
        }
        Optional<LimitActionParser> hooksParserOptional = limitActionParserList.stream()
                .filter(v -> v.accept(type)).findFirst();
        if (hooksParserOptional.isPresent()) {
            return hooksParserOptional.get();
        }
        return null;
    }

    /**
     * 根据类型获取机制能力处理类
     *
     * @param type 函数方法名
     * @return 返回 LimitConditionParser
     */
    public LimitConditionParser getLimitConditionParserByType(String type) {
        if (CollectionUtils.isEmpty(limitConditionParserList)) {
            return null;
        }
        if (ObjectUtils.isEmpty(type)) {
            return null;
        }
        Optional<LimitConditionParser> hooksParserOptional = limitConditionParserList.stream()
                .filter(v -> v.accept(type)).findFirst();
        if (hooksParserOptional.isPresent()) {
            return hooksParserOptional.get();
        }
        return null;
    }

}
