package com.digiwin.athena.datamap.mechanism.convert.parsers.fieldValueType;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.datamap.mechanism.convert.CustomFuncResult;
import com.digiwin.athena.datamap.mechanism.convert.MechanismParserFactory;
import com.digiwin.athena.mechanism.common.MechanismEnum;
import com.digiwin.athena.mechanism.common.MechanismVariable;
import com.digiwin.athena.mechanism.widgets.config.MechanismConditionConfig;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.text.ParseException;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @Description 日期类型解析器
 * @Author liugl
 * @Date 2024/7/22 10:29
 **/
@Service
@Slf4j
public class DateParser implements FieldValueTypeParser {

    @Autowired
    MechanismParserFactory mechanismParserFactory;

    @Override
    public boolean accept(String dataType) {
        return MechanismEnum.FieldValueType.DATE.getCode().equalsIgnoreCase(dataType);
    }

    @Override
    public JSONArray parse(MechanismConditionConfig conditionConfig, JSONArray metaDataList) {
        JSONArray result = new JSONArray();
        // 处理全局类型的判断值，例如自定义值，机制函数，枚举值
        CustomFuncResult rightCondResult = mechanismParserFactory.getRightCondParserByType(conditionConfig.getRightCondType())
                .preParse(conditionConfig, metaDataList);
        if (!rightCondResult.getIsNeedNextJudge()) {
            return rightCondResult.getMetaDataList();
        }
        String opType = conditionConfig.getOpType();
        Date rightToDate = null;
        Date rightFromDate = null;
        try {
            MechanismVariable rightTo = conditionConfig.getRight().getTo();
            Object rightToValueObj = rightTo.getValue();
            // 当前日期比较类型 对应页面上的当天，本周，当月，当年
            String currentDateType = conditionConfig.getRight().getType();
            // 主要是机制参数日期
            if (rightToValueObj.toString().matches("^[1-9][0-9]{3}.[0-9]{1,2}.[0-9]{1,2}.*$")) {
                rightToDate = convertStringToDate(rightToValueObj.toString());
            } else {
                // 设计态页面日期
                Integer rightToOffsetValue = Integer.valueOf(rightToValueObj.toString().split("[/.]")[0]);
                rightToDate = getOffsetDate(currentDateType, rightToOffsetValue);
            }

            if (opType.equals(MechanismEnum.OperateType.BETWEEN.getCode())) {
                MechanismVariable rightFrom = conditionConfig.getRight().getFrom();
                Object rightFromValueObj = rightFrom.getValue();
                // 主要是机制参数日期
                if (rightFromValueObj.toString().matches("^[1-9][0-9]{3}.[0-9]{1,2}.[0-9]{1,2}.*$")) {
                    rightFromDate = convertStringToDate(rightToValueObj.toString());
                } else {
                    // 设计态页面日期
                    Integer rightToOffsetValue = Integer.valueOf(rightFromValueObj.toString().split("[/.]")[0]);
                    rightFromDate = getOffsetDate(currentDateType, rightToOffsetValue);
                }
            }
        } catch (Exception e) {
            log.error("DateParser.parse 右侧条件判断值转换失败，入参：{}", conditionConfig);
            // 如果条件值不是数值类型或者是空值则不比较，直接返回空数据
            return result;
        }
        String dataName = conditionConfig.getLeft().getData_name();
        result.addAll(execCondition(opType, dataName, rightFromDate, rightToDate, metaDataList));
        return result;
    }

    /**
     * 执行判断条件
     *
     * @param opType         操作符
     * @param dataName       字段名称
     * @param rightFromValue 右侧开始值
     * @param rightToValue   右侧截至值
     * @param metaDataList   任务卡表格数据
     * @return 返回满足条件的表格数据
     */
    private JSONArray execCondition(String opType, String dataName, Date rightFromValue, Date rightToValue, JSONArray metaDataList) {
        if (CollectionUtils.isEmpty(metaDataList)) {
            return new JSONArray();
        }
        List<Object> resultList = metaDataList.stream().filter(v -> {
            JSONObject metaDataObj = JSON.parseObject(JSON.toJSONString(v));

            String sourceValue = null;
            try {
                sourceValue = metaDataObj.getString(dataName);
            } catch (Exception e) {
                log.warn("NumericParser.parse 表格数据转换失败，数据为：{}", v);

            }
            return isSatisfyCond(opType, sourceValue, rightFromValue, rightToValue);
        }).collect(Collectors.toList());
        return JSONArray.parseArray(JSON.toJSONString(resultList));
    }

    /**
     * 执行操作符逻辑
     *
     * @param opType        操作符
     * @param sourceValue   任务卡表格字段数值
     * @param rightFromDate 右侧比较 开始值
     * @param rightToDate   右侧比较 截至值
     * @return
     */
    private Boolean isSatisfyCond(String opType, String sourceValue, Date rightFromDate, Date rightToDate) {
        Date sourceDate = convertStringToDate(sourceValue);
        if (sourceDate == null) {
            return Boolean.FALSE;
        }
        int toResult = sourceDate.compareTo(rightToDate);
        switch (opType) {
            case "eq":
                return toResult == 0;
            case "gt":
                return toResult == 1;
            case "lt":
                return toResult == -1;
            case "gte":
                return toResult == 1 || toResult == 0;
            case "lte":
                return toResult == -1 || toResult == 0;
            case "between":
                int fromResult = sourceDate.compareTo(rightFromDate);
                // 大于等于开始值 && 小于等于截至值
                return (fromResult == 1 || fromResult == 0) && (toResult == -1 || toResult == 0);
            default:
                return Boolean.FALSE;
        }
    }

    /**
     * 将string字段串转化为date
     * @param sourceValue
     * @return 结果
     */
    private Date convertStringToDate(String sourceValue) {
        if (ObjectUtils.isEmpty(sourceValue)) {
            return null;
        }
        // 如果表格数据不满足日期格式则直接退出
        if (!sourceValue.matches("^[1-9][0-9]{3}.[0-9]{1,2}.[0-9]{1,2}.*$")) {
            return null;
        }
        Date sourceDate;
        if (sourceValue.matches("[\\d]+")) {
            // 如果是纯数字格式，必须保证有前8位，即yyyyMMdd
            if (sourceValue.length() < 8) {
                return null;
            }
            try {
                sourceDate = DateUtils.parseDate(sourceValue.substring(0, 8), "yyyyMMdd");
            } catch (ParseException e) {
                log.error("DateParser.isSatisfyCond 时间转换失败，表格数据为{}", sourceValue);
                return null;
            }
        } else {
            String[] split = sourceValue.split("[^\\d]");
            LocalDate sourceLocalDate = LocalDate.of(Integer.parseInt(split[0]), Integer.parseInt(split[1]), Integer.parseInt(split[2]));
            sourceDate = Date.from(sourceLocalDate.atStartOfDay(ZoneId.systemDefault()).toInstant());
        }
        return sourceDate;
    }

    /**
     * 根据比较日期类型以及偏移量获取日期
     * @param currentDateType
     * @param offsetValue
     * @return 返回偏移后的日期
     */
    private Date getOffsetDate(String currentDateType, Integer offsetValue) {
        switch (currentDateType) {
            // 目前只有当天类型，如有其他再增加
            case "current_day":
                // 精确到天
                DateTime currentDay = DateUtil.beginOfDay(DateUtil.date());
                return DateUtils.addDays(new Date(currentDay.getTime()), offsetValue);
            default:
                return null;
        }

    }

    @Override
    public Boolean isBothEqual(Object source, Object targetLeft, Object targetRight, String opType) {
        return Boolean.FALSE;
    }
}
