package com.digiwin.athena.datamap.mechanism.convert.parsers.judgeBasis;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.datamap.mechanism.MechanismHelpService;
import com.digiwin.athena.datamap.mechanism.convert.HookDynamicFunction;
import com.digiwin.athena.datamap.mechanism.convert.HooksInfo;
import com.digiwin.athena.datamap.mechanism.convert.MechanismParserFactory;
import com.digiwin.athena.datamap.mechanism.convert.parsers.hooks.HooksParser;
import com.digiwin.athena.datamap.mechanism.convert.parsers.rightCond.CustomFunctionParser;
import com.digiwin.athena.datamap.mechanism.support.BindingContext;
import com.digiwin.athena.datamap.spi.DataMapKgService;
import com.digiwin.athena.datamap.utils.DataMapUtils;
import com.digiwin.athena.domain.core.view.PageView;
import com.digiwin.athena.mechanism.bo.DecisionAbilityBO;
import com.digiwin.athena.mechanism.common.MechanismAbility;
import com.digiwin.athena.mechanism.common.MechanismEnum;
import com.digiwin.athena.mechanism.dto.MechanismApiInfoDTO;
import com.digiwin.athena.mechanism.widgets.config.DecisionScene;
import com.digiwin.athena.mechanism.widgets.config.DecisionTarget;
import com.digiwin.athena.mechanism.widgets.config.MechanismConditionConfig;
import com.digiwin.athena.mechanism.widgets.config.MechanismConditionGroup;
import com.jayway.jsonpath.JsonPath;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

@Service
@Slf4j
public class ConditionParser implements JudgeBasisParser {

    @Autowired
    MechanismParserFactory mechanismParserFactory;

    @Autowired
    DataMapKgService dataMapKgService;

    @Autowired
    MechanismHelpService mechanismHelpService;

    @Autowired
    CustomFunctionParser customFunctionParser;

    @Override
    public boolean accept(String type) {
        return MechanismEnum.ConditionType.PRIMARY.getCode().equalsIgnoreCase(type) ||
                MechanismEnum.ConditionType.HIGH_GRADE.getCode().equalsIgnoreCase(type);
    }

    @Override
    public void parse(MechanismAbility ability, PageView pageView, BindingContext context, HooksInfo hooksInfoAll) {
        DecisionAbilityBO bo = (DecisionAbilityBO) ability;
        context.getMechanismVariables().put("hookSelectorInitDataFieldParam", bo.getVariable().getData_name());
        if (CollectionUtils.isEmpty(bo.getDecisionOption().getScenes())) {
            log.error("无场景数据，入参：{}", ability);
            return;
        }
        String layoutId = DataMapUtils.getTableLayoutId(pageView);
        Map<String, Object> apiInfo = new HashMap<>();
        Map<String, Object> paramsInfo = new HashMap();
        Map<String, Object> req = new HashMap();
        String tableDataKey = DataMapUtils.getTableDataKey(pageView);
        Object dataSourceTable = pageView.getDataSources().get(tableDataKey);
        // TODO 这里解析的 tableDataKey 随着开发平台调整，有可能不对，所以先兼容使用第一个数据源
        if (dataSourceTable == null) {
            dataSourceTable = pageView.getDataSources().values().iterator().next();
        }
        Object dataKeys = JsonPath.read(dataSourceTable, "$.dataKeys");

        try {
            // step1：处理每个类型特有的hooks
            MechanismConditionConfig mechanismCondition = bo.getDecisionOption().getScenes().get(0).getCheck().getGroups().get(0).getConditions().get(0);
            HooksParser hooksParser = mechanismParserFactory.getHooksParserByConfig(mechanismCondition);
            if (null == hooksParser) {
                log.error("ApiParser.parseScene 获取HooksParser失败，入参：{}", bo);
            }
            HooksInfo hooksInfo = hooksParser.parse(mechanismCondition, context, pageView, null);
            hooksInfoAll.getDynamicFunctions().addAll(hooksInfo.getDynamicFunctions());
        } catch (Exception e) {
            log.error("解析pageView获取决策条件失败，参数scenes：{}", bo);
            return;
        }

        // url 或者参数携带是哪一个场景，在调用api时需要传递对应的场景，然后将符合场景的bk_info返回到界面，由前端在页面用场景号进行匹配
        // 场景号用法类似于api条件中的url
        paramsInfo.put("api", "/mechanism/postQueryApiResult");
        paramsInfo.put("requestJson", new Object());
        paramsInfo.put("ability", ability);
        paramsInfo.put("abilityType", "MechanismDecision");
        paramsInfo.put("keyFieldNameList", JSONArray.parseArray(JSON.toJSONString(dataKeys)));
        // 内部api不需要Esp，因此设置为false
        paramsInfo.put("isNeedEsp", Boolean.FALSE);
        req.put("req", paramsInfo);
        apiInfo.put("params", req);
        // 多场景时，仅调用一次api,但每个场景都会有决策方案，如果决策方案选择从页面获取，无法将其和场景一一对应上
        // 因此选择从api结果返回决策方案
        apiInfo.put("type", "api");
        hooksInfoAll.getApiList().add(apiInfo);

        // 设定调用api的hooks
        HashMap<String, Object> callApiParams = new HashMap<>();
        callApiParams.put("hookCallApiAppType", MechanismEnum.ConditionType.STANDARD.getCode());
        callApiParams.put("hookCallApiFieldParam", bo.getVariable().getData_name());
        callApiParams.put("hookCallApiApiListParam", hooksInfoAll.getApiList());
        callApiParams.put("hookCallApiKeyFieldNameList", JSONArray.parseArray(JSON.toJSONString(dataKeys)));
        callApiParams.put("hookCallApiTableDataKey", tableDataKey);

        hooksInfoAll.getDynamicFunctions().add(new HookDynamicFunction()
                .setFunctionName("hookCallApi")
                .setHookName("gridReadyHook")
                .setSourceId(layoutId)
                .setParameter(callApiParams));
        hooksInfoAll.getDynamicFunctions().add(new HookDynamicFunction()
                .setFunctionName("hookCallParamPrepare")
                .setHookName("afterViewInitHook")
                .setSourceId(layoutId)
                .setParameter(callApiParams));
    }

    /**
     * 解析数据
     * @param apiInfoDTO
     * @return 返回符合场景条件的所有表格数据 格式为：{}
     */
    public Object parseScene(MechanismApiInfoDTO apiInfoDTO) {
        DecisionAbilityBO bo = mechanismHelpService.convert(apiInfoDTO.getAbility(), DecisionAbilityBO.class);
        if (CollectionUtils.isEmpty(bo.getDecisionOption().getScenes())) {
            log.error("无场景数据，入参：{}", apiInfoDTO);
            return null;
        }
        if (ObjectUtils.isEmpty(apiInfoDTO.getTableData())) {
            log.error("无场表格数据，入参：{}", apiInfoDTO);
            return null;
        }
        // 总逻辑
        // 筛选出符合场景的所有表格数据
        // step1：获取数据
        JSONArray metaDataList = JSON.parseArray(JSON.toJSONString(apiInfoDTO.getTableData()));
        // 满足场景的表格数据map {主键拼接字符， 决策方案}
        Map<String, Object> accordTableDataMap = new HashMap<>();
        // step2：循环遍历场景处理数据
        for (int i = 1; i <= bo.getDecisionOption().getScenes().size(); i++) {
            DecisionScene decisionScene = bo.getDecisionOption().getScenes().get(i - 1);
            List<DecisionTarget> targets = decisionScene.getTargets();
            List<MechanismConditionGroup> groups = decisionScene.getCheck().getGroups();
            // 遍历场景根据主键依次覆盖数据
            accordTableDataMap.putAll(parseSingleScene(groups, metaDataList, apiInfoDTO, targets));
        }
        return accordTableDataMap.values();
    }

    /**
     * 解析单个场景数据
     * @param groups
     * @param metaDataList
     * @param apiInfoDTO
     * @param targets
     * @return 返回解析结果 {主键拼接字符， 决策方案}
     */
    public Map<String, Object> parseSingleScene(List<MechanismConditionGroup> groups, JSONArray metaDataList,
                                                MechanismApiInfoDTO apiInfoDTO, List<DecisionTarget> targets) {
        LinkedList<String> keyFieldNameList = new LinkedList<>();
        if (apiInfoDTO != null) {
            keyFieldNameList.addAll(apiInfoDTO.getKeyFieldNameList());
        }
        // 满足场景的表格数据map {主键拼接字符， 决策方案}
        Map<String, Object> accordSceneDataMap = new HashMap<>();
        if (apiInfoDTO == null && CollectionUtils.isEmpty(groups)) {
            accordSceneDataMap.put("list", metaDataList);
            return accordSceneDataMap;
        }
        JSONArray accordSceneDataArray = new JSONArray();
        // step2.1：单个场景的条件组间是 【或】 关系
        for (int i = 0; i < groups.size(); i++) {
            List<MechanismConditionConfig> conditions = groups.get(i).getConditions();
            // step2.1.1：单个条件组中的条件是 【且】 的关系
            JSONArray accordDataArray = new JSONArray();
            accordDataArray.addAll(metaDataList);
            for (int j = 0; j < conditions.size(); j++) {
                MechanismConditionConfig conditionConfig = conditions.get(j);
                // 如果是高级条件，提前处理左侧自定义函数
                if (MechanismEnum.ConditionType.HIGH_GRADE.getCode().equalsIgnoreCase(conditionConfig.getType())) {
                    customFunctionParser.preParseLeftCond(conditionConfig.getLeft(), accordDataArray);
                }
                String dataType = conditionConfig.getLeft().getData_type();
                // step2.1.2：将整个表格数据循环依次通过条件筛选，即【且】关系判断
                JSONArray tempAccordDataArray = mechanismParserFactory.getFieldTypeParserByConfig(dataType)
                        .parse(conditionConfig, accordDataArray);
                // 没有数据符合当前条件则立即退出，后续条件不再判断
                if (CollectionUtils.isEmpty(tempAccordDataArray)) {
                    accordDataArray.clear();
                    break;
                }
                accordDataArray.clear();
                accordDataArray.addAll(tempAccordDataArray);
            }
            if (apiInfoDTO == null) {
                accordSceneDataArray.addAll(accordDataArray);
                continue;
            }
            // step2.2：将符合任一组条件的数据做合并，即【或】取并集
            if (!CollectionUtils.isEmpty(accordDataArray)) {
                accordDataArray.stream().forEach(k -> {
                    JSONObject rowData = JSON.parseObject(JSON.toJSONString(k));
                    JSONObject accordRowData = new JSONObject();
                    JSONObject keyObj = new JSONObject();
                    accordRowData.put("key", keyObj);
                    accordRowData.put("field_value", targets);
                    accordRowData.put("row_data", k);
                    StringBuilder priKeyValue = new StringBuilder();
                    for (int n = 0; n < keyFieldNameList.size(); n++) {
                        keyObj.put(keyFieldNameList.get(n), rowData.get(keyFieldNameList.get(n)));
                        priKeyValue.append(rowData.get(keyFieldNameList.get(n)));
                    }
                    accordSceneDataMap.put(priKeyValue.toString(), accordRowData);
                });
            }
        }
        if (apiInfoDTO == null) {
            accordSceneDataMap.put("list", accordSceneDataArray);
        }
        return accordSceneDataMap;
    }
}
