package com.digiwin.athena.datamap.service.impl;

import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONPath;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.container.exceptions.DWException;
import com.digiwin.app.resource.DWResourceBundleUtils;
import com.digiwin.app.service.DWServiceContext;
import com.digiwin.athena.config.GroupCondition;
import com.digiwin.athena.kmservice.aspect.MyExceptionHandler;
import com.digiwin.athena.datamap.povo.rebuild.UserComponentDto;
import com.digiwin.athena.kg.domain.TenantSummaryField;
import com.digiwin.athena.kg.domain.TenantSummaryFields;
import com.digiwin.athena.kmservice.constants.KnowledgeGraphDb;
import com.digiwin.athena.kmservice.action.execution.ProductNameResolver;
import com.digiwin.athena.datamap.mechanism.paradigm.DataMapParadigmService;
import com.digiwin.athena.datamap.povo.*;
import com.digiwin.athena.datamap.service.ITaskService;
import com.digiwin.athena.datamap.service.inner.DataMapPickService;
import com.digiwin.athena.datamap.service.inner.KmService;
import com.digiwin.athena.datamap.spi.DbKnowledgeGraphService;
import com.digiwin.athena.datamap.spi.IKnowledgeGraphService;
import com.digiwin.athena.datamap.spi.DataMapIamService;
import com.digiwin.athena.datamap.spi.DataMapKgService;
import com.digiwin.athena.datamap.utils.CommonUtils;
import com.digiwin.athena.domain.common.BaseEntity;
import com.digiwin.athena.domain.core.*;
import com.digiwin.athena.domain.core.app.Application;
import com.digiwin.athena.domain.core.app.ApplicationRelation;
import com.digiwin.athena.domain.core.app.FunctionCtrl;
import com.digiwin.athena.domain.core.dtdflow.CustBizTask;
import com.digiwin.athena.domain.core.dtdflow.Asserts;
import com.digiwin.athena.domain.core.flow.FlowGraph;
import com.digiwin.athena.domain.core.flow.FlowNode;
import com.digiwin.athena.domain.core.view.PageUIElement;
import com.digiwin.athena.domain.core.view.PageView;
import com.digiwin.athena.domain.customBizTask.CustBizTaskConfigDTO;
import com.digiwin.athena.domain.definition.ReminderRule;
import com.digiwin.athena.domain.definition.features.DataFeatureSet;
import com.digiwin.athena.domain.definition.features.DataTagging;
import com.digiwin.athena.dto.*;
import com.digiwin.athena.kmservice.dtd.DtdService;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kmservice.support.DapContext;
import com.digiwin.athena.kmservice.utils.I18nUtils;
import com.digiwin.athena.kmservice.utils.MergeUtil;
import com.digiwin.athena.kmservice.utils.ServiceUtils;
import com.digiwin.athena.repository.neo4j.TaskRepository;
import com.google.common.collect.Maps;
import com.mongodb.client.model.Filters;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.bson.conversions.Bson;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executor;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 *  任务和项目（从 kg 迁移过来的）相关 Service.
 **/

@Lang
@MyExceptionHandler
@Service
@Slf4j
public class DataMapTaskService implements ITaskService {

    public static final String SUMMARY = "summary";

    public static final String BUSINESS_INFO = "businessInfo";

    public static final String SEARCH_KEY = "searchKey";

    public static final String PROJECT_INFO = "projectInfo";

    @Autowired
    ProjectService projectService;

    @Autowired
    DataMapPickService dataPickService;

    @Autowired
    DataMapParadigmService dataMapParadigmService;

    @Autowired
    private DataMapViewService dataMapViewService;

    @Autowired
    private IKnowledgeGraphService knowledgeGraphService;

    @Autowired
    private DataMapKgService dataMapKgService;

    @Autowired
    private ProductNameResolver productNameResolver;

    @Autowired
    KmService kmService;

    @Autowired
    DataMapIamService dataMapIamService;

    @Autowired
    private Executor taskExecutor;

    @Autowired
    private DbKnowledgeGraphService dbKnowledgeGraphService;

    @Autowired
    DtdService dtdService;

    @Autowired
    TaskRepository taskRepository;


    @Override
    public TaskPathResp postExecutePath(com.digiwin.athena.kmservice.povo.DataRequest request) throws DWException {

        return dtdService.postExecutePath(request);
    }

    /**
     * 判断当前项目状态是否是最终状态
     * @param current 项目当前状态
     * @param to 项目最终状态
     * @param project 项目
     * @return 判断结果
     */
    private boolean reachProjectEnd(String current, String to, Project project) {
        if (to.equals(current)) {
            return true;
        }
        if (null != project.getEnds()) {
            for (DataState state : project.getEnds()) {
                if (state.getCode().equals(current)) {
                    return true;
                }
            }
        }

        return false;
    }

    @Override
    public List<Task> postTasks(BasicQuery query) throws DWException {
        return dataPickService.findBetters(query, Task.class);
    }

    @Override
    public Task postTask(BasicQuery request) throws DWException {
        return dataPickService.findBetter(request, Task.class);
    }

   // @Override
    public TaskPathResp postExecutePathBak(DataRequest request) throws DWException {
        com.digiwin.athena.kmservice.utils.ServiceUtils.getContext().setEocInfo(request.getEocInfo());
        TaskPathResp resp = new TaskPathResp();
        String tenantId = com.digiwin.athena.kmservice.utils.ServiceUtils.getTenantId();
        request.setTenantId(tenantId);
        // 判断项目信息
        Project pro = projectService.getProjectByCode(request.getProjectCode());
        if (null == pro) {
            throw new DWException("P.DM.500.0001", I18nUtils.getValue("P.DM.500.0001") + request);
        }
        String topProjectCode = pro.getCode();
        if (null != pro.getPrimaryProjectCode()) {
            topProjectCode = pro.getPrimaryProjectCode();
        }
        Application application = dbKnowledgeGraphService.tenantApp(pro.getApplication());
        // 应用是否悬停
        boolean individualAll = Optional.ofNullable(application).map(Application::getIndividualAll).orElse(false);
        Project up = projectService.getUserProject(topProjectCode, request.getFromStateCode());
        resp.setStartProject(up);
        DataState begin = getDataState(request.getFromStateCode());
        resp.setFromState(begin);
        if (reachProjectEnd(request.getFromStateCode(), request.getToStateCode(), pro)) {
            return resp;
        }
        // 找任务
        List<Task> tasks = bestTaskByFrom(request.getFromStateCode(), request, pro);
        if (!tasks.isEmpty()) {
            Map<String, List<Task>> groups = new HashMap<>();
            // 在悬停情况下，移除掉租户id等于SYSTEM的
            tasks.removeIf(t -> individualAll && "SYSTEM".equals(t.getTenantId()));
            tasks.forEach(t -> {
                if (null == t.getGroup()) {
                    t.setGroup("default");
                }
                groups.computeIfAbsent(t.getGroup(), k -> new ArrayList<>()).add(t);
            });
            for (Map.Entry<String, List<Task>> entry : groups.entrySet()) {
                List<Task> v = entry.getValue();
                sortTasks(v, request);
                TaskPathVo vo = new TaskPathVo();
                vo.setTasks(v);
                Task first = v.get(0);
                first = fillTaskDetail(first);
                dealFeatures(first);
                vo.setTask(first);
                v.set(0, first);
                resp.getPaths().add(vo);
            }
        }
        return resp;
    }

    /**
     * 根据数据特征得分进行排序
     * @param tasks 任务集合
     * @param request
     */
    private void sortTasks(List<Task> tasks, DataRequest request) {
        for (Task t : tasks) {
            if (null == t.getPriority()) {
                t.setPriority(0);
            }
            if (null == t.getScore()) {
                t.setScore(0);
            }
            t.setScore(t.getPriority() * 100 + t.getScore());
            if (CollectionUtils.isNotEmpty(t.getDataFeatures()) && !request.getDateFeatures().isEmpty()) {
                for (DataTagging dt : t.getDataFeatures()) {
                    if (request.getDateFeatures().contains(dt.getCode())) {
                        t.setScore(t.getScore() + dt.getWeight());
                    }
                }
            }
        }
        tasks.sort((o1, o2) -> o2.getScore() - o1.getScore());
    }

    @Override
    public Task getTaskByCode(String code) throws DWException {
        Task detail = dataPickService.findBetter(code, Task.class);
        detail = fillTaskDetail(detail);
        if (detail == null) {
            return null;
        }
        dealFeatures(detail);
        return detail;
    }

    @Deprecated
    @Override
    public List<TaskSchedule> getTaskSchedule(String code) throws DWException {
        Map<String, Object> q = new HashMap<>();
        q.put("taskCode", code);
        return dataPickService.findBetters(BasicQuery.of(q, null), TaskSchedule.class);
    }

    @Override
    public Task postTaskByCode(DataRequest request) throws DWException {
        Map<String, Object> param = new HashMap<>();
        param.put("code", request.getCode());
        Task detail = dataPickService.findBetter(BasicQuery.of(param, request.getEocInfo()), Task.class);
        detail = fillTaskDetail(detail);
        dealFeatures(detail);
        return detail;
    }

    private void dealFeatures(Task detail) {
        if (null == detail) {
            return;
        }
        detail.setFeatures(detail.getFeeInfo());
    }

    @Override
    public TaskAndActivity getTaskAndMilestoneActivity(String code) throws DWException {
        TaskAndActivity resp = new TaskAndActivity();
        Task t = getTaskByCode(code);
        resp.setTask(t);
        if (null != t.getFlow()) {
            Activity activity = milestoneActivity(t.getFlow());
            resp.setActivity(activity);
        }
        return resp;
    }

    @Override
    public FlowGraph getFlowByCode(String code) throws DWException {
        return dataPickService.findByCode(code, FlowGraph.class);
    }

    @Override
    public Activity getActivityByCode(String code) throws DWException {
        Activity activity = dataPickService.findBetter(code, Activity.class);
        return processActivityConditions(activity);
    }

    private Activity processActivityConditions(Activity activity) {
        if (null != activity) {
            if ("PARADIGM".equals(activity.getPattern())) {
                try {
                    String expression = dataMapParadigmService.getExpression((String) activity.getConfig().get("_mechanismCode"), (String) activity.getConfig().get("_dataCode"));
                    if (null != expression) {
                        List<GroupCondition> conditions = JSON.parseArray(JSON.toJSONString(activity.getConfig().get("conditions")), GroupCondition.class);
                        conditions.forEach(cond -> {
                            if (null != cond.getCondition()) {
                                cond.setCondition(expression);
                            }
                        });
                        activity.getConfig().put("conditions", conditions);
                    }
                } catch (Exception e) {
                    log.error(e.getMessage(), e);
                }

            }
        }

        return activity;
    }

    @Override
    public Activity postActivityByCode(DataRequest request) throws DWException {

        Activity activity;
        Map<String, Object> param = new HashMap<>();
        param.put("code", request.getCode());
        activity = dataPickService.findBetter(BasicQuery.of(param, request.getEocInfo()), Activity.class);
        return processActivityConditions(activity);
    }

    @Override
    public DataDescription getDataDescription(String code) throws DWException {
        return dataPickService.findByCode(code, DataDescription.class);
    }

    @Override
    public DataDescription getDataDescriptionByStateCode(String stateCode) throws DWException {

        DataState desc = dataPickService.findByCode(stateCode, DataState.class);
        if (null != desc) {
            return getDataDescription(desc.getDataCode());
        }

        return null;
    }

    public DataState getDataStateBasic(String code) throws DWException {
        return dataPickService.findByCode(code, DataState.class);
    }

    @Override
    public DataState getDataState(String code) throws DWException {
        return getDataStateBasic(code);
    }

    @Override
    public DataStateVO getDataStateWithDesc(String code) throws DWException {
        DataState state = getDataStateBasic(code);
        DataDescription desc = getDataDescriptionByStateCode(code);
        DataStateVO vo = new DataStateVO();
        vo.setDataState(state);
        vo.setDataDesc(desc);
        return processDataFeatures(vo);
    }

    private DataStateVO processDataFeatures(DataStateVO vo) throws DWBusinessException {
        if (null != vo.getDataState() && CollectionUtils.isNotEmpty(vo.getDataState().getDataFeatureSets())) {
            Map<String, Object> query = new HashMap<>();
            query.put("code", vo.getDataState().getDataFeatureSets());
            List<DataFeatureSet> tags = dataPickService.find(BasicQuery.of(query, null), DataFeatureSet.class);
            vo.setDataFeatures(tags);
        }
        return vo;
    }

    @Deprecated
    @Override
    public DataStateVO getDataStateTagged(String code) throws DWException {

        DataStateVO vo = getDataStateWithDesc(code);
        return processDataFeatures(vo);
    }

    public List<DataStateVO> postDataStateNear(DataRequest request) throws DWException {
        List<DataStateVO> result = new ArrayList<>();
        if(null==request.getFromStateCode()){return result;}
        List<String> states = taskRepository.nearStates(request.getFromStateCode());
        states.removeAll(request.getExcludeStateCodes());
        if(states.size()>0){
            Map<String,Object> query = new HashMap<>();
            query.put("code",states);
            List<DataState> dataStates = dataPickService.findBetters(BasicQuery.of(query, null), DataState.class);
            if(dataStates.size()==0){return result;}
            Set<String> tagCodes = new HashSet<>();
            dataStates.forEach(dataState -> {
                if(null!=dataState.getDataFeatureSets()){
                    tagCodes.addAll(dataState.getDataFeatureSets());
                }
            });
            Map<String, DataFeatureSet> mapTag = new HashMap<>();
            if(tagCodes.size()>0){
                Map<String,Object> query2 = new HashMap<>();
                query2.put("code",tagCodes);
                List<DataFeatureSet> tags = dataPickService.findBetters(BasicQuery.of(query2, null), DataFeatureSet.class);
                tags.forEach(tag->{
                    mapTag.put(tag.getCode(),tag);
                });
            }

            for(DataState dataState:dataStates){
                DataStateVO vo = new DataStateVO();
                vo.setDataState(dataState);
                if(null!=dataState.getDataFeatureSets()){
                    dataState.getDataFeatureSets().forEach(dt->{
                        vo.getDataFeatures().add(mapTag.get(dt));
                    });
                }
                result.add(vo);
            }
        }

        return result;
    }

    @Override
    public DataFeatureSet getDataFeatureSet(String code) throws DWException {
        return dataPickService.findByCode(code, DataFeatureSet.class);
    }

    @Override
    public Object clearTenantTaskField(String tenantId, String taskCode, String version, List<String> fields) {
        if (tenantId == null || taskCode == null || CollectionUtils.isEmpty(fields)) {
            return ResponseEntity.ok();
        }
        // 先查询有没有租户级数据
        Query query = new Query(Criteria.where("tenantId").is(tenantId).and("code").is(taskCode));
        Task tenantTask = dataPickService.tenantTemplate().findOne(query, Task.class, dataPickService.tenantCol(Task.class));
        if (tenantTask != null) {
            Update update = new Update();
            for (String field : fields) {
                update.unset(field);
            }
            dataPickService.tenantTemplate().updateMulti(query, update, dataPickService.tenantCol(Task.class));
        }
        return null;
    }

    /**
     * 根据项目起始状态查询任务集合
     *
     * @param from    项目的起始状态
     * @param request 请求
     * @param pro 项目
     * @return
     * @throws DWBusinessException
     */
    private List<Task> bestTaskByFrom(String from, DataRequest request, Project pro) throws DWBusinessException {
        List<String> codes = new ArrayList<>();
        codes.add(from);
        Map<String, Object> param = new HashMap<>();
        param.put("from", codes);
        List<Task> tasks = dataPickService.find(BasicQuery.of(param, request.getEocInfo()), Task.class);
        // 使用迭代器迭代
        Iterator<Task> iterator = tasks.iterator();
        List<String> assetTaskCodes = Optional.ofNullable(pro.getAsserts()).map(Asserts::getTaskCodes).orElse(Collections.emptyList());
        while (iterator.hasNext()) {
            Task next = iterator.next();
            // DTD复用专案，
            // - 如果任务没有应用信息可运行--机制任务
            // - 如果任务属于当前应用正常执行
            // - 如果任务不属于当前应用但在资产列表范围内可运行
            if (next.getApplication() == null || next.getApplication().equals(pro.getApplication()) || assetTaskCodes.contains(next.getCode())) {
                for (StateMap sm : next.getStateMaps()) {
                    if (from.equals(sm.getInput())) {
                        if (null != sm.getGroup()) {
                            next.setGroup(sm.getGroup());
                        }
                        break;
                    }
                }
            } else {
                iterator.remove();
            }
        }

        return tasks;
    }

    /**
     * 根据流程节点获取里程碑任务
     * @param flow 流程数据
     * @return activity
     * @throws DWException DwException
     */
    public Activity milestoneActivity(FlowGraph flow) throws DWException {

        List<String> codes = new ArrayList<>();
        for (FlowNode node : flow.getNodes()) {
            if ("activity".equalsIgnoreCase(node.getType())) {
                codes.add(node.getActivityCode());
            }
        }
        if (!codes.isEmpty()) {
            Map<String, Object> q = new HashMap<>();
            q.put("code", codes);
            q.put("milestone", true);
            List<Activity> activities = dataPickService.findBetters(BasicQuery.of(q, null), Activity.class);
            if (!activities.isEmpty()) {
                return activities.get(0);
            }

        }

        return null;
    }

    private Task fillTaskDetail(Task task) throws DWException {
        if (task == null) {
            return null;
        }
        task = dataPickService.findBetter(task.getCode(), Task.class);
        if (null != task.getFlowCode()) {
            FlowGraph graph = dataPickService.findByCode(task.getFlowCode(), FlowGraph.class);
            task.setFlow(graph);
        }
        return task;
    }

    @Override
    public Map<String, Object> getDefinition(String taskId) throws Exception {
        return this.getDefinitionByTaskId(taskId);
    }

    /*
     * 通过项目code查询项目信息，根据组件列表移除phases中被禁用的任务。 todo 遗留问题：里程碑内任务全部移除的话，phase应该也被移除
     */
    @Override
    public Map<String, Object> getDefinitionByTaskId(String taskId) throws Exception {
        String tenantId = ServiceUtils.getTenantId();
        ViewRequest viewRequest = new ViewRequest();
        viewRequest.setProjectCode(taskId);
        Map<String, Object> dm = dataMapViewService.postProject(viewRequest);
        try {
            filterTaskByTenantAppComponentList(dm, tenantId, taskId);
        } catch (Exception e) {
            log.error("Filter task error!", e);
        }
        return dm;
    }

    @SuppressWarnings("unchecked")
    @Override
    public Map<String, Object> getDefinitionWithMultiLanguage(String taskId, String pageCode) throws Exception {
        Map<String, Object> task = this.getDefinitionByTaskId(taskId);
        if (null == task) {
            return null;
        }
        try {
            Map<String, Object> pages = (Map<String, Object>) task.get("pages");
            if (null != pages) {
                task.put("pages", pages.get(pageCode));
            }
            processSummaryFields(pageCode, task);
        } catch (Exception e) {
            log.warn("processSummaryFields error:", e);
        }
        return task;
    }

    @Override
    public Object postDefinitionByTaskIdAndOperationUnit(String taskId, OperationUnit operationUnit) throws Exception {
        String tenantId = ServiceUtils.getTenantId();
        ViewRequest viewRequest = new ViewRequest();
        viewRequest.setProjectCode(taskId);
        viewRequest.setEocInfo(KmService.ecoInfo(operationUnit));
        Map<String, Object> dm = dataMapViewService.postProject(viewRequest);
        try {
            filterTaskByTenantAppComponentList(dm, tenantId, taskId);
        } catch (Exception e) {
            log.error("Filter task error!", e);
        }
        return dm;
    }

    @Deprecated
    @Override
    public Map<String, Object> getActivities(String taskId) throws Exception {
        return getActivitiesWithMultiLanguage(taskId);
    }

    @Override
    public Map<String, Object> getActivitiesWithMultiLanguage(String taskId) throws Exception {
        ViewRequest viewRequest = new ViewRequest();
        viewRequest.setProjectCode(taskId);
        return dataMapViewService.postTasks(viewRequest);
    }

    @Override
    // @Cache(key = CacheConfig.cachePrefix_tenantUserLang+"'getActivityDefinition:'+activityId+pageCode")
    public Map<String, Object> getActivityDefinition(String activityId, String pageCode) throws Exception {
        Map<String, Object> dm = getTaskInfoFromDataMap(null, activityId, pageCode, ServiceUtils.getTenantId(), null);
        if (null == dm) {
            return null;
        }
        try {
            Application application = kmService.getAppByCodeAndType(activityId, "activity");
            processApplication(activityId, dm, application);
            processSummaryFields(pageCode, dm);
        } catch (Exception e) {
            log.warn("parse activity error:", e);
        }
        return dm;
    }

    @Override
    public Map<String, Object> getActivityDefinitionWithMergeRule(String taskId, String activityId, String pageCode) throws Exception {
        ApplicationRelation applicationRelation = knowledgeGraphService.relation(activityId, null);
        Application application = null;
        if (null != applicationRelation) {
            application = knowledgeGraphService.application(applicationRelation.getAppCode());
        }
        if (application != null && application.getLang() != null) {
            Map nameLangMap = application.getLang().get("name");
            String language = null == nameLangMap.get(ServiceUtils.getCurrentLocale()) ? application.getName() : (String) nameLangMap.get(ServiceUtils.getCurrentLocale());
            applicationRelation.setAppName(org.springframework.util.StringUtils.isEmpty(language) ? applicationRelation.getAppName() : language);
        }
        Map<String, Object> dm = getTaskInfoFromDataMap(taskId, activityId, pageCode, ServiceUtils.getTenantId(), null);
        if (null == dm) {
            return dm;
        }
        try {
            processApplication(activityId, dm, application);
            processSummaryFields(pageCode, dm);
            if (null != application) {
                processMergeRule(dm, taskId, activityId, application.getFunctionCtrl());
            }
            handleReminderRule(dm);
        } catch (Exception e) {
            log.warn("getActivityDefinitionWithMergeRule:{}", e.getMessage());
        }
        return dm;
    }

    private void processMergeRule(Map<String, Object> task, String taskId, String activityId, FunctionCtrl functionCtrl) throws Exception {
        boolean enableCustomize = null == functionCtrl ? false : functionCtrl.isTaskCardMergeCustomizeFlag();
        if (!enableCustomize) {
            return;
        }
        Map<String, Object> activityConfig = getActivityConfig(taskId, activityId);
        MergeRule mergeRule = getTenantMergeConfig("task", activityId);
        if (null == mergeRule || mergeRule.getDelete()) {
            return;
        } else {
            task.put("merge", mergeRule.getMerge());
            // 将租户级mergeRule转为应用级结构
            Map<String, Object> mergeConfig = MergeRule.convertToMergeConfig(mergeRule.getMergeRules(), activityConfig);
            ((Map<String, Object>) task.get("config")).put("mergeConfig", mergeConfig);
        }
    }

    private MergeRule getTenantMergeConfig(String type, String code) throws Exception {
        String tenantId = ServiceUtils.getTenantId();
        String tenantVersion = dataPickService.tenantVersion(tenantId);
        Bson bson = Filters.and(
                Filters.eq("type", type),
                Filters.eq("code", code),
                Filters.eq("tenantId", tenantId),
                Filters.eq("version", tenantVersion));
        return knowledgeGraphService.findOne(bson, MergeRule.class, "mergeRule", KnowledgeGraphDb.TENANT);
    }

    /**
     * 处理催办规则
     * @param task 任务
     */
    private void handleReminderRule(Map<String, Object> task) {
        Object reminderRuleObject = task.get("reminderRule");
        if (reminderRuleObject == null) {
            return;
        }
        ReminderRule reminderRule = JSONObject.parseObject(JSONObject.toJSONString(reminderRuleObject), ReminderRule.class);
        // #17520需求，本次需求之前旧数据没有rules，需要兼容一下
        if (org.springframework.util.CollectionUtils.isEmpty(reminderRule.getRules())) {
            String currentLocale = ServiceUtils.getCurrentLocale();
            String titlePrefix = "规则";
            if ("zh_TW".equals(currentLocale)) {
                titlePrefix = "規則";
            } else if ("en_US".equals(currentLocale)) {
                titlePrefix = "rule";
            }
            Map<String, Object> rule = new HashMap<>();
            rule.put("aheadTime", reminderRule.getAheadTime());
            rule.put("notices", reminderRule.getNotices());
            rule.put("rule", reminderRule.getRule());
            rule.put("title", titlePrefix + "1");
            rule.put("identities", Collections.emptyList());
            reminderRule.setRules(Collections.singletonList(rule));
            reminderRule.setRule(null);
            reminderRule.setNotices(null);
            reminderRule.setAheadTime(null);
        }
        if (!org.springframework.util.CollectionUtils.isEmpty(reminderRule.getRules())) {
            for (Map<String, Object> ruleItem : reminderRule.getRules()) {
                Map map = MapUtils.getMap(ruleItem, "rule", Collections.emptyMap());
                String ruleType = MapUtils.getString(map, "rule");
                if ("count".equals(ruleType)) {
                    map.put("interval", null);
                } else {
                    map.put("count", null);
                }
            }
        }
        task.put("reminderRule", reminderRule);
    }

    @Override
    public Map<String, Object> getActivityConfig(String taskId, String activityId) throws Exception {
        Task task = dataPickService.findByCode(activityId, Task.class);
        Map<String, Object> config = task.getConfig();
        if (null == config) {
            config = new HashMap<>();
        }
        config.put("merge", task.getMerge());
        return config;
    }

    @Override
    // TODO
    // @Cache(key = CacheConfig.cachePrefix_tenantUserLang+"'postActivityDefinition:'+taskId+activityId+pageCode+operationUnit")
    public Map<String, Object> postActivityDefinition(String taskId, String activityId, String pageCode, OperationUnit operationUnit) throws Exception {
        return getTaskInfoFromDataMap(taskId, activityId, pageCode, ServiceUtils.getTenantId(), operationUnit);
    }

    @Override
    // TODO
    // @Cache(key = CacheConfig.cachePrefix_tenantUserLang+"'getActivityStartTargetCode:'+taskId+activityId+pageCode")
    public Map<String, Object> getActivityStartTargetCode(String taskId, String activityId, String pageCode) throws Exception {
        ViewRequest viewRequest = new ViewRequest();
        viewRequest.setProjectCode(taskId);
        viewRequest.setTaskCode(activityId);
        viewRequest.setPageCode(pageCode);
        return dataMapViewService.postActivityStartTargetCode(viewRequest);
    }

    @Override
    public Map<String, Object> getTaskInfoFromDataMap(String activityId) throws DWException {
        return getTaskInfoFromDataMap(null, activityId, null, null, null);
    }

    @Override
    public Map<String, Object> getDetail(String taskId) throws DWException {
        ViewRequest viewRequest = new ViewRequest();
        viewRequest.setProjectCode(taskId);
        return dataMapViewService.postProject(viewRequest);
    }

    @Override
    public Map<String, Object> getInputData(String taskId) throws DWException {
        ViewRequest viewRequest = new ViewRequest();
        viewRequest.setProjectCode(taskId);
        return dataMapViewService.postProjectInputData(viewRequest);
    }

    @Override
    public String postUpdateTaskConfig(Project data) throws DWException {
        dataMapViewService.postUpdateProject(data);
        return "success";
    }

    // @Override
    // public List<TaskDTO> postTaskByApp(String applicationCode, OperationUnit operationUnit) throws Exception {
    // String tenantId = ServiceUtils.getTenantId();
    // List<String> codes = kmService.getCodeByTypeAndAppCodeWithComponentList(applicationCode, "task", tenantId);
    // Map<String, Object> q = new HashMap<>();
    // q.put("code", codes);
    // BasicQuery basicQuery = BasicQuery.of(q, KmService.ecoInfo(operationUnit));
    // List<Project> tasks = dataPickService.find(basicQuery, Project.class);
    // List<TaskDTO> result = new ArrayList<>();
    // for (Project project : tasks) {
    // TaskDTO dto = new TaskDTO();
    // dto.setTask(project);
    // result.add(dto);
    // }
    //
    // return result;
    // }

    @Override
    public List<TaskDTO> postTaskByApp(String applicationCode, OperationUnit operationUnit) throws Exception {
        String tenantId = ServiceUtils.getTenantId();
        String tenantVersion = dataPickService.tenantVersion(tenantId);
        List<String> taskIdList = kmService.getCodeByTypeAndAppCodeWithComponentList(applicationCode, "task", tenantId);
        DWServiceContext context = DWServiceContext.getContext();
        if (org.springframework.util.CollectionUtils.isEmpty(taskIdList)) {
            return Collections.emptyList();
        }
        List<TaskDTO> result = new CopyOnWriteArrayList<>();
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        DapContext dapContext = ServiceUtils.getContext();
        CompletableFuture
                .allOf(
                        taskIdList.stream().map(taskId -> CompletableFuture.runAsync(() -> {
                            DWServiceContext.setContext(context);
                            RequestContextHolder.setRequestAttributes(requestAttributes);
                            ServiceUtils.setContext(dapContext);
                            try {
                                TaskDTO taskDTO = new TaskDTO();
                                Map<String, Object> definitionWithMultiLanguage = (Map<String, Object>) this.postDefinitionByTaskIdAndOperationUnit(taskId, operationUnit);
                                if (null == definitionWithMultiLanguage) {
                                    return;
                                }
                                if (Boolean.TRUE.equals(JSONPath.eval(definitionWithMultiLanguage, "$.config.hideDeliverySetting"))) {
                                    return;
                                }
                                taskDTO.setTask(definitionWithMultiLanguage);
                                Bson filter = Filters.and(
                                        Filters.eq("taskCode", taskId),
                                        Filters.eq("version", tenantVersion));
                                List<TaskMonitorRule> taskMechanism = knowledgeGraphService.find(filter, TaskMonitorRule.class, "taskMonitorRule", KnowledgeGraphDb.SYSTEM);
                                if (!org.springframework.util.CollectionUtils.isEmpty(taskMechanism)) {
                                    List<String> collect = taskMechanism.stream().map(TaskMonitorRule::getMonitorRuleCode).collect(Collectors.toList());
                                    taskDTO.setRelatedMonitorRules(collect);
                                }
                                result.add(taskDTO);

                            } catch (Exception e) {
                                log.error("postTaskByApp error", e);
                            }
                        }, taskExecutor))
                                .toArray(CompletableFuture[]::new))
                .join();
        return result;
    }

    @Override
    public Object postFixTask(String taskId, OperationUnit operationUnit) {
        // TODO 代码逻辑比较奇怪，需要确认
        return null;
    }

    @Override
    public List<Map<String, Object>> getPageUIElement(String code, String activityId, String pageCode) throws Exception {
        String tenantId = ServiceUtils.getTenantId();
        return getPageUIElementFromDataMap(code, activityId, pageCode, tenantId);
    }

    /*
     * 查询用户授权的手动发起列表 查询授权的应用列表=》查询应用下的项目=》删除组件列表中禁用的=》删除公共应用项目没有被搭售的
     */
    @Override
    public List<ProjectVo> getManualList() throws Exception {
        UserComponentDto dto = kmService.userComponents("task",null, ServiceUtils.getLocale());
        if(null==dto){return new ArrayList<>();}
        Set<String> codesFiltered = dto.getCodesFiltered();
        List<String> commonAppCodes = dto.getCommonAppCodes();
        Map<String, ApplicationRelation> codeAppCode = dto.getCodeRelations();
        //查项目基础信息
        Map<String, Object> query2 = new HashMap<>();
        query2.put("code",codesFiltered);
        query2.put("manualAble", true);
        List<ProjectVo> projects = dataPickService.find(BasicQuery.of(query2, null), ProjectVo.class);

        Set<String> appCodesHasAuthPrefix = new HashSet<>();
        for (ProjectVo project : projects) {
            project.setCommonApp(false);
            String appCode = codeAppCode.get(project.getCode()).getAppCode();
            if (commonAppCodes.contains(appCode)) {
                project.setCommonApp(true);
            }
            if (StringUtils.isNotEmpty(project.getAuthorityPrefix())) {
                appCodesHasAuthPrefix.add(appCode);
            }
        }
        //用户授权过滤
        Map<String,String> userPemissions = kmService.userPermission(appCodesHasAuthPrefix,"allow");
        Iterator<ProjectVo> iterator2 = projects.iterator();
        while (iterator2.hasNext()) {
            ProjectVo project = iterator2.next();
            project.setEffect("allow");
            if(shouldRemoveByPermission(project,userPemissions)){
                iterator2.remove();
            }
        }
        //添加分组信息
        addGroupInfo(projects);
        return projects;
    }

    private boolean shouldRemoveByPermission(ProjectVo project,Map<String,String> userPemissions){
        String eff = userPemissions.get(project.getCode());
        if(StringUtils.isNotEmpty(project.getAuthorityPrefix()) && !userPemissions.containsKey(project.getCode())){
            return true;
        }
        if(Boolean.TRUE.equals(project.getCommonApp()) && StringUtils.isEmpty(project.getAuthorityPrefix())){
            return true;
        }
        //todo 临时
        if(null!=eff && !eff.equals(project.getEffect())){
            return true;
        }
        return false;
    }

    /*
     * 查询用户授权的手动发起列表 查询授权的应用列表=》查询应用下的项目=》删除组件列表中禁用的=》删除公共应用项目没有被搭售的
     */
    @Override
    public List<ProjectVo> getManualListMobile() throws Exception {
        System.out.println(Thread.currentThread().getName()+" datamap getManualListMobile...");
        UserComponentDto dto = kmService.userComponents("task",null, ServiceUtils.getLocale());
        if(null==dto){return new ArrayList<>();}
        Set<String> codesFiltered = dto.getCodesFiltered();
        List<String> commonAppCodes = dto.getCommonAppCodes();
        Map<String, ApplicationRelation> codeAppCode = dto.getCodeRelations();
        //查项目基础信息
        Map<String, Object> query2 = new HashMap<>();
        query2.put("code",codesFiltered);
        query2.put("manualAble", true);
        List<ProjectVo> projects = dataPickService.find(BasicQuery.of(query2, null), ProjectVo.class);

        Set<String> appCodesHasAuthPrefix = new HashSet<>();
        for (ProjectVo project : projects) {
            project.setCommonApp(false);
            String appCode = codeAppCode.get(project.getCode()).getAppCode();
            if (commonAppCodes.contains(appCode)) {
                project.setCommonApp(true);
            }
            if (StringUtils.isNotEmpty(project.getAuthorityPrefix())) {
                appCodesHasAuthPrefix.add(appCode);
            }
        }
        //用户授权过滤
        Map<String,String> userPemissions = kmService.userPermission(appCodesHasAuthPrefix,"allow");
        Iterator<ProjectVo> iterator2 = projects.iterator();
        while (iterator2.hasNext()) {
            ProjectVo project = iterator2.next();
            project.setEffect("allow");
            if(StringUtils.isNotEmpty(project.getAuthorityPrefix()) && !userPemissions.containsKey(project.getCode())){
                iterator2.remove();
            }
            String eff = userPemissions.get(project.getCode());
            if(StringUtils.isNotEmpty(eff)){
                project.setEffect(eff);
            }
        }
        List<String> codes = projects.stream().map(ProjectVo::getCode).collect(Collectors.toList());
        Map<String, Object> param = new HashMap<>();
        param.put("project", codes);
        param.put("pageType", 1);
        List<PageView> pageViewList = dataPickService.find(BasicQuery.of(param, null), PageView.class);
        Map<String, PageView> pageViewMap = new HashMap<>();
        pageViewList.forEach(pv -> pageViewMap.put(pv.getProject(), pv));
        for (ProjectVo vo : projects) {
            PageView view = pageViewMap.get(vo.getCode());
            if (null != view) {
                vo.setPresetActivities(Collections.singletonList(view));
            }
        }
        //添加分组信息
        addGroupInfo(projects);
        return projects;
    }

    @Override
    public Map<String, Object> getManualPage(String code, String pageCode) throws Exception {
        Map<String, Object> pageViewMap = new HashMap<>();
        Map<String, Object> param = new HashMap<>();
        param.put("project", code);
        param.put("pageType", 1);
        BasicQuery query = BasicQuery.of(param, null);
        query.setReturnFields(Arrays.asList("code"));
        PageView pageView = dataPickService.findOne(query, PageView.class);
        if(null!=pageView){
            return getActivityDefinition(pageView.getCode(),pageCode);
        }
        return pageViewMap;
    }

    @Override
    public PageView getManualPageCode(String code) throws Exception {
        Map<String, Object> param = new HashMap<>();
        param.put("project", code);
        param.put("pageType", 1);
        BasicQuery query = BasicQuery.of(param, null);
        query.setReturnFields(Arrays.asList("code"));
        PageView pageView = dataPickService.findOne(query, PageView.class);
        return pageView;
    }

    private void addGroupInfo(List<ProjectVo> projects) throws DWBusinessException {
        String lang = DWResourceBundleUtils.getCurrentLocale().toString();
        String tenantVersion = dataPickService.tenantVersion();
        setGroupInfoWithGroupCode(
                projects.stream()
                        .filter(project -> null != project.getGroupCode()).collect(Collectors.toList()),
                lang, tenantVersion);
        setGroupInfoWithoutGroupCode(
                projects.stream()
                        .filter(project -> null == project.getGroupCode()).collect(Collectors.toList()),
                lang, tenantVersion);
    }

    private void setGroupInfoWithGroupCode(List<ProjectVo> projectsWithGroupCode, String lang, String tenantVersion) {
        if (projectsWithGroupCode.isEmpty()) {
            return;
        }
        Set<String> groupCodes =
                projectsWithGroupCode.stream().map(ProjectVo::getGroupCode).collect(Collectors.toSet());
        Bson filter = Filters.and(
                Filters.exists("appCode", false),
                Filters.in("groupCode", groupCodes),
                Filters.eq("version", tenantVersion));
        List<ApplicationGroup> applicationGroups = knowledgeGraphService.find(filter, ApplicationGroup.class, "applicationGroup", KnowledgeGraphDb.SYSTEM);
        Map<String, ApplicationGroup> applicationGroupMap = applicationGroups.stream()
                .collect(Collectors.toMap(ApplicationGroup::getGroupCode, Function.identity(), (a, b) -> a));
        for (ProjectVo project : projectsWithGroupCode) {
            ApplicationGroup applicationGroup = applicationGroupMap.get(project.getGroupCode());
            if (null != applicationGroup) {
                setGroupInfo(project, applicationGroup, lang);
            }
        }
    }

    private void setGroupInfoWithoutGroupCode(List<ProjectVo> projectsWithoutGroupCode, String lang, String tenantVersion) {
        if (projectsWithoutGroupCode.isEmpty()) {
            return;
        }
        Set<String> appCodes = projectsWithoutGroupCode.stream()
                .map(this::getAppCode)
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(Collectors.toSet());
        Set<String> projectCodes = projectsWithoutGroupCode.stream()
                .map(BaseEntity::getCode)
                .collect(Collectors.toSet());
        Bson filter = Filters.and(
                Filters.in("appCode", appCodes),
                Filters.in("activityCode", projectCodes),
                Filters.eq("version", tenantVersion));
        List<ApplicationGroup> applicationGroups = knowledgeGraphService.find(filter, ApplicationGroup.class, "applicationGroup", KnowledgeGraphDb.SYSTEM);
        Map<String, ApplicationGroup> applicationGroupMap = applicationGroups.stream()
                .collect(Collectors
                        .toMap(each -> each.getAppCode() + ":" + each.getActivityCode(),
                                Function.identity(), (a, b) -> a));
        for (ProjectVo project : projectsWithoutGroupCode) {
            Optional<String> appCode = getAppCode(project);
            if (!appCode.isPresent()) {
                continue;
            }
            String projectCode = project.getCode();
            ApplicationGroup applicationGroup = applicationGroupMap.get(appCode.get() + ":" + projectCode);
            if (null != applicationGroup) {
                setGroupInfo(project, applicationGroup, lang);
            }
        }
    }

    private static void setGroupInfo(ProjectVo project, ApplicationGroup applicationGroup, String lang) {
        project.setGroupCode(applicationGroup.getGroupCode());
        project.setGroupSort(StringUtils.isEmpty(applicationGroup.getGroupSort()) ? null : applicationGroup.getGroupSort());
        String groupName = applicationGroup.getGroupName();
        if (null != applicationGroup.getLang()) {
            Map<String, String> nameLangMap = applicationGroup.getLang().get("groupName");
            if (null == nameLangMap) {
                nameLangMap = applicationGroup.getLang().get("name") == null ? Collections.emptyMap() : applicationGroup.getLang().get("name");
            }
            if (null != nameLangMap.get(lang)) {
                groupName = nameLangMap.get(lang);
            }
        }
        project.setGroupName(groupName);
        project.setClassifyCode(applicationGroup.getClassifyCode());
        project.setClassifyName(applicationGroup.getClassifyName());
    }

    private Optional<String> getAppCode(ProjectVo project) {
        if (null != project.getApplication()) {
            return Optional.of(project.getApplication());
        }
        if (null != project.getSourceId()) {
            return Optional.of(project.getSourceId());
        }
        return Optional.empty();
    }

    @SuppressWarnings("unchecked")
    private List<Map<String, Object>> getPageUIElementFromDataMap(String taskId, String activityId, String pageCode, String tenantId) throws DWException {
        ViewRequest viewRequest = new ViewRequest();
        viewRequest.setCode(taskId);
        viewRequest.setTaskCode(activityId);
        viewRequest.setPageCode(pageCode);
        List<PageUIElement> uiElements = (List<PageUIElement>) dataMapViewService.getPageUIElement(viewRequest);
        // TODO 由于设置 productName 的方法是从 kg 迁移的，所以为了适配，先将 PageUIElement 转为 Map
        List<Map<String, Object>> dm = new ArrayList<>();
        for (PageUIElement each : uiElements) {
            dm.add(MergeUtil.mergeObjectToMap(new HashMap<>(), each, null));
        }
        if (!dm.isEmpty()) {
            try {
                productNameResolver.batchSetProductInfoForList(dm, tenantId);
            } catch (Exception ee) {
                log.error("batchSetProductInfoForList error:{}", ee.toString());
            }
        }
        return dm;
    }

    private void processSummaryFields(String pageCode, Map<String, Object> result) throws DWBusinessException {
        processSysSummaryFields(result);
        // 获取租户级summaryFields
        processTenantSummaryFields(result, pageCode);
    }

    private Map<String, Object> getTaskInfoFromDataMap(String taskId, String activityId, String pageCode, String tenantId, OperationUnit operationUnit) throws DWException {
        ViewRequest viewRequest = new ViewRequest();
        viewRequest.setProjectCode(taskId);
        viewRequest.setTaskCode(activityId);
        viewRequest.setPageCode(pageCode);
        if (null != operationUnit) {
            viewRequest.getEocInfo().put("eoc_company_id", operationUnit.getEoc_company_id());
            viewRequest.getEocInfo().put("eoc_site_id", operationUnit.getEoc_site_id());
            viewRequest.getEocInfo().put("eoc_region_id", operationUnit.getEoc_region_id());
            viewRequest.getEocInfo().put("operation_unit_v2", operationUnit.getOperation_unit_v2());
        }
        Map<String, Object> dm = dataMapViewService.postTask(viewRequest);
        if (null != dm) {
            productNameResolver.batchSetProductInfoForTaskView(tenantId, dm);
        }
        return dm;
    }

    private void processApplication(String activityId, Map<String, Object> task, Application application) {
        String locale = ServiceUtils.getLocale();
        if (null != task) {
            if (null != application) {
                task.put("application", application.getCode());
                task.put("appName", ServiceUtils.localeName(application, "name", locale));
            }
            // dto
            task.put("activityId", activityId);
            task.put("activityName", ServiceUtils.localeName(task, "name", locale));
        }
    }

    /**
     * 过滤里程碑中有授权的数据
     * @param dmObject 项目数据
     * @param tenantId 租户
     * @param projectCode 项目code
     * @throws Exception exception
     */
    @SuppressWarnings("unchecked")
    private void filterTaskByTenantAppComponentList(Map<String, Object> dmObject, String tenantId, String projectCode) throws Exception {
        if (null == dmObject) {
            return;
        }
        List<Phase> phases = (List<Phase>) dmObject.get("phases");
        if (CollectionUtils.isEmpty(phases)) {
            return;
        }
        // 查询应用组件关系
        CustBizTaskConfigDTO taskConfigDTO = dataPickService.tenantTemplate().findOne(Query.query(Criteria.where("projectCode").is(projectCode)
                .and("tenantId").is(ServiceUtils.getTenantId())), CustBizTaskConfigDTO.class);
        // 查询可跳过的任务
        if(null!=taskConfigDTO && null!=taskConfigDTO.getBizTasks()){
            List<String> skipTaskCodeList = taskConfigDTO.getBizTasks().stream().map(v -> {
                if (v.getCustomSkip()) {
                    return v.getCode();
                }
                return null;
            }).filter(Objects::nonNull).distinct().collect(Collectors.toList());
            phases = phases.stream().filter(v -> !skipTaskCodeList.contains(v.getCode())).collect(Collectors.toList());
        }
        // 将过滤后的里程碑数据重新设置到项目数据中
        dmObject.put("phases", phases);

        String applicationCode = getAppCodeByApplication(projectCode);
        List<String> codes = new ArrayList<>();
        for (Phase phase : phases) {
            if (null != phase.getTaskCodes()) {
                codes.addAll(phase.getTaskCodes());
            }
        }
        // TODO replace Activity with applicationRelation enum
        List<String> disabledCodes = kmService.tenantDisableComponentCodes(applicationCode, tenantId, "Activity", codes);
        if (CollectionUtils.isEmpty(disabledCodes)) {
            return;
        }
        for (Phase phase : phases) {
            if (null != phase.getTaskCodes()) {
                phase.getTaskCodes().removeAll(disabledCodes);
            }
        }
    }

    /**
     * 根据项目code从应用组件关系表获取应用code
     * @param projectCode 项目code
     * @return appCode 应用code
     * @throws Exception exception
     */
    private String getAppCodeByApplication(String projectCode) throws Exception {
        ApplicationRelation applicationRelation = knowledgeGraphService.relation(projectCode, null);
        return applicationRelation == null ? null : applicationRelation.getAppCode();
    }

    /**
     * 是否自定义卡面
     *
     * @param functionCtrl 功能控制
     * @return result
     */
    private boolean isCardCustom(FunctionCtrl functionCtrl) {
        return functionCtrl != null
                && (functionCtrl.isProjectCardCustomizeFlag() || functionCtrl.isTaskCardCustomizeFlag());
    }

    @SuppressWarnings("unchecked")
    private void processSysSummaryFields(Map<String, Object> task) {
        if (null != task) {
            Map<String, Object> pages = (Map<String, Object>) task.get("pages");
            if (!org.springframework.util.CollectionUtils.isEmpty(pages)) {
                List<Map<String, Object>> dataStates = (List<Map<String, Object>>) pages.get("dataStates");
                List<Map<String, Object>> summaryFieldsCandidate = (List<Map<String, Object>>) pages.get("summaryFields");
                if (!org.springframework.util.CollectionUtils.isEmpty(dataStates)) {
                    // 新结构处理
                    dataStates.forEach(dataState -> {
                        if (SUMMARY.equals(dataState.get("type"))) {
                            task.put("summary", dataState);
                            task.put("summaryFields", (List<Map<String, Object>>) dataState.get("fieldMappings"));
                        }
                        if (BUSINESS_INFO.equals(dataState.get("type"))) {
                            task.put("businessInfo", dataState);
                        }
                        if (SEARCH_KEY.equals(dataState.get("type"))) {
                            task.put("searchKey", dataState);
                        }
                        if (PROJECT_INFO.equals(dataState.get("type"))) {
                            task.put("projectInfo", dataState);
                        }
                    });

                    // 不存在新结构，按原逻辑处理
                    if (!checkNewSummary(task)) {
                        List<Map<String, Object>> summaryFields = (List<Map<String, Object>>) dataStates.get(0).get("summaryFields");
                        task.put("summaryFields", summaryFields);
                    }
                } else if (!org.springframework.util.CollectionUtils.isEmpty(summaryFieldsCandidate)) {
                    task.put("summaryFields", summaryFieldsCandidate);
                }
            }
        }
    }

    /**
     * 判断是否是新的summary结构
     */
    @SuppressWarnings("unchecked")
    private boolean checkNewSummary(Map<String, Object> task) {
        return task.get("summary") != null && !((Map<String, Object>) task.get("summary")).isEmpty()
                || task.get("businessInfo") != null && !((Map<String, Object>) task.get("businessInfo")).isEmpty()
                || task.get("searchKey") != null && !((Map<String, Object>) task.get("searchKey")).isEmpty()
                || task.get("projectInfo") != null && !((Map<String, Object>) task.get("projectInfo")).isEmpty();
    }

    @SuppressWarnings({"unchecked", "rawtypes"})
    private void processTenantSummaryFields(Map<String, Object> task, String pageCode) throws DWBusinessException {
        if (null != task) {
            String application = (String) task.get("application");
            String code = (String) task.get("code");
            String type;
            switch (pageCode) {
                case "project-card":
                    type = "project";
                    break;
                case "task-card":
                    type = "task";
                    break;
                default:
                    type = null;
            }
            if (checkNewSummary(task)) {
                // 新结构
                newProcessTenantSummaryFields(task, pageCode);
            } else {
                // 开发平台增加了标识，不是所有字段都能作为摘要字段
                List<Map<String, Object>> summaryFields = (List<Map<String, Object>>) task.get("summaryFields");
                task.put("canSelectedSummaryFields", ObjectUtil.cloneByStream(summaryFields));
                if (type != null && null != application) {
                    TenantSummaryFields tenantSummaryFields = queryTenantSummaryFields(application, type, code);
                    if (null != tenantSummaryFields) {
                        boolean usedTenantSummaryFields;
                        boolean ifUseTenantSummaryFields = judgeIfUseTenantSummaryFields(tenantSummaryFields, summaryFields);
                        boolean ifUseTenantTitle = judgeIfUseTenantTitle(tenantSummaryFields);
                        if (ifUseTenantSummaryFields) {
                            if (!org.springframework.util.CollectionUtils.isEmpty(tenantSummaryFields.getSummaryFields())) {
                                List<Map<String, Object>> transformedTenantSummaryFields = transformTenantFieldsToSys(tenantSummaryFields, summaryFields);
                                List<Map<String, Object>> mergedSummaryFields = mergeSummaryFields(summaryFields, transformedTenantSummaryFields);
                                Map<String, Object> pages = (Map<String, Object>) task.get("pages");
                                if (null != pages.get("dataStates") && !org.springframework.util.CollectionUtils.isEmpty((List) pages.get("dataStates"))) {
                                    List<Map<String, Object>> dataStates = (List<Map<String, Object>>) pages.get("dataStates");
                                    Map<String, Object> dataState = dataStates.get(0);
                                    dataState.put("summaryFields", mergedSummaryFields);
                                } else {
                                    pages.put("summaryFields", mergedSummaryFields);
                                }
                                task.put("summaryFields", mergedSummaryFields);
                            }
                            usedTenantSummaryFields = true;
                            task.put("usedTenantSummaryFields", true);
                        } else {
                            usedTenantSummaryFields = false;
                            task.put("usedTenantSummaryFields", false);
                        }
                        // 判断是否使用租户自定义标题
                        if (ifUseTenantTitle && usedTenantSummaryFields) {
                            TenantCardTitleDefinition tenantCardTitleDefinition = new TenantCardTitleDefinition();
                            tenantCardTitleDefinition.setFields(tenantSummaryFields.getTitle());
                            tenantCardTitleDefinition.setTitlePrefix(tenantSummaryFields.getTitlePrefix());
                            task.put("tenantCardTitleDefinition", tenantCardTitleDefinition);
                        }
                        task.put("tenantSummaryFields", tenantSummaryFields);
                    } else {
                        task.put("usedTenantSummaryFields", false);
                    }
                } else {
                    task.put("usedTenantSummaryFields", false);
                }
                filterSummaryFields(task);
            }
        }
    }

    @SuppressWarnings("unchecked, rawtypes")
    private void newProcessTenantSummaryFields(Map<String, Object> task, String pageCode) throws DWBusinessException {
        if (null != task) {
            String application = (String) task.get("application");
            String code = (String) task.get("code");
            String type;
            switch (pageCode) {
                case "project-card":
                    type = "project";
                    break;
                case "task-card":
                    type = "task";
                    break;
                default:
                    type = null;
            }
            List<Map<String, Object>> newSummaryFieldes =
                    (List<Map<String, Object>>) ((Map<String, Object>) task.getOrDefault("summary", Collections.emptyMap())).getOrDefault("fieldMappings", Collections.emptyList());
            // 开发平台增加了标识，不是所有字段都能作为摘要字段
            task.put("canSelectedSummaryFields", ObjectUtil.cloneByStream(newSummaryFieldes));
            if (type != null && null != application) {
                TenantSummaryFields tenantSummaryFields = queryTenantSummaryFields(application, type, code);
                if (null != tenantSummaryFields) {
                    boolean ifUseTenantSummaryFields = judgeIfUseTenantSummaryFields(tenantSummaryFields, newSummaryFieldes);
                    boolean ifUseTenantTitle = judgeIfUseTenantTitle(tenantSummaryFields);
                    if (ifUseTenantSummaryFields) {
                        if (!org.springframework.util.CollectionUtils.isEmpty(tenantSummaryFields.getSummaryFields())) {
                            List<Map<String, Object>> mergedSummaryFields = transformTenantFieldsToSys(tenantSummaryFields, newSummaryFieldes);
                            Map<String, Object> pages = (Map<String, Object>) task.get("pages");
                            if (null != pages.get("dataStates") && !org.springframework.util.CollectionUtils.isEmpty((List) pages.get("dataStates"))) {
                                for (Map<String, Object> dataState : (List<Map<String, Object>>) pages.get("dataStates")) {
                                    if (SUMMARY.equals(dataState.get("type"))) {
                                        dataState.put("fieldMappings", mergedSummaryFields);
                                    }
                                }
                            }
                            task.put("summaryFields", mergedSummaryFields);
                        }
                        task.put("usedTenantSummaryFields", true);
                    } else {
                        task.put("usedTenantSummaryFields", false);
                    }
                    // 判断是否使用租户自定义标题
                    if (ifUseTenantTitle && Boolean.TRUE.equals(task.get("usedTenantSummaryFields"))) {
                        TenantCardTitleDefinition tenantCardTitleDefinition = new TenantCardTitleDefinition();
                        tenantCardTitleDefinition.setFields(tenantSummaryFields.getTitle());
                        tenantCardTitleDefinition.setTitlePrefix(tenantSummaryFields.getTitlePrefix());
                        task.put("tenantCardTitleDefinition", tenantCardTitleDefinition);
                    }
                    task.put("tenantSummaryFields", tenantSummaryFields);
                } else {
                    task.put("usedTenantSummaryFields", false);
                }
            } else {
                task.put("usedTenantSummaryFields", false);
            }
            newFilterSummaryFields(task);
        }
    }

    /**
     * 新的过滤摘要字段
     */
    @SuppressWarnings("unchecked")
    private void newFilterSummaryFields(Map<String, Object> task) throws DWBusinessException {
        if (task.get("summaryFields") != null && !((List<Map<String, Object>>) task.get("summaryFields")).isEmpty() && Boolean.FALSE.equals(task.get("usedTenantSummaryFields"))) {
            // 只有应用级摘要字段需要按照“默认展示”标识过滤
            // 历史数据默认为true
            ((List<Map<String, Object>>) task.get("summaryFields")).removeIf(map -> !MapUtils.getBoolean(map, "defaultEffective", true));
            Map<String, Object> pages = (Map<String, Object>) task.get("pages");
            if (null != pages.get("dataStates") && !org.springframework.util.CollectionUtils.isEmpty((List) pages.get("dataStates"))) {
                for (Map<String, Object> dataState : (List<Map<String, Object>>) pages.get("dataStates")) {
                    if (SUMMARY.equals(dataState.get("type"))) {
                        dataState.put("fieldMappings", task.get("summaryFields"));
                    }
                }
            }
        }
        if (task.get("canSelectedSummaryFields") != null && !((List<Map<String, Object>>) task.get("canSelectedSummaryFields")).isEmpty()) {
            // 过滤掉不是摘要字段的，历史数据默认为true
            ((List<Map<String, Object>>) task.get("canSelectedSummaryFields")).removeIf(map -> !MapUtils.getBoolean(map, "summary", true));
            // 如果之前设置过的租户级摘要字段不在本次可选择的字段中，为了兼容历史数据，需要将租户级字段添加到可选择的字段中
            List<Map<String, Object>> canSelectedSummaryFields = ((List<Map<String, Object>>) task.get("canSelectedSummaryFields"));
            List<Map<String, Object>> tenantSummaryFieldList = (List<Map<String, Object>>) task.get("summaryFields");
            String locale = Optional.ofNullable(ServiceUtils.getCurrentLocale()).orElse("zh_CN");
            String actionId = MapUtils.getString(canSelectedSummaryFields.get(0), "actionId");
            List<ApiDataFieldLocaleMetadataDTO> fields = null;
            if (StringUtils.isNotEmpty(actionId)) {
                try {
                    GetActionLocaleResponseDTO actionLocaleResponseDTO = dataMapKgService.getActionMetaData(actionId);
                    ApiDataFieldLocaleMetadataDTO data = actionLocaleResponseDTO.getResponse().getData();
                    fields = data.getField();
                } catch (Exception e) {
                    throw new DWBusinessException("查询元数据定义出错，actionId：{}", actionId);
                }
            }
            canSelectedSummaryFields.forEach(field -> {
                if (StringUtils.isEmpty(MapUtils.getString(field, "description"))) {
                    Object title = JSONPath.eval(field, "$.tags[0].title");
                    Object localeTitle = JSONPath.eval(field, "$.tags[0].lang.title." + locale);
                    field.put("description", localeTitle != null ? localeTitle : title);
                }
                if (StringUtils.isEmpty(MapUtils.getString(field, "data_name"))) {
                    String value = MapUtils.getString(field, "value", "");
                    field.put("data_name", value.contains(".") ? value.split("\\.")[1] : "");
                }
                if (StringUtils.isEmpty(MapUtils.getString(field, "data_type"))) {
                    field.put("data_type", "string");
                }
            });
            for (Map<String, Object> tenantSummaryField : tenantSummaryFieldList) {
                boolean b = canSelectedSummaryFields.stream().noneMatch(map -> MapUtils.getString(map, "value", "").equals(MapUtils.getString(tenantSummaryField, "value", "")));
                if (b && MapUtils.getBoolean(tenantSummaryField, "summary", true)) {
                    if (fields != null && !fields.isEmpty()) {
                        Optional<ApiDataFieldLocaleMetadataDTO> dataName = fields.stream().filter(f -> f.getData_name().equals(MapUtils.getString(tenantSummaryField, "data_name", ""))).findFirst();
                        dataName.ifPresent(apiDataFieldLocaleMetadataDTO -> tenantSummaryField.put("description", apiDataFieldLocaleMetadataDTO.getDescription()));
                    }
                    canSelectedSummaryFields.add(tenantSummaryField);
                }
            }
        }
    }

    private TenantSummaryFields queryTenantSummaryFields(String application, String type, String code) throws DWBusinessException {
        String tenantId = ServiceUtils.getTenantId();
        Bson query = Filters.and(
                Filters.eq("tenantId", tenantId),
                Filters.eq("type", type),
                Filters.eq("code", code),
                Filters.eq("application", application));
        return knowledgeGraphService.findOne(query, TenantSummaryFields.class, "summaryFields", KnowledgeGraphDb.TENANT);
    }

    private boolean judgeIfUseTenantSummaryFields(TenantSummaryFields tenantSummaryFields, List<Map<String, Object>> summaryFields) {
        if (null == tenantSummaryFields) {
            return false;
        } else if (org.springframework.util.CollectionUtils.isEmpty(summaryFields)) {
            return false;
        } else {
            return summaryFields.get(0).get("actionId").equals(tenantSummaryFields.getActionId());
        }
    }

    private boolean judgeIfUseTenantTitle(TenantSummaryFields tenantSummaryFields) {
        if (null == tenantSummaryFields) {
            return false;
        } else {
            String titlePrefix = tenantSummaryFields.getTitlePrefix();
            List<String> title = tenantSummaryFields.getTitle();
            return !(org.springframework.util.CollectionUtils.isEmpty(title) && StringUtils.isEmpty(titlePrefix));
        }
    }

    private List<Map<String, Object>> transformTenantFieldsToSys(TenantSummaryFields tenantSummaryFields, List<Map<String, Object>> summaryFields) {
        List<TenantSummaryField> tenantSummaryFieldList = tenantSummaryFields.getSummaryFields();
        if (!org.springframework.util.CollectionUtils.isEmpty(tenantSummaryFieldList)) {
            List<Map<String, Object>> sysSummaryFields = new ArrayList<>(6);
            for (int i = 0; i < tenantSummaryFieldList.size(); i++) {
                List<String> values = tenantSummaryFieldList.get(i).getValue();
                for (String fieldPath : values) {
                    // 通过字段匹配应用级摘要配置；租户级配置覆盖应用级配置
                    Optional<Map<String, Object>> first = summaryFields.stream()
                            .filter(map -> fieldPath.equals(MapUtil.getStr(map, "value"))).findFirst();
                    Map<String, Object> sysSummaryField = Maps.newHashMap();
                    if (first.isPresent()) {
                        sysSummaryField = first.get();
                    }
                    Map<String, Object> sysSummaryFieldCopy = ObjectUtil.cloneByStream(sysSummaryField);
                    sysSummaryFieldCopy.put("summary", true);
                    sysSummaryFieldCopy.put("actionId", tenantSummaryFields.getActionId());
                    sysSummaryFieldCopy.put("value", fieldPath);
                    List<Map<String, Object>> tagList = new ArrayList<>(1);
                    Map<String, Object> tag = new HashMap<>();
                    tag.put("code", "group_" + i);
                    tag.put("title", tenantSummaryFieldList.get(i).getTitle());
                    tag.put("customTitle", true);
                    if (StringUtils.isEmpty(MapUtils.getString(sysSummaryFieldCopy, "description"))) {
                        sysSummaryFieldCopy.put("description", tenantSummaryFieldList.get(i).getTitle());
                        sysSummaryFieldCopy.put("data_type", "string");
                        sysSummaryFieldCopy.put("data_name", fieldPath.contains(".") ? fieldPath.split("\\.")[1] : fieldPath);

                    }
                    if (!tag.containsKey("lang")) {
                        JSONObject lang = new JSONObject();
                        JSONObject titleLang = new JSONObject();
                        titleLang.fluentPut("zh_CN", tenantSummaryFieldList.get(i).getTitle())
                                .fluentPut("zh_TW", tenantSummaryFieldList.get(i).getTitle())
                                .fluentPut("en_US", tenantSummaryFieldList.get(i).getTitle());
                        lang.put("title", titleLang);
                        tag.put("lang", lang);
                    }
                    tagList.add(tag);
                    sysSummaryFieldCopy.put("tags", tagList);
                    sysSummaryFields.add(sysSummaryFieldCopy);
                }
            }
            return sysSummaryFields;
        }
        return summaryFields;
    }

    /**
     * 合并摘要
     * @param summaryFields 系统级摘要字段
     * @param transformedTenantSummaryFields 租户级摘要字段
     * @return 合并后的摘要字段
     */
    private List<Map<String, Object>> mergeSummaryFields(List<Map<String, Object>> summaryFields, List<Map<String, Object>> transformedTenantSummaryFields) {
        if (org.springframework.util.CollectionUtils.isEmpty(summaryFields)) {
            return transformedTenantSummaryFields;
        } else {
            List<Map<String, Object>> mergedSummaryFields = new ArrayList<>();
            mergedSummaryFields.addAll(transformedTenantSummaryFields);
            // 非摘要字段，历史资讯或者其他字段
            mergedSummaryFields.addAll(summaryFields.stream().filter(map -> !MapUtils.getBoolean(map, "summary", true)).collect(Collectors.toList()));
            return mergedSummaryFields;
        }
    }

    /**
     * 过滤摘要字段
     * @param task 任务定义
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    private void filterSummaryFields(Map<String, Object> task) throws DWBusinessException {
        List<Map<String, Object>> summaryFields = (List<Map<String, Object>>) task.get("summaryFields");
        Boolean usedTenantSummaryFields = (Boolean) task.get("usedTenantSummaryFields");
        List<Map<String, Object>> canSelectedSummaryFields = (List<Map<String, Object>>) task.get("canSelectedSummaryFields");
        if (!org.springframework.util.CollectionUtils.isEmpty(summaryFields) && !Boolean.TRUE.equals(usedTenantSummaryFields)) {
            // 只有应用级摘要字段需要按照“默认展示”标识过滤
            // 历史数据默认为true
            summaryFields.removeIf(map -> !MapUtils.getBoolean(map, "defaultEffective", true) && MapUtils.getBoolean(map, "summary", true));
            Map<String, Object> pages = (Map<String, Object>) task.get("pages");
            if (null != pages.get("dataStates") && !org.springframework.util.CollectionUtils.isEmpty((List) pages.get("dataStates"))) {
                List<Map<String, Object>> dataStates = (List<Map<String, Object>>) pages.get("dataStates");
                Map<String, Object> dataState = dataStates.get(0);
                dataState.put("summaryFields", summaryFields);
            } else {
                pages.put("summaryFields", summaryFields);
            }
        }
        if (!org.springframework.util.CollectionUtils.isEmpty(canSelectedSummaryFields)) {
            // 过滤掉不是摘要字段的，历史数据默认为true
            canSelectedSummaryFields.removeIf(map -> !MapUtils.getBoolean(map, "summary", true));
        }
        // 如果之前设置过的租户级摘要字段不在本次可选择的字段中，为了兼容历史数据，需要将租户级字段添加到可选择的字段中
        if (!org.springframework.util.CollectionUtils.isEmpty(canSelectedSummaryFields)) {
            String locale = Optional.ofNullable(ServiceUtils.getCurrentLocale()).orElse("zh_CN");
            String actionId = MapUtils.getString(canSelectedSummaryFields.get(0), "actionId");
            List<ApiDataFieldLocaleMetadataDTO> fields = null;
            if (StringUtils.isNotEmpty(actionId)) {
                try {
                    GetActionLocaleResponseDTO actionLocaleResponseDTO = dataMapKgService.getActionMetaData(actionId);
                    ApiDataFieldLocaleMetadataDTO data = actionLocaleResponseDTO.getResponse().getData();
                    fields = data.getField();
                } catch (Exception e) {
                    throw new DWBusinessException("查询元数据定义出错，actionId：{}", actionId);
                }
            }
            canSelectedSummaryFields.forEach(field -> {
                if (StringUtils.isEmpty(MapUtils.getString(field, "description"))) {
                    Object title = JSONPath.eval(field, "$.tags[0].title");
                    Object localeTitle = JSONPath.eval(field, "$.tags[0].lang.title." + locale);
                    field.put("description", localeTitle != null ? localeTitle : title);
                }
                if (StringUtils.isEmpty(MapUtils.getString(field, "data_name"))) {
                    String value = MapUtils.getString(field, "value", "");
                    field.put("data_name", value.contains(".") ? value.split("\\.")[1] : "");
                }
                if (StringUtils.isEmpty(MapUtils.getString(field, "data_type"))) {
                    field.put("data_type", "string");
                }
            });
            for (Map<String, Object> tenantSummaryField : summaryFields) {
                boolean b = canSelectedSummaryFields.stream().noneMatch(map -> MapUtils.getString(map, "value", "").equals(MapUtils.getString(tenantSummaryField, "value", "")));
                if (b && MapUtils.getBoolean(tenantSummaryField, "summary", true)) {
                    if (null != fields && !fields.isEmpty()) {
                        Optional<ApiDataFieldLocaleMetadataDTO> dataName = fields.stream().filter(f -> f.getData_name().equals(MapUtils.getString(tenantSummaryField, "data_name", ""))).findFirst();
                        dataName.ifPresent(apiDataFieldLocaleMetadataDTO -> tenantSummaryField.put("description", apiDataFieldLocaleMetadataDTO.getDescription()));
                    }
                    canSelectedSummaryFields.add(tenantSummaryField);
                }
            }
        }
    }

    /**
     * 按条件分页查询人工任务类型列表
     * @param request
     * @return
     * @throws DWException
     */
    @Override
    public Object postFindTaskFromUser(FindTaskFromUserRequest request) throws DWException {
        if (Objects.isNull(request) || StringUtils.isEmpty(request.getTenantId())
                || Objects.isNull(request.getPageNum()) || Objects.isNull(request.getPageSize())) {
            throw new DWBusinessException("请求参数为空");
        }
        String version = dataPickService.tenantVersion();
        if (StringUtils.isEmpty(version)) {
            version = "1.0";
        }
        Criteria criteria = Criteria.where("version").is(version);
        if (CollectionUtils.isNotEmpty(request.getApplicationList())) {
            List<Application> applications = dbKnowledgeGraphService.tenantAppRelations(request.getApplicationList());
            Map<String, Boolean> applicationIndividualMap = new HashMap<>();
            if (CollectionUtils.isNotEmpty(applications)) {
                // 转成map
                applicationIndividualMap = applications.stream().collect(Collectors.toMap(Application::getCode, app -> Optional.ofNullable(app.getIndividual()).orElse(false)));
            }
            List<Criteria> orCriteriaList = new ArrayList<>();
            for (String appCode: request.getApplicationList()) {
                if (Boolean.TRUE.equals(applicationIndividualMap.get(appCode))) {
                    // 应用悬停，只查个案数据
                    orCriteriaList.add(Criteria.where("application").is(appCode).and("tenantId").is(request.getTenantId()));
                } else {
                    orCriteriaList.add(Criteria.where("application").is(appCode).and("tenantId").in("SYSTEM", null));
                }
            }
            if (!orCriteriaList.isEmpty()) {
                criteria.andOperator(new Criteria().orOperator(orCriteriaList.toArray(new Criteria[0])));
            }
        }
        if (StringUtils.isNotEmpty(request.getExecuteType())) {
            criteria.and("executeType").is(request.getExecuteType());
        }
        if (StringUtils.isNotEmpty(request.getTaskDefName())) {
            // 任务类型名称多语言模糊查询
            Pattern compile = Pattern.compile("^.*" + request.getTaskDefName() + ".*$", Pattern.CASE_INSENSITIVE);
            String currentLocale = CommonUtils.getCurrentLocale();
            if (StringUtils.isEmpty(currentLocale)) {
                // 为空设置为简体中文
                currentLocale = "zh_CN";
            }
            criteria.and("lang.name." + currentLocale).regex(compile);
        }
        // 包含的业务类型
        if (CollectionUtils.isNotEmpty(request.getIncludeCategoryList())) {
            if (CollectionUtils.isNotEmpty(request.getDrivenCategory())) {
                criteria.orOperator(Criteria.where("category").in(request.getIncludeCategoryList()), Criteria.where("drivenCategory").in(request.getDrivenCategory()));
            } else {
                criteria.and("category").in(request.getIncludeCategoryList());
            }
        }
        // 不包含的业务类型
        if (CollectionUtils.isNotEmpty(request.getExcludeCategoryList())) {
            criteria.and("category").nin(request.getExcludeCategoryList());
            if (CollectionUtils.isNotEmpty(request.getDrivenCategory())) {
                criteria.and("drivenCategory").nin(request.getDrivenCategory());
            }
        }
        Query query = new Query(criteria);
        long total = dataPickService.systemTemplate().count(query, Task.class);
        if (total <= 0) {
            return new FindTaskFromUserResp(null, total);
        }
        // 分页
        Pageable pageable = PageRequest.of(request.getPageNum() - 1, request.getPageSize());
        // 查询分页数据
        List<Task> tasks = dataPickService.systemTemplate().find(query.with(pageable), Task.class);
        return new FindTaskFromUserResp(tasks, total);
    }

    @Override
    public CustBizTaskConfigDTO getCustBizTaskByProject(String appCode, String projectCode) throws DWException {
        CustBizTaskConfigDTO custBizTaskConfig = new CustBizTaskConfigDTO();
        String version = dataPickService.tenantVersion();
        if (StringUtils.isEmpty(version)) {
            version = "2.0";
        }
        // step1: 查询任务定义，获取跳过定义
        List<Task> taskList = dataPickService.systemTemplate().find(Query.query(Criteria.where("application").is(appCode)
                .and("executeType").is("manual").and("version").is(version)), Task.class);
        // step2: 查询项目定义，获取里程碑，用于过滤任务数据
        Project project = dataPickService.systemTemplate().findOne(Query.query(
                Criteria.where("code").is(projectCode).and("version").is(version).and("tenantId").is("SYSTEM")), Project.class);
        if (ObjectUtils.isEmpty(taskList)) {
            return null;
        }
        if (project == null || ObjectUtils.isEmpty(project.getMainLineTasks())) {
            return null;
        }
        // step3: 获取业务流自定义配置
        CustBizTaskConfigDTO taskConfigDTO = dataPickService.tenantTemplate().findOne(Query.query(Criteria.where("projectCode").is(projectCode)
                .and("tenantId").is(ServiceUtils.getTenantId())), CustBizTaskConfigDTO.class);
        // {任务code:是否跳过}
        Map<String, Boolean> bizTaskSkipMap = new HashMap<>();
        if (null != taskConfigDTO && !CollectionUtils.isEmpty(taskConfigDTO.getBizTasks())) {
            bizTaskSkipMap.putAll(taskConfigDTO.getBizTasks().stream().collect(Collectors.toMap(CustBizTask::getCode, CustBizTask::getCustomSkip)));
        }
        // 过滤出当前项目的任务
        List<Task> taskResults = taskList.stream().filter(task -> project.getMainLineTasks().contains(task.getCode())).collect(Collectors.toList());
        Map<String, Integer> taskSequenceMap = getTaskSequenceByProject(project, version);
        if (!ObjectUtils.isEmpty(taskResults)) {
            List<CustBizTask> custBizTasks = taskResults.stream().map(task -> {
                CustBizTask custBizTask = new CustBizTask();
                custBizTask.setCode(task.getCode());
                custBizTask.setName(task.getName());
                custBizTask.setSequence(taskSequenceMap.get(task.getCode()));
                // custBizTask.setCustomSkip(task.getCustomSkip());
                // 如果customSkip未获取到值，前端默认为false，即选中状态
                custBizTask.setCustomSkip(bizTaskSkipMap.get(task.getCode()));
                custBizTask.setCustomCanSkip(task.getCustomCanSkip());
                return custBizTask;
                // 根据任务顺序排序，没有值的任务放到最后
            }).sorted(Comparator.comparing(CustBizTask::getSequence, Comparator.nullsLast(Integer::compareTo))).collect(Collectors.toList());
            custBizTaskConfig.setBizTasks(custBizTasks);
        }

        custBizTaskConfig.setCode(taskConfigDTO == null ? null : taskConfigDTO.getCode());
        custBizTaskConfig.setCustomGroupSkip(taskConfigDTO == null ? null : taskConfigDTO.getCustomGroupSkip());
        custBizTaskConfig.setProjectCode(projectCode);
        custBizTaskConfig.setGroupSkipConfig(taskConfigDTO == null ? null : taskConfigDTO.getGroupSkipConfig());
        custBizTaskConfig.setTenantId(ServiceUtils.getTenantId());
        return custBizTaskConfig;
    }

    /**
     * 获取任务顺序
     * @param project 项目数据
     * @param version 租户版本
     * @return 返回任务排序
     */
    private Map<String, Integer> getTaskSequenceByProject(Project project, String version) {
        Map<String, Integer> taskSequenceMap = new HashMap<>();
        Integer sequence = 1;
        DataState initDataState = project.getInit();
        DataState endDataState = project.getEnd();
        String taskStateCode = initDataState.getCode();

        // 遍历任务状态，获取任务顺序 默认最多100个任务节点
        while (!endDataState.getCode().equals(taskStateCode) && sequence < 100) {
            List<Task> taskList = dataPickService.systemTemplate().find(Query.query(
                    Criteria.where("from").in(taskStateCode).and("version").is(version)), Task.class);
            // 一个节点有多个任务时，直接退出
            if (CollectionUtils.isEmpty(taskList) || taskList.size() > 1) {
                return taskSequenceMap;
            }
            Task task = taskList.get(0);
            // 一个任务有多个状态时，对此任务设置完状态后直接退出
            if (CollectionUtils.isEmpty(task.getTo()) || task.getTo().size() > 1) {
                taskSequenceMap.put(task.getCode(), sequence);
                return taskSequenceMap;
            }
            taskStateCode = task.getTo().get(0);
            taskSequenceMap.put(task.getCode(), sequence);
            sequence++;
        }
        return taskSequenceMap;
    }
}
