package com.digiwin.athena.datamap.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.kmservice.aspect.MyExceptionHandler;
import com.digiwin.athena.kmservice.constants.KnowledgeGraphDb;
import com.digiwin.athena.datamap.povo.*;
import com.digiwin.athena.datamap.service.IAuthorityService;
import com.digiwin.athena.datamap.service.IProjectService;
import com.digiwin.athena.datamap.spi.DatamapGMCService;
import com.digiwin.athena.datamap.spi.IKnowledgeGraphService;
import com.digiwin.athena.domain.core.Project;
import com.digiwin.athena.kmservice.locale.Lang;
import com.mongodb.client.model.Filters;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.security.SecureRandom;
import java.util.*;
import java.util.stream.Collectors;

@Lang
@MyExceptionHandler
@Service
public class DatamapAuthorityService implements IAuthorityService {

    private static final String START_PROJECT_CODE = "startProject";
    private static final String START_PROJECT_NAME = "发起项目";
    private static final Logger log = LoggerFactory.getLogger(DatamapAuthorityService.class);

    @Autowired
    private IKnowledgeGraphService knowledgeGraphService;

    @Autowired
    private IProjectService projectService;

    @Autowired
    private DatamapGMCService datamapGmcService;

    @Override
    public Object postStartProjectAuthorityByAppCode(String appCode) throws Exception {
        List<Project> taskList = knowledgeGraphService.find(
                Filters.eq("authorityPrefix", appCode + ":" + START_PROJECT_CODE),
                Project.class, "taskConfigs", KnowledgeGraphDb.SYSTEM);
        List<Project> projects = new ArrayList<>(taskList);
        try {
            List<Project> datamapTasks = projectService.getProjectByAuthorityPrefix(appCode + ":" + START_PROJECT_CODE);
            if (!CollectionUtils.isEmpty(datamapTasks)) {
                projects.addAll(taskList);
            }
        } catch (Exception e) {
            log.error("getProjectByAuthorityPrefix error", e);
        }
        if (CollectionUtils.isEmpty(projects)) {
            throw new DWBusinessException("没找到需要生成权限的发起项目信息");
        }
        AuthorityConfigVoFromKg configVO = this.generateToStartProjectAuthorityConfig(appCode, null, projects);
        return this.postGenerateAuthority(configVO);
    }

    private AuthorityConfigVoFromKg generateToStartProjectAuthorityConfig(String appCode, String appName, List<Project> taskList) {
        AuthorityConfigVoFromKg configVO = new AuthorityConfigVoFromKg();

        AppVO app = new AppVO();
        app.setId(appCode);
        app.setName(StringUtils.isEmpty(appName) ? appCode : appName);
        configVO.setApp(app);

        configVO.setType("UPDATE");

        List<ModuleVO> moduleVOList = new ArrayList<>();
        ModuleVO moduleVO = new ModuleVO();
        moduleVO.setId(START_PROJECT_CODE);
        moduleVO.setName(START_PROJECT_NAME);
        moduleVOList.add(moduleVO);
        configVO.setModules(moduleVOList);

        List<ActionVO> actionVOList = new ArrayList<>();
        for (Project task : taskList) {
            ActionVO actionVO = new ActionVO();
            actionVO.setId(task.getCode());
            actionVO.setModuleId(START_PROJECT_CODE);
            actionVO.setName(task.getName());

            List<ConditionVO> conditionVOList = new ArrayList<>();

            ConditionVO conditionVO = new ConditionVO();
            conditionVO.setKey("enable");
            conditionVO.setName("项目启用");
            conditionVO.setTypeParameter(this.getTypeParameter());
            conditionVO.setDefaultValue(this.getTypeParameter().get(0));

            conditionVOList.add(conditionVO);
            actionVO.setCondition(conditionVOList);

            actionVOList.add(actionVO);
        }
        configVO.setActions(actionVOList);
        return configVO;
    }

    private List<String> getTypeParameter() {
        List<String> result = new ArrayList<>();
        result.add("allow");
        result.add("forbidden");
        return result;
    }

    private Object postGenerateAuthority(AuthorityConfigVoFromKg config) throws Exception {
        log.info("GenerateAuthority config:{}", JSON.toJSONString(config));
        this.check(config);

        Object cloudGood = this.datamapGmcService.getCloudGood(config.getApp().getId());
        if (cloudGood == null) {
            throw new DWBusinessException("没有查到商品信息:" + config.getApp().getId());
        }
        JSONObject goodObj = JSON.parseObject(JSON.toJSONString(cloudGood));
        log.info("getCloudGood:{}", goodObj);
        JSONObject modules = goodObj.getJSONObject("modules");
        if (modules == null || (StringUtils.isEmpty(modules.getString("appToken")) && StringUtils.isEmpty(config.getAppToken()))) {
            throw new DWBusinessException("appToken缺失");
        }

        this.upsertAuthorityConfig(goodObj, config);
        log.info("toUpdateGoodsInfo:{}", goodObj);

        Object updateCloudGoodModulesRequestObj = this.getUpdateCloudGoodModulesRequestObj(goodObj);
        Object updateCloudGoodModulesResultObj = this.datamapGmcService.updateCloudGoodModules(updateCloudGoodModulesRequestObj);
        JSONObject updateCloudGoodModulesResult = JSON.parseObject(JSON.toJSONString(updateCloudGoodModulesResultObj));
        if (!updateCloudGoodModulesResult.getBoolean("success")) {
            throw new DWBusinessException(String.format("商品(%s)更新模组信息失败", config.getApp().getId()));
        }
        return "success";
    }

    private void check(AuthorityConfigVoFromKg config) throws Exception {
        if (config == null) {
            throw new DWBusinessException("权限配置信息缺失");
        }

        AppVO app = config.getApp();
        if (app == null || StringUtils.isEmpty(app.getId()) || StringUtils.isEmpty(app.getName())) {
            throw new DWBusinessException("appId或者name信息缺失");
        }

        if (!"ADD".equals(config.getType()) && !"UPDATE".equals(config.getType())
                && !"DELETE_MODULE".equals(config.getType()) && !"DELETE_ACTION".equals(config.getType()) && !"DELETE_CONDITION".equals(config.getType())) {
            throw new DWBusinessException("type类型错");
        }

        if ("ADD".equals(config.getType()) || "UPDATE".equals(config.getType())) {
            List<ModuleVO> modules = config.getModules();
            if (CollectionUtils.isEmpty(modules)) {
                throw new DWBusinessException("modules为");
            }
            for (ModuleVO e : modules) {
                if (StringUtils.isEmpty(e.getId()) || StringUtils.isEmpty(e.getName())) {
                    throw new DWBusinessException("module信息缺失");
                }
            }

            List<ActionVO> actions = config.getActions();
            if (CollectionUtils.isEmpty(actions)) {
                throw new DWBusinessException("actions为空");
            }
            for (ActionVO actionVO : actions) {
                if (StringUtils.isEmpty(actionVO.getId()) || StringUtils.isEmpty(actionVO.getName())) {
                    throw new DWBusinessException("action信息缺失");
                }
                if (StringUtils.isEmpty(actionVO.getModuleId())) {
                    throw new DWBusinessException("action中moduleId信息为空");
                }
                if (CollectionUtils.isEmpty(actionVO.getCondition())) {
                    throw new DWBusinessException("action中Condition信息为空");
                }
                for (ConditionVO conditionVO : actionVO.getCondition()) {
                    // if(StringUtils.isEmpty(conditionVO.getActionId())){
                    // throw new DWBusinessException("condition中actionId信息为空");
                    // }
                    if (StringUtils.isEmpty(conditionVO.getKey())) {
                        throw new DWBusinessException("condition中key信息为空");
                    }
                    if (StringUtils.isEmpty(conditionVO.getName())) {
                        throw new DWBusinessException("condition中name信息为空");
                    }
                    if (StringUtils.isEmpty(conditionVO.getDefaultValue())) {
                        throw new DWBusinessException("condition中defaultValue信息为空");
                    }
                    if (CollectionUtils.isEmpty(conditionVO.getTypeParameter())) {
                        throw new DWBusinessException("condition中typeParameter信息为空");
                    }
                    if (!conditionVO.getTypeParameter().contains(conditionVO.getDefaultValue())) {
                        throw new DWBusinessException("condition中defaultValue不在typeParameter集合中");
                    }
                }

            }
        }

    }

    private void upsertAuthorityConfig(JSONObject sourceObj, AuthorityConfigVoFromKg config) {
        this.makeUpConfig(config);
        JSONObject modules = sourceObj.getJSONObject("modules");
        if (modules == null) {
            JSONObject configObj = JSON.parseObject(JSON.toJSONString(config));
            configObj.remove("type");
            sourceObj.put("modules", configObj);
        } else {
            if ("ADD".equals(config.getType())) {
                modules.put("modules", JSONArray.parseArray(JSON.toJSONString(config.getModules())));
                modules.put("actions", JSONArray.parseArray(JSON.toJSONString(config.getActions())));
            } else if ("UPDATE".equals(config.getType())) {
                JSONArray modulesArray = modules.getJSONArray("modules");
                Map<String, List<ModuleVO>> collect = config.getModules().stream().collect(Collectors.groupingBy(e -> e.getId()));
                List<ModuleVO> moduleToAddList = new ArrayList<>();
                for (int i = 0; i < modulesArray.size(); i++) {
                    JSONObject jsonObject = modulesArray.getJSONObject(i);
                    String moduleId = jsonObject.getString("id");
                    if (!CollectionUtils.isEmpty(collect.get(moduleId))) {
                        jsonObject.put("name", collect.get(moduleId).get(0).getName());
                    }
                    collect.remove(moduleId);
                }
                for (Map.Entry<String, List<ModuleVO>> entry : collect.entrySet()) {
                    moduleToAddList.addAll(entry.getValue());
                }
                if (!CollectionUtils.isEmpty(moduleToAddList)) {
                    modulesArray.addAll(moduleToAddList);
                }
                // 处理action
                JSONArray actionsArray = modules.getJSONArray("actions");
                Map<String, List<ActionVO>> collect1 = config.getActions().stream().collect(Collectors.groupingBy(e -> e.getId() + "###" + e.getModuleId()));
                List<ActionVO> actionsToAddList = new ArrayList<>();
                for (int i = 0; i < actionsArray.size(); i++) {
                    JSONObject jsonObject = actionsArray.getJSONObject(i);
                    String actionBelongModuleId = jsonObject.getString("moduleId");
                    String actionId = jsonObject.getString("id");
                    List<ActionVO> removeAction = collect1.remove(actionId + "###" + actionBelongModuleId);
                    if (!CollectionUtils.isEmpty(removeAction)) {
                        ActionVO actionVo = removeAction.get(0);
                        jsonObject.put("name", actionVo.getName());
                        JSONArray conditionArray = jsonObject.getJSONArray("condition");
                        List<ConditionVO> conditionsToAddList = new ArrayList<>();
                        Map<String, List<ConditionVO>> collect2 = removeAction.get(0).getCondition().stream().collect(Collectors.groupingBy(e -> e.getKey()));
                        for (int j = 0; j < conditionArray.size(); j++) {
                            JSONObject conditionObj = conditionArray.getJSONObject(j);
                            String conditionKey = conditionObj.getString("key");
                            List<ConditionVO> removeCondition = collect2.remove(conditionKey);
                            if (!CollectionUtils.isEmpty(removeCondition)) {
                                ConditionVO conditionVO = removeCondition.get(0);
                                conditionObj.put("name", conditionVO.getName());
                                conditionObj.put("type", conditionVO.getType());
                                conditionObj.put("typeParameter", conditionVO.getTypeParameter());
                                conditionObj.put("parameter", conditionVO.getParameter());
                                conditionObj.put("defaultValue", conditionVO.getDefaultValue());
                            }
                        }
                        for (Map.Entry<String, List<ConditionVO>> entry : collect2.entrySet()) {
                            conditionsToAddList.addAll(entry.getValue());
                        }
                        if (!CollectionUtils.isEmpty(conditionsToAddList)) {
                            conditionArray.addAll(conditionsToAddList);
                        }
                    }
                }
                for (Map.Entry<String, List<ActionVO>> entry : collect1.entrySet()) {
                    actionsToAddList.addAll(entry.getValue());
                }
                if (!CollectionUtils.isEmpty(actionsToAddList)) {
                    actionsArray.addAll(actionsToAddList);
                }
            } else if ("DELETE_MODULE".equals(config.getType())) { // 删除整个模组级模组下的信息，包含action和action下的condition
                this.deleteModule(modules, config);
            } else if ("DELETE_ACTION".equals(config.getType())) { // 删除指定Aciton的信息 包含action下的condition
                this.deleteAction(modules, config);
            } else if ("DELETE_CONDITION".equals(config.getType())) { // 删除对应condition的信息
                this.deleteCondition(modules, config);
            }

        }

    }

    private void makeUpConfig(AuthorityConfigVoFromKg config) {
        // 补充app中信息
        String id = config.getApp().getId();
        config.getApp().setCategoryId(id);
        config.getApp().setIdFirst(id);

        // 补充condition中信息
        for (ActionVO actionVO : config.getActions()) {
            String name = actionVO.getName();
            String actionVOId = actionVO.getId();
            for (ConditionVO conditionVO : actionVO.getCondition()) {
                conditionVO.setSid(this.getSid());
                conditionVO.setType("single");
                conditionVO.setParameter(JSON.toJSONString(conditionVO.getTypeParameter()));
                conditionVO.setActionName(name);
                conditionVO.setActionId(actionVOId);
            }
        }
    }

    private static String getSid() {
        StringBuilder sb = new StringBuilder();
        // 产生16位的强随机数
        Random rd = new SecureRandom();
        for (int i = 0; i < 16; i++) {
            sb.append(rd.nextInt(10));
        }
        return sb.toString();
    }

    private void deleteModule(JSONObject modules, AuthorityConfigVoFromKg config) {
        List<ModuleVO> configModules = config.getModules();
        if (CollectionUtils.isEmpty(configModules)) {
            return;
        }
        Set<String> configModuleIdSet = configModules.stream().filter(e -> StringUtils.isNotEmpty(e.getId())).map(ModuleVO::getId).collect(Collectors.toSet());
        if (CollectionUtils.isEmpty(configModuleIdSet)) {
            return;
        }

        JSONArray modulesArray = modules.getJSONArray("modules");
        if (!ObjectUtils.isEmpty(modulesArray)) {
            Iterator<Object> iterator = modulesArray.iterator();
            while (iterator.hasNext()) {
                Object next = iterator.next();
                JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(next));
                if (jsonObject != null && configModuleIdSet.contains(jsonObject.getString("id"))) {
                    iterator.remove();
                }
            }
        }

        JSONArray actionsArray = modules.getJSONArray("actions");
        if (!ObjectUtils.isEmpty(actionsArray)) {
            Iterator<Object> iterator = actionsArray.iterator();
            while (iterator.hasNext()) {
                Object next = iterator.next();
                JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(next));
                if (jsonObject != null && configModuleIdSet.contains(jsonObject.getString("moduleId"))) {
                    iterator.remove();
                }
            }
        }

    }

    private void deleteAction(JSONObject modules, AuthorityConfigVoFromKg config) {
        List<ActionVO> configActions = config.getActions();
        if (CollectionUtils.isEmpty(configActions)) {
            return;
        }
        Set<String> configModuleAndActionIdSet = configActions.stream().filter(e -> StringUtils.isNotEmpty(e.getId()) && StringUtils.isNotEmpty(e.getModuleId()))
                .map(e -> e.getModuleId() + "###" + e.getId()).collect(Collectors.toSet());
        if (CollectionUtils.isEmpty(configModuleAndActionIdSet)) {
            return;
        }

        JSONArray actionsArray = modules.getJSONArray("actions");
        if (!ObjectUtils.isEmpty(actionsArray)) {
            Iterator<Object> iterator = actionsArray.iterator();
            while (iterator.hasNext()) {
                Object next = iterator.next();
                JSONObject jsonObject = JSON.parseObject(JSON.toJSONString(next));
                if (jsonObject != null && configModuleAndActionIdSet.contains(jsonObject.getString("moduleId") + "###" + jsonObject.getString("id"))) {
                    iterator.remove();
                }
            }
        }
    }

    private void deleteCondition(JSONObject modules, AuthorityConfigVoFromKg config) {
        List<ActionVO> configActions = config.getActions();
        if (CollectionUtils.isEmpty(configActions)) {
            return;
        }
        Map<String, List<ActionVO>> collect1 = configActions.stream().collect(Collectors.groupingBy(e -> e.getModuleId() + "###" + e.getId()));
        if (CollectionUtils.isEmpty(collect1)) {
            return;
        }

        JSONArray actionsArray = modules.getJSONArray("actions");
        if (!ObjectUtils.isEmpty(actionsArray)) {
            for (int i = 0; i < actionsArray.size(); i++) {
                JSONObject jsonObject = actionsArray.getJSONObject(i);
                String key = jsonObject.getString("moduleId") + "###" + jsonObject.getString("id");
                List<ActionVO> actionVOList = collect1.get(key);
                if (CollectionUtils.isEmpty(actionVOList)) {
                    continue;
                }
                Set<String> collect = actionVOList.get(0).getCondition().stream()
                        .filter(e -> StringUtils.isNotEmpty(e.getKey())).map(ConditionVO::getKey).collect(Collectors.toSet());
                JSONArray conditionArray = jsonObject.getJSONArray("condition");
                Iterator<Object> iterator = conditionArray.iterator();
                while (iterator.hasNext()) {
                    Object next = iterator.next();
                    JSONObject jsonObject1 = JSON.parseObject(JSON.toJSONString(next));
                    if (jsonObject1 != null && collect.contains(jsonObject1.getString("key"))) {
                        iterator.remove();
                    }
                }
            }
        }
    }

    private Object getUpdateCloudGoodModulesRequestObj(JSONObject goodObj) {
        Map<String, Object> result = new HashMap<>();
        result.put("code", goodObj.getString("code"));

        List<Object> multiLanguageResourceList = new ArrayList<>();
        multiLanguageResourceList.add(this.generateMultiLanguageRequest(goodObj));
        multiLanguageResourceList.add(this.generateMultiLanguageRequestTW(goodObj));
        result.put("multiLanguageResource", multiLanguageResourceList);

        result.put("modules", goodObj.getJSONObject("modules"));

        return result;
    }

    private Object generateMultiLanguageRequest(JSONObject goodObj) {
        Map<String, Object> result = new HashMap<>();
        result.put("dataSid", goodObj.getString("id"));
        result.put("language", "zh-CN");
        List<Map<String, String>> dataList = new ArrayList<>();
        Map<String, String> goodsNameMap = new HashMap<>();
        goodsNameMap.put("fieldName", "goodsName");
        // TODO: ChineseUtils.toSimplified
        // goodsNameMap.put("content", ChineseUtils.toSimplified(goodObj.getString("displayName")));

        Map<String, String> modulesMap = new HashMap<>();
        modulesMap.put("fieldName", "modules");
        // TODO: ChineseUtils.toSimplified
        // modulesMap.put("content", ChineseUtils.toSimplified(JSON.toJSONString(goodObj.getJSONObject("modules"))));

        Map<String, String> customunitMap = new HashMap<>();
        customunitMap.put("fieldName", "customunit");
        // TODO: ChineseUtils.toSimplified
        // customunitMap.put("content", ChineseUtils.toSimplified(goodObj.getString("customUnit")));

        dataList.add(goodsNameMap);
        dataList.add(modulesMap);
        dataList.add(customunitMap);

        result.put("data", dataList);

        return result;
    }

    private Object generateMultiLanguageRequestTW(JSONObject goodObj) {
        Map<String, Object> result = new HashMap<>();
        result.put("dataSid", goodObj.getString("id"));
        result.put("language", "zh-TW");
        List<Map<String, String>> dataList = new ArrayList<>();
        Map<String, String> goodsNameMap = new HashMap<>();
        goodsNameMap.put("fieldName", "goodsName");
        // TODO: ChineseUtils.toTraditional
        // goodsNameMap.put("content", translateUtils.translateContentToTw(goodObj.getString("displayName")));

        Map<String, String> modulesMap = new HashMap<>();
        modulesMap.put("fieldName", "modules");
        // TODO: ChineseUtils.toTraditional
        // modulesMap.put("content", translateUtils.translateContentToTw(goodObj.getJSONObject("modules")));

        Map<String, String> customunitMap = new HashMap<>();
        customunitMap.put("fieldName", "customunit");
        // TODO: ChineseUtils.toTraditional
        // customunitMap.put("content", translateUtils.translateContentToTw(goodObj.getString("customUnit")));

        dataList.add(goodsNameMap);
        dataList.add(modulesMap);
        dataList.add(customunitMap);

        result.put("data", dataList);

        return result;
    }

}
