package com.digiwin.athena.datamap.service.impl;

import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.app.container.exceptions.DWException;
import com.digiwin.athena.kmservice.aspect.MyExceptionHandler;
import com.digiwin.athena.datamap.povo.DataRequest;
import com.digiwin.athena.datamap.service.IProjectService;
import com.digiwin.athena.datamap.service.inner.DataMapPickService;
import com.digiwin.athena.domain.common.Constants;
import com.digiwin.athena.domain.core.MonitorRuleAdapter;
import com.digiwin.athena.domain.core.Project;
import com.digiwin.athena.domain.core.Task;
import com.digiwin.athena.domain.core.dtdflow.CustBizTask;
import com.digiwin.athena.domain.core.view.PageView;
import com.digiwin.athena.domain.customBizTask.CustBizTaskConfigDTO;
import com.digiwin.athena.domain.customBizTask.CustConfigGroupSkip;
import com.digiwin.athena.domain.customBizTask.CustGroupSkipCondition;
import com.digiwin.athena.domain.customBizTask.CustomSkipExpression;
import com.digiwin.athena.dto.BasicQuery;
import com.digiwin.athena.kmservice.locale.Lang;
import com.digiwin.athena.kmservice.utils.ServiceUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;

import java.util.*;

/**
 * @program: athena_backend
 * @description: 描述
 * @author: Tuo
 * @create: 2021-04-02 10:37
 **/
@Lang
@MyExceptionHandler
@Service
@Slf4j
public class ProjectService implements IProjectService {

    @Autowired
    @Qualifier("dataMapSystem")
    MongoTemplate mongoTemplate;

    @Autowired
    DataMapPickService dataPickService;

    public Project getUserProject(String projectCode, String stateCode) throws DWException {
        Project up = null;
        Map<String, Object> param = new HashMap<>();
        param.put("primaryProjectCode", projectCode);
        param.put("init.code", stateCode);
        param.put("executeType", "user");
        up = dataPickService.findOne(BasicQuery.of(param, null), Project.class);
        return up;
    }

    public Task manualProjectPageTask(String projectCode) throws DWBusinessException {
        Map<String, Object> query2 = new HashMap<>();
        query2.put("project", projectCode);
        query2.put("pageType", 1);
        PageView pageView = dataPickService.findBetter(BasicQuery.of(query2, null), PageView.class);
        if (null != pageView) {
            Map<String, Object> query3 = new HashMap<>();
            query3.put("pageCode", pageView.getCode());
            List<Task> tasks3 = dataPickService.find(BasicQuery.of(query2, null), Task.class);
            if (CollectionUtils.isNotEmpty(tasks3)) {
                Task taskMaster3 = tasks3.get(0);
                return taskMaster3;
            }
        }
        return null;
    }

    @Override
    public List<Project> postProjects(BasicQuery query) throws DWException {
        List<Project> list = dataPickService.find(query, Project.class);
        return list;
    }

    @Override
    public Project postProject(BasicQuery request) throws DWException {
        Project project = dataPickService.findBetter(request, Project.class);
        return project;
    }

    @Override
    public Project getProjectByCode(String code) throws DWException {
        Project p = dataPickService.findBetter(code, Project.class);
        return p;
    }

    @Deprecated
    @Override
    public Project postFixProjectByCode(String code) throws DWException {
        log.info("fix Project:{}", code);
        String version = dataPickService.tenantVersion();
        String tenantId = ServiceUtils.getTenantId();
        Project ts = dataPickService.tenantTemplate().findOne(Query.query(Criteria.where("code").is(code).and("tenantId").is(tenantId).and("version").lte(version)), Project.class);
        ts.setDueDateTimeDistance(null);
        ts.setDateTimeDistance(null);
        dataPickService.tenantTemplate().save(ts);
        return ts;
    }

    @Override
    public Project postProjectByCode(DataRequest request) throws DWException {
        Map<String, Object> param = new HashMap<>();
        param.put("code", request.getCode());
        Project p = dataPickService.findBetter(BasicQuery.of(param, request.getEocInfo()), Project.class);
        return p;
    }

    @Override
    public Project getProjectByCode2(String code, String tenantId) throws DWException {
        com.digiwin.athena.kmservice.utils.ServiceUtils.getContext().getProfiles().put("tenantId", tenantId);
        Project p = dataPickService.findBetter(code, Project.class);
        return p;
    }

    @Override
    public MonitorRuleAdapter getMonitorConfig(String id) throws DWException {
        String tenantId = com.digiwin.athena.kmservice.utils.ServiceUtils.getTenantId();
        MonitorRuleAdapter ruleAdapter = mongoTemplate.findOne(Query.query(Criteria.where("dataEventId").is(id)), MonitorRuleAdapter.class);
        ruleAdapter.setTenantId(tenantId);
        return ruleAdapter;
    }

    @Override
    public List<Project> getProjectByAuthorityPrefix(String authorityPrefix) throws DWException {
        List<Project> projects = mongoTemplate.find(Query.query(Criteria.where("authorityPrefix").is(authorityPrefix)
                .and("tenantId").is(Constants.defaultTenantId)), Project.class);
        return projects;
    }

    @Override
    public List<Project> getCustBizProjectByApp(String appCode) throws DWException {
        String version = dataPickService.tenantVersion();
        if (StringUtils.isEmpty(version)) {
            version = "2.0";
        }
        Query query = Query.query(Criteria.where("application").is(appCode)
                .and("taskCanCustom").is(Boolean.TRUE).and("version").is(version));
        List<Project> projectList = dataPickService.systemTemplate().find(query, Project.class);
        return projectList;
    }

    @Override
    public List<CustGroupSkipCondition> getCustBizField(String appCode, String projectCode) throws DWException {
        String version = dataPickService.tenantVersion();
        if (StringUtils.isEmpty(version)) {
            version = "2.0";
        }
        Query query = Query.query(Criteria.where("code").is(projectCode).and("version").is(version));
        Project project = dataPickService.systemTemplate().findOne(query, Project.class);
        if (null == project || null == project.getCustomSkipConfig()) {
            return null;
        }
        return project.getCustomSkipConfig().getCustomSKipConditions();
    }

    @Override
    public CustBizTaskConfigDTO saveCustBizConfig(CustBizTaskConfigDTO configDTO) throws DWException {
        if (StringUtils.isEmpty(configDTO.getCode())) {
            // 新增数据，生成code
            configDTO.setCode(UUID.randomUUID().toString().replace("-", ""));
        } else {
            // 修改数据，根据code删除旧数据，重新插入
            Query query = Query.query(Criteria.where("code").is(configDTO.getCode()));
            dataPickService.tenantTemplate().remove(query, "customBizTaskConfig");
        }
        // 将业务流自定义租户数据保存到配置表
        dataPickService.tenantTemplate().save(configDTO, "customBizTaskConfig");
        List<CustBizTask> bizTasks = configDTO.getBizTasks();
        // 遍历业务流task，设置自定义配置
        bizTasks.stream().forEach(v -> {
            try {
                setTaskCustomConfig(v, configDTO);
            } catch (DWException e) {
                throw new RuntimeException(e);
            }
        });
        return configDTO;
    }

    /**
     * 将自定义配置设置到业务流task定义表中
     *
     * @param custBizTask      业务流task
     * @param configDTO 特定场景配置
     */
    private void setTaskCustomConfig(CustBizTask custBizTask, CustBizTaskConfigDTO configDTO) throws DWException {
        String version = dataPickService.tenantVersion();
        if (StringUtils.isEmpty(version)) {
            version = "2.0";
        }
        Task task = dataPickService.systemTemplate().findOne(
                Query.query(Criteria.where("code").is(custBizTask.getCode())
                        .and("version").is(version)),
                Task.class);
        // 没有系统级数据直接退出
        if (null == task) {
            return;
        }
        task.setProjectCode(configDTO.getProjectCode());
        // 获取租户级任务定义数据,pluginId=null是排除机制租户数据，eocInfo=null是排除运营单元级数据
        Task tanantTask = dataPickService.tenantTemplate().findOne(
                Query.query(Criteria.where("code").is(custBizTask.getCode())
                        .and("tenantId").is(ServiceUtils.getTenantId())
                        .and("pluginId").is(null)
                        .and("eocInfo").is(null)
                // .and("projectCode").is(projectCode)
                ), Task.class);
        if (null == tanantTask) {
            // 租户级数据不存在，则复制系统级数据
            BeanUtils.copyProperties(custBizTask, task, Task.class);
        } else {
            // CustomCanSkip必须以系统级数据为主
            tanantTask.setCustomCanSkip(task.getCustomCanSkip());
            // 存在租户级数据，则复制租户级数据
            BeanUtils.copyProperties(tanantTask, task, Task.class);
            task.setCustomSkip(custBizTask.getCustomSkip());
        }
        List<CustConfigGroupSkip> groupSkipConfigs = configDTO.getGroupSkipConfig();
        // task数据已经为租户级数据，设置特定场景设置任务
        if (!CollectionUtils.isEmpty(groupSkipConfigs)) {
            List<CustGroupSkipCondition> groupSkipConditions = new ArrayList<>();
            groupSkipConfigs.stream().forEach(v -> {
                CustGroupSkipCondition groupSkipCondition = new CustGroupSkipCondition();
                Optional<Task> groupCustTask = v.getBizTasks().stream().filter(t -> t.getCode().equals(custBizTask.getCode())).findFirst();
                // 存在特定场景设置任务
                if (groupCustTask.isPresent()) {
                    groupSkipCondition.setCustomSkip(groupCustTask.get().getCustomSkip());
                    List<CustomSkipExpression> customSkipExpressions = new ArrayList<>();
                    v.getCustomSkipConditions().stream().forEach(c -> {
                        CustomSkipExpression customSkipExpression = new CustomSkipExpression();
                        BeanUtils.copyProperties(c, customSkipExpression, CustomSkipExpression.class);
                        customSkipExpression.setVar(c.getKey());
                        customSkipExpressions.add(customSkipExpression);
                    });
                    groupSkipCondition.setExpressions(customSkipExpressions);
                    groupSkipConditions.add(groupSkipCondition);
                }
            });
            if (!CollectionUtils.isEmpty(groupSkipConditions)) {
                task.setCustomSkipConditions(groupSkipConditions);
            }
        }
        if (tanantTask != null) {
            dataPickService.tenantTemplate().remove(
                    Query.query(Criteria.where("_id").is(tanantTask.getId())
                            .and("tenantId").is(ServiceUtils.getTenantId())
                    // .and("projectCode").is(projectCode)
                    ), Task.class);
        }

        task.setTenantId(ServiceUtils.getTenantId());
        task.setCustomGroupSkip(configDTO.getCustomGroupSkip());
        dataPickService.tenantTemplate().save(task, "task");
    }
}
