package com.digiwin.athena.datamap.service.inner;

import com.digiwin.app.container.exceptions.DWBusinessException;
import com.digiwin.athena.domain.common.Constants;
import com.digiwin.athena.domain.common.TenantObject;
import com.digiwin.athena.domain.component.bo.TenantAppComponentListBO;
import com.digiwin.athena.domain.core.app.Application;
import com.digiwin.athena.domain.core.app.ApplicationRelation;
import com.digiwin.athena.domain.core.app.TenantAppRelation;
import com.digiwin.athena.domain.common.HierarchicalEntity;
import com.digiwin.athena.domain.neo4j.TenantEntity;
import com.digiwin.athena.kmservice.service.DataPickService;
import com.digiwin.athena.kmservice.utils.MergeUtil;
import com.digiwin.athena.kmservice.utils.ServiceUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;


@Service("kgDataPickService")
public class DataMapKgDataPickService extends DataPickService {

    public static List<Class<?>> tenantClasses = Collections.singletonList(TenantAppComponentListBO.class);

    public static List<Class<?>> sysClasses = Arrays.asList(Application.class, ApplicationRelation.class, TenantEntity.class, TenantAppRelation.class);


    @Autowired
    @Qualifier("kgSystem")
    private MongoTemplate mongoTemplate;

    @Autowired
    @Qualifier("kgTenant")
    private MongoTemplate mongoTemplateTenant;

    @Autowired
    @Lazy
    KmService kmService;


    /**
     * @param code 唯一code
     * @param c    类
     * @param col  集合
     * @return 返回找到的第一个匹配的实体对象，如果没有匹配的对象则返回null
     * @throws DWBusinessException 业务异常
     */
    public <T extends TenantObject> T findByCode(String code, Class<T> c, String col) throws DWBusinessException {
        Map<String, Object> q = new HashMap<>();
        q.put("code", code);
        List<T> ts = find(q, c, col);
        if (!ts.isEmpty()) {
            return ts.get(0);
        }
        return null;
    }

    /**
     * 查询列表
     *
     * @param query 查询条件
     * @param c     类
     * @param col   集合
     * @param <T>   泛型
     * @return 返回查询到的实体对象列表
     * @throws DWBusinessException 业务异常
     */
    public <T extends TenantObject> List<T> find(Map<String, Object> query, Class<T> c, String col) throws DWBusinessException {
        List<T> ts = mongoTemplate.find(buildQuery(query), c, col);
        ts = excludeWithSameCode(ts);
        return ts;
    }

    /**
     * 查询单个
     *
     * @param criteria 查询条件
     * @param c        类
     * @param col      集合
     * @param <T>      泛型
     * @return 返回查询到的实体对象列表
     * @throws DWBusinessException 业务异常
     */
    public <T extends TenantObject> T findOneByCondition(Criteria criteria, Class<T> c, String col) throws DWBusinessException {
        List<T> ts = find(criteria, c, col);
        if (!ts.isEmpty()) {
            return ts.get(0);
        }
        return null;
    }

    /**
     * 查询单个，指定租户id
     *
     * @param criteria 查询条件
     * @param tenantId 租户id
     * @param c        类
     * @param col      集合
     * @param <T>      泛型
     * @return 返回查询到的实体对象列表
     * @throws DWBusinessException 业务异常
     */
    public <T extends TenantObject> T findOneByCondition(Criteria criteria, String tenantId, Class<T> c, String col) throws DWBusinessException {
        List<T> ts = find(criteria, tenantId, c, col);
        if (!ts.isEmpty()) {
            return ts.get(0);
        }
        return null;
    }

    /**
     * 查询列表
     *
     * @param criteria 查询条件
     * @param c        类
     * @param col      集合
     * @param <T>      泛型
     * @return 返回查询到的实体对象列表
     * @throws DWBusinessException 业务异常
     */
    public <T extends TenantObject> List<T> find(Criteria criteria, Class<T> c, String col) throws DWBusinessException {
        List<T> ts = mongoTemplate.find(buildQuery(criteria, null), c, col);
        ts = excludeWithSameCode(ts);
        return ts;
    }

    /**
     * 查询列表， 指定租户id
     *
     * @param criteria 查询条件
     * @param c        类
     * @param col      集合
     * @param <T>      泛型
     * @return 返回查询到的实体对象列表
     * @throws DWBusinessException 业务异常
     */
    public <T extends TenantObject> List<T> find(Criteria criteria, String tenantId, Class<T> c, String col) throws DWBusinessException {
        List<T> ts = mongoTemplate.find(buildQuery(criteria, tenantId), c, col);
        ts = excludeWithSameCode(ts);
        return ts;
    }

    /**
     * 删除重复的实体对象
     * 只适用于使用唯一键查询的的数据列表（因为这里没有判断应用是否悬停），根据sourceLevel排序，取最小的
     *
     * @param ts  实体对象列表
     * @param <T> 泛型
     * @return 返回删除重复的实体对象列表
     */
    public <T extends TenantObject> List<T> excludeWithSameCode(List<T> ts) {
        if (CollectionUtils.isEmpty(ts)) {
            return ts;
        }
        Map<String, List<T>> maps = new HashMap<>();
        for (T t : ts) {
            String code = t.entityBizCode();
            List<T> ts1 = maps.computeIfAbsent(code, k -> new ArrayList<>());
            ts1.add(t);
        }
        List<T> result = new ArrayList<>();
        maps.forEach((k, v) -> {
            T t = chooseOneWithSameCode(v);
            result.add(t);
        });
        return result;
    }

    /**
     * 删除重复的实体对象
     * 只适用于使用唯一键查询的的数据列表（因为这里没有判断应用是否悬停），根据sourceLevel排序，取最小的
     *
     * @param ts  实体对象列表
     * @param <T> 泛型
     * @return 返回删除重复的实体对象列表
     */
    public <T extends TenantObject> List<T> excludeWithSameCodeAndFilterTenant(List<T> ts) throws DWBusinessException {
        return excludeWithSameCodeAndFilterTenant(ts, null);
    }

    /**
     * 删除重复的实体对象
     * 只适用于使用唯一键查询的的数据列表（因为这里没有判断应用是否悬停），根据sourceLevel排序，取最小的
     *
     * @param ts  实体对象列表
     * @param <T> 泛型
     * @return 返回删除重复的实体对象列表
     */
    public <T extends TenantObject> List<T> excludeWithSameCodeAndFilterTenant(List<T> ts, String tenantId) throws DWBusinessException {
        if (CollectionUtils.isEmpty(ts)) {
            return ts;
        }
        tenantId = StringUtils.isEmpty(tenantId) ? ServiceUtils.getTenantId() : tenantId;
        String finalTenantId = tenantId;
        ts.removeIf(t -> t.getTenantId() != null && !t.getTenantId().equals(finalTenantId) && !t.getTenantId().equals("SYSTEM"));
        return excludeWithSameCode(ts);
    }

    /**
     * 删除重复的实体对象
     * 只适用于使用唯一键查询的的数据列表（因为这里没有判断应用是否悬停），根据sourceLevel排序，取最小的
     *
     * @param ts  相同业务主键的实体对象列表
     * @param <T> 泛型
     * @return 返回删除重复的实体对象列表
     */
    public <T extends TenantObject> T chooseOneWithSameCode(List<T> ts) {
        if (CollectionUtils.isEmpty(ts)) {
            return null;
        }
        for (T t : ts) {
            if ("SYSTEM".equals(t.getTenantId()) || StringUtils.isEmpty(t.getTenantId())) {
                t.setSourceLevel(1000);
            }
            if (null == t.getSourceLevel()) {
                t.setSourceLevel(200);
            }
        }
        ts.sort(Comparator.comparing(T::getSourceLevel));
        return ts.get(0);
    }

    /**
     * 构建查询条件
     *
     * @param query 查询条件
     * @return 返回查询条件
     * @throws DWBusinessException 业务异常
     */
    private Query buildQuery(Map<String, Object> query) throws DWBusinessException {
        Query query1 = new Query();
        List<Criteria> criteriaList = new ArrayList<>();
        String tenantId = MapUtils.getString(query, "tenantId", ServiceUtils.getTenantIdSilent());
        if (null != tenantId) {
            if (!query.containsKey("version")) {
                String tenantVersion = getTenantVersion(tenantId);
                Criteria versionCriteria = new Criteria().and("version").is(tenantVersion);
                criteriaList.add(versionCriteria);
            }
            Criteria tenantCriteria = Criteria.where("tenantId").in(tenantId, "SYSTEM", null);
            criteriaList.add(tenantCriteria);
        } else {
            Criteria tenantCriteria = Criteria.where("tenantId").in("SYSTEM", null);
            criteriaList.add(tenantCriteria);
        }
        query.forEach((k, v) -> {
            if (v instanceof Collection) {
                List list = new ArrayList();
                Collection collection = (Collection) v;
                collection.forEach(ele -> {
                    list.add(ele);
                });
                Criteria in = Criteria.where(k).in(list);
                criteriaList.add(in);
            } else if (v instanceof Criteria) {
                criteriaList.add(Criteria.where(k).elemMatch((Criteria) v));
            } else {
                Criteria criteria = Criteria.where(k).is(v);
                criteriaList.add(criteria);
            }
        });
        Criteria criteria = new Criteria().andOperator(criteriaList.toArray(new Criteria[0]));
        query1.addCriteria(criteria);

        return query1;
    }


    /**
     * 构建查询条件
     *
     * @param criteria 查询条件
     * @param tenantId 查询指定租户
     * @return 查询条件
     * @throws DWBusinessException 抛出异常
     */
    private Query buildQuery(Criteria criteria, String tenantId) throws DWBusinessException {
        Query query = new Query();
        List<Criteria> criteriaList = new ArrayList<>();
        tenantId = StringUtils.isEmpty(tenantId) ? ServiceUtils.getTenantIdSilent() : tenantId;
        if (null != tenantId) {
            if (!criteria.getCriteriaObject().containsKey("version")) {
                String tenantVersion = getTenantVersion(tenantId);
                Criteria versionCriteria = new Criteria().and("version").is(tenantVersion);
                criteriaList.add(versionCriteria);
            }
            Criteria tenantCriteria = Criteria.where("tenantId").in(tenantId, "SYSTEM", null);
            criteriaList.add(tenantCriteria);
        } else {
            // 如果方法没传租户id，并且token也拿不到，则默认查询SYSTEM
            criteria.and("tenantId").in("SYSTEM", null);
        }
        criteriaList.add(criteria);
        Criteria criteria1 = new Criteria().andOperator(criteriaList.toArray(new Criteria[0]));
        query.addCriteria(criteria1);
        return query;
    }

    /**
     * 查询应用，过滤重复code
     *
     * @param appCode  应用code
     * @param tenantId 指定租户id，不传则从token取
     * @return 应用
     * @throws DWBusinessException 抛出异常
     */
    public Application tenantAppRelation(String appCode, String tenantId) throws DWBusinessException {
        List<Application> relation = tenantAppRelations(Collections.singletonList(appCode), tenantId);
        if (!CollectionUtils.isEmpty(relation)) {
            return relation.get(0);
        }
        return null;
    }

    /**
     * 查找当前租户下悬停的应用
     *
     * @param tenantId 租户id
     * @return 悬停的应用code
     * @throws DWBusinessException 抛出异常
     */

    public List<String> findIndividualAppCodes(String tenantId) throws DWBusinessException {
        tenantId = StringUtils.isEmpty(tenantId) ? ServiceUtils.getTenantIdSilent() : tenantId;
        if (StringUtils.isEmpty(tenantId)) {
            return Collections.emptyList();
        }
        Criteria criteria = Criteria.where("tenantId").is(tenantId).and("individualAll").is(true);
        List<Application> apps = mongoTemplate.find(Query.query(criteria), Application.class);
        return apps.stream().map(Application::getCode).collect(Collectors.toList());
    }

    /**
     * 查询应用列表并去重
     *
     * @param appCodes 应用编码
     * @param tenantId 指定租户
     * @return 应用关系
     */
    public List<Application> tenantAppRelations(List<String> appCodes, String tenantId) throws DWBusinessException {
        tenantId = StringUtils.isEmpty(tenantId) ? ServiceUtils.getTenantIdSilent() : tenantId;
        Criteria criteria = Criteria.where("code").in(appCodes);
        if (StringUtils.isEmpty(tenantId)) {
            criteria.and("tenantId").in("SYSTEM", null);
        } else {
            String tenantVersion = getTenantVersion(tenantId);
            criteria.and("tenantId").in(tenantId, "SYSTEM", null).and("version").is(tenantVersion);
        }
        List<Application> relations = mongoTemplate.find(Query.query(criteria), Application.class);
      //  relations = excludeWithSameCode(relations);
        relations = MergeUtil.excludeSameCode(relations,Application.class,null);
        return relations;
    }

    /**
     * queryLevel 0查系统级别，1查个案级别 2系统+个案
     *
     * @param appCode  应用编码
     * @param type     组件类型
     * @param subType  组件子类型
     * @param tenantId 指定租户
     * @param version  指定版本
     * @return 应用关系
     */
    public List<ApplicationRelation> applicationRelations(String appCode, String type, String subType, String tenantId, String version) throws DWBusinessException {
        int queryLevel = 0;
        if ("base_entry".equals(subType) || "statement".equals(subType)) {
            queryLevel = 2;
        } else {
            Application tenantAppRelation = tenantAppRelation(appCode, tenantId);
            if (null != tenantAppRelation && Boolean.TRUE.equals(tenantAppRelation.getIndividualAll())) {
                queryLevel = 1;
            }
        }
        tenantId = StringUtils.isEmpty(tenantId) ? ServiceUtils.getTenantIdSilent() : tenantId;
        if (StringUtils.isEmpty(tenantId)) {
            // 取不到租户id，查系统级数据
            queryLevel = 0;
        }
        String tenantVersion = StringUtils.isEmpty(version) ? getTenantVersion(tenantId) : version;
        Query query = new Query();
        query.addCriteria(Criteria.where("appCode").is(appCode).and("version").is(tenantVersion).and("type").is(type));
        if (null != subType) {
            query.addCriteria(Criteria.where("subType").is(subType));
        }
        switch (queryLevel) {
            case 0:
                query.addCriteria(new Criteria().orOperator(Criteria.where("tenantId").is("SYSTEM"), Criteria.where("tenantId").is(tenantId).and("sourceLevel").is(null)));
                break;
            case 1:
                query.addCriteria(Criteria.where("tenantId").is(tenantId));
                break;
            case 2:
                query.addCriteria(Criteria.where("tenantId").in("SYSTEM", tenantId, null));
                break;
        }
        List<ApplicationRelation> relations = mongoTemplate.find(query, ApplicationRelation.class);
      //  relations = excludeWithSameCode(relations);
        relations = MergeUtil.excludeSameCode(relations,ApplicationRelation.class,null);
        return relations;
    }

    /**
     * 查询应用关联关系
     *
     * @param appCodes 应用编码
     * @param type     组件类型
     * @param subType  组件子类型
     * @param tenantId 指定租户id
     * @param version  指定版本号
     * @return 应用关联关系
     * @throws DWBusinessException 抛出异常
     */
    public List<ApplicationRelation> applicationRelations(List<String> appCodes, String type, String subType, String tenantId, String version) throws DWBusinessException {
        List<ApplicationRelation> result = new ArrayList<>();
        List<Application> tenantAppRelations = tenantAppRelations(appCodes, tenantId);
        Map<String, Application> taMaps = new HashMap<>();
        tenantAppRelations.forEach(t -> taMaps.put(t.getCode(), t));
        tenantId = StringUtils.isEmpty(tenantId) ? ServiceUtils.getTenantIdSilent() : tenantId;
        String tenantVersion = StringUtils.isEmpty(version) ? getTenantVersion(tenantId) : version;
        Query query = new Query();
        query.addCriteria(Criteria.where("appCode").in(appCodes).and("version").is(tenantVersion));
        if (null != type) {
            query.addCriteria(Criteria.where("type").is(type));
        }
        if (null != subType) {
            query.addCriteria(Criteria.where("subType").is(subType));
        }
        List<ApplicationRelation> relations = mongoTemplate.find(query, ApplicationRelation.class);
        for (ApplicationRelation relation : relations) {
            Application tenantAppRelation1 = taMaps.get(relation.getAppCode());
            if (null == tenantAppRelation1) {
                result.add(relation);
                continue;
            }
            int queryLevel = 0;
            if ("base_entry".equals(relation.getSubType()) || "statement".equals(relation.getSubType())) {
                queryLevel = 2;
            } else {
                if (Boolean.TRUE.equals(tenantAppRelation1.getIndividualAll())) {
                    queryLevel = 1;
                }
            }
            if (StringUtils.isEmpty(tenantId)) {
                // 取不到租户id，查系统级数据
                queryLevel = 0;
                tenantId = "SYSTEM";
            }
            switch (queryLevel) {
                case 0:
                    if ("SYSTEM".equals(relation.getTenantId()) || StringUtils.isEmpty(relation.getTenantId()) || (tenantId.equals(relation.getTenantId()) && relation.getSourceLevel() == null)) {
                        result.add(relation);
                    }
                    break;
                case 1:
                    if (tenantId.equals(relation.getTenantId())) {
                        result.add(relation);
                    }
                    break;
                case 2:
                    result.add(relation);
                    break;
            }

        }

     //   result = excludeWithSameCode(result);
        result = MergeUtil.excludeSameCode(result,ApplicationRelation.class,null);
        return result;
    }

    /**
     * 过滤掉同code的数据
     *
     * @param applicationRelations 关联组件
     * @return 过滤后的关联组件
     * @throws DWBusinessException 业务异常
     */
    public List<ApplicationRelation> filterApplicationRelations(List<ApplicationRelation> applicationRelations) throws DWBusinessException {
        // 找到当前租户悬停应用
        List<ApplicationRelation> baseEntryAndStatement = new ArrayList<>();
        List<ApplicationRelation> otherRelations = new ArrayList<>();
        for (ApplicationRelation t : applicationRelations) {
            if ("base_entry".equals(t.getSubType()) || "statement".equals(t.getSubType())) {
                baseEntryAndStatement.add(t);
            } else {
                otherRelations.add(t);
            }
        }
        // 报表和基础资料个案+标准 去重
       // List<ApplicationRelation> result = excludeWithSameCode(baseEntryAndStatement);
        List<ApplicationRelation>  result = MergeUtil.excludeSameCode(baseEntryAndStatement,ApplicationRelation.class,null);
        // 其他根据应用是否悬停过滤
        result.addAll(filterByIndividual(otherRelations, null, ApplicationRelation::getAppCode));
        return result;
    }

    /**
     * 获取租户应用关联关系code
     *
     * @param type 组件类型
     * @return 组件code集合
     * @throws DWBusinessException 业务异常
     */
    public List<String> tenantApplicationRelationCodes(String type) throws DWBusinessException {
        String tenantId = ServiceUtils.getTenantId();
        List<String> appCodes = kmService.tenantApps(tenantId);
        List<String> codes = new ArrayList<>();
        List<ApplicationRelation> relations = applicationRelations(appCodes, type, null, tenantId, null);
        relations.forEach(r -> {
            if (!codes.contains(r.getCode())) {
                codes.add(r.getCode());
            }
        });
        return codes;
    }

    /**
     * 根据应用code和类型获取应用下的组件
     *
     * @param type     组件类型
     * @param appCodes 应用code
     * @return 组件code集合
     */
    public List<String> tenantProductCodes(String type, List<String> appCodes) throws DWBusinessException {
        List<ApplicationRelation> relations = applicationRelations(appCodes, type, null, null, null);
        List<String> codes = new ArrayList<>();
        relations.forEach(r -> {
            if (!codes.contains(r.getCode())) {
                codes.add(r.getCode());
            }
        });
        return codes;
    }

    /**
     * 根据类型和code批量获取应用关联关系
     *
     * @param type  组件类型
     * @param codes 组件code集合
     * @return 组件关联关系集合
     * @throws DWBusinessException 业务异常
     */
    public List<ApplicationRelation> applicationRelations(String type, List<String> codes) throws DWBusinessException {
        Map<String, Object> param = new HashMap<>();
        param.put("type", type);
        param.put("code", codes);
        Query query = buildQuery(param);
        List<ApplicationRelation> relations = mongoTemplate.find(query, ApplicationRelation.class);
        excludeWithSameCode(relations);
        return relations;
    }

    /**
     * 批量去除重复数据
     *
     * @param ts                  待去除重复数据集合
     * @param uniqueKeyFunc       数据唯一标识
     * @param appCode             应用code
     * @param tenantId            租户id
     * @param needJudgeIndividual 是否需要判断悬停
     * @return 去除重复数据集合
     * @throws DWBusinessException 业务异常
     */
    public List<Map<String, Object>> excludeWithSameCode(List<Map<String, Object>> ts, Function<Map<String, Object>, String> uniqueKeyFunc, String appCode, String tenantId, boolean needJudgeIndividual) throws DWBusinessException {
        boolean individualAll = false;
        if (needJudgeIndividual) {
            Application tenantAppRelation = tenantAppRelation(appCode, tenantId);
            individualAll = null != tenantAppRelation && Boolean.TRUE.equals(tenantAppRelation.getIndividualAll());
        }
        List<Map<String, Object>> result;
        // 模块悬停，只返回个案数据
        if (individualAll) {
            // 只保留sourceLevel=100的数据
            result = ts.stream().filter(t -> MapUtils.getInteger(t, "sourceLevel", 1000) < 1000).collect(Collectors.toList());
        } else {
            // 模块非悬停，数据去重
            // 根据uniqueKeyFunc返回的字符串去重，保留sourceLevel最小的，如果sourceLevel为null则视为1000
            result = new ArrayList<>(ts.stream().collect(Collectors.toMap(uniqueKeyFunc, t -> t, (t1, t2) -> MapUtils.getInteger(t1, "sourceLevel", 1000) <= MapUtils.getInteger(t2, "sourceLevel", 1000) ? t1 : t2, LinkedHashMap::new)).values());
        }
        return result;
    }

    /**
     * 批量去除重复数据
     *
     * @param ts                  待去除重复数据集合
     * @param appCode             应用code
     * @param tenantId            租户id
     * @param needJudgeIndividual 是否需要判断悬停
     * @return 去除重复数据集合
     * @throws DWBusinessException 业务异常
     */
    public <T extends HierarchicalEntity> List<T> excludeWithSameCode(List<T> ts, String appCode, String tenantId, boolean needJudgeIndividual) throws DWBusinessException {
        boolean individualAll = false;
        if (needJudgeIndividual) {
            Application tenantAppRelation = tenantAppRelation(appCode, tenantId);
            individualAll = null != tenantAppRelation && Boolean.TRUE.equals(tenantAppRelation.getIndividualAll());
        }
        List<T> result;
        // 模块悬停，只返回个案数据
        if (individualAll) {
            // 只保留sourceLevel=100的数据
            result = ts.stream().filter(t -> Optional.ofNullable(t.getSourceLevel()).orElse(1000) < 1000).collect(Collectors.toList());
        } else {
            // 模块非悬停，数据去重
            // 保留sourceLevel最小的，如果sourceLevel为null则视为1000
            result = new ArrayList<>(ts.stream().collect(Collectors.toMap(T::entityBizCode, t -> t, (t1, t2) -> Optional.ofNullable(t1.getSourceLevel()).orElse(1000) <= Optional.ofNullable(t2.getSourceLevel()).orElse(1000) ? t1 : t2, LinkedHashMap::new)).values());
        }
        return result;
    }

    /**
     * 批量去除重复数据
     *
     * @param ts              待去除重复数据集合
     * @param uniqueKeyFunc   唯一键函数
     * @param sourceLevelFunc 获取数据sourceLevel的函数
     * @return 去除重复数据集合
     * @throws DWBusinessException 业务异常
     */
    public <T> List<T> excludeWithSameCode(List<T> ts, Function<T, String> uniqueKeyFunc, Function<T, Integer> sourceLevelFunc) throws DWBusinessException {
        return new ArrayList<>(ts.stream().collect(Collectors.toMap(uniqueKeyFunc, t -> t, (t1, t2) -> Optional.ofNullable(sourceLevelFunc.apply(t1)).orElse(1000) <= Optional.ofNullable(sourceLevelFunc.apply(t2)).orElse(1000) ? t1 : t2, LinkedHashMap::new)).values());
    }



    public String getTenantVersion(String tenantId) {
        String version =tenantVersion(tenantId);
        if(version==null){
            version = Constants.prodVersion;
        }
        return version;
    }


    @Override
    public MongoTemplate systemTemplate() {
        return mongoTemplate;
    }

    @Override
    public MongoTemplate tenantTemplate() {
        return mongoTemplateTenant;
    }

    @Override
    public String tenantVersion(String tenantId) {
        return kmService.tenantVersion(tenantId);
    }

    @Override
    public List<String> getCodeByTypeAndAppCode(String type, String app) {
        return kmService.getCodeByTypeAndAppCode(type, app);
    }

    @SuppressWarnings("rawtypes")
    @Override
    public boolean isTenantCol(Class c) {
        return tenantClasses.contains(c);
    }

    @SuppressWarnings("rawtypes")
    @Override
    public boolean isSystemCol(Class c) {
        return sysClasses.contains(c);
    }

    /**
     * 对于非唯一键查询的列表，按照应用是否悬停进行过滤
     *
     * @param ts          列表
     * @param tenantId    租户id
     * @param appCodeFunc 获取数据应用code的函数
     * @param <T>         泛型
     * @return 过滤后的列表
     * @throws DWBusinessException 业务异常
     */
    public <T extends TenantObject> List<T> filterByIndividual(List<T> ts, String tenantId, Function<T, String> appCodeFunc) throws DWBusinessException {
        if (CollectionUtils.isEmpty(ts)) {
            return ts;
        }
        List<String> individualAppCodes = findIndividualAppCodes(tenantId);
        List<T> collect = ts.stream().filter(t -> {
            String appCode = appCodeFunc.apply(t);
            int sourceLevel = Optional.ofNullable(t.getSourceLevel()).orElse(1000);
            // 应用悬停? sourceLevel < 1000 : sourceLevel == 1000
            return (individualAppCodes.contains(appCode) && sourceLevel < 1000) || (!individualAppCodes.contains(appCode) && sourceLevel == 1000);
        }).collect(Collectors.toList());
        return excludeWithSameCode(collect);
    }

    /**
     * 添加租户条件
     *
     * @param criteria 查询条件
     * @param appCode  应用code
     * @param tenantId 指定租户id
     * @throws DWBusinessException 业务异常
     */
    public void addTenantCondition(Criteria criteria, String appCode, String tenantId) throws DWBusinessException {
        Application application = tenantAppRelation(appCode, tenantId);
        if (application != null && application.getIndividualAll()) {
            criteria.and("tenantId").is(tenantId);
        } else {
            criteria.and("tenantId").is("SYSTEM");
        }
    }
}
