package com.digiwin.athena.dao.mongodao;

import com.digiwin.athena.constant.modelDriven.ModelDrivenConstant;
import com.digiwin.athena.mongodb.domain.action.ActionMetaData;
import com.digiwin.athena.mongodb.repository.MongoMultiRepositoryDecorator;
import com.digiwin.athena.utils.QueryUtils;
import com.digiwin.athena.utils.UpdateUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Field;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;

import jakarta.annotation.Resource;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 行动dao层
 *
 * @author yuhbb
 * @version 1.0
 * @date 2024/03/11 15:33
 */
@Slf4j
@Repository
public class ActionMetadataMongoDao {

    @Resource
    private MongoMultiRepositoryDecorator mongoMultiRepositoryDecorator;

    public Long count(Query query) {
        return mongoMultiRepositoryDecorator.count(query, ActionMetaData.class);
    }

    public List<ActionMetaData> find(Query query) {
        return mongoMultiRepositoryDecorator.find(query, ActionMetaData.class);
    }

    public ActionMetaData findByActionId(String actionId) {
        Query query = new Query(Criteria.where("actionId").is(actionId));
        return mongoMultiRepositoryDecorator.findOne(query, ActionMetaData.class);
    }

    public List<ActionMetaData> findList(Collection<String> list) {
        Query query = new Query(Criteria.where("actionId").in(list));
        return find(query);
    }

    public void insert(ActionMetaData actionMetaData) {
        mongoMultiRepositoryDecorator.insert(actionMetaData);
    }

    public void delete(String actionId) {
        mongoMultiRepositoryDecorator.delete(QueryUtils.findByActionId(actionId), ActionMetaData.class);
    }

    public void updateById(ActionMetaData actionMetaData) {
        mongoMultiRepositoryDecorator.updateFirst(QueryUtils.findById(actionMetaData.getObjectId()),
                UpdateUtils.getUpdate(actionMetaData), ActionMetaData.class);
    }

    public void delete(Query query) {
        mongoMultiRepositoryDecorator.delete(query, ActionMetaData.class);
    }


    public List<ActionMetaData> findByAdpApplication(String adpApplication) {
        Query query = new Query(Criteria.where("adpApplication").is(adpApplication));
        return mongoMultiRepositoryDecorator.find(query, ActionMetaData.class);
    }

    public ActionMetaData selectByActionId(String actionId) {
        Query query = new Query(Criteria.where(ModelDrivenConstant.MODEL_ACTIONID).is(actionId));
        return mongoMultiRepositoryDecorator.findOne(query, ActionMetaData.class);
    }

    public ActionMetaData selectByActionIdOrServiceName(String value) {
        Criteria criteria = new Criteria();
        criteria.orOperator(Criteria.where(ModelDrivenConstant.MODEL_ACTIONID).is(value),
                Criteria.where(ModelDrivenConstant.MODEL_SERVICENAME).is(value));
        return mongoMultiRepositoryDecorator.findOne(new Query(criteria), ActionMetaData.class);
    }

    public List<String> selectActionIdByApplicationAndPageable(String application, Pageable pageable) {
        Criteria criteria = Criteria.where("application").is(application);
        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.ASC,"_id"));
        query.with(pageable);
        Field fields = query.fields();
        fields.include("actionId");
        return mongoMultiRepositoryDecorator.find(query,ActionMetaData.class).stream().map(ActionMetaData::getActionId).collect(Collectors.toList());
    }

    public ActionMetaData selectByActionIdAndApplication(String actionId, String application) {
        Criteria criteria = Criteria.where("actionId").is(actionId).and("application").is(application);
        return mongoMultiRepositoryDecorator.findOne(new Query(criteria),ActionMetaData.class);
    }


    private static Criteria getActionIdOrActionNameCriteria(String condition) {
        Criteria criteria = new Criteria();
        if (StringUtils.isNotEmpty(condition)) {
            condition = ".*" + condition + ".*";
            criteria.orOperator(
                    Criteria.where("actionId").regex(condition, "si")
                    , Criteria.where("actionName").regex(condition, "si"));
        }
        return criteria;
    }

    public List<ActionMetaData> selectBasicInfoByPageable(String condition,List<String> appCodes,Boolean needEsp,String label, Pageable pageable) {
        Criteria criteria = getActionIdOrActionNameCriteria(condition);
        criteria.andOperator(new Criteria().orOperator(Criteria.where("application").in(appCodes),Criteria.where("adpApplication").in(appCodes)));

        if (!needEsp){
            criteria.and("application").ne("espCommon");
        }
        if (!StringUtils.isEmpty(label)){
            criteria.and("label").is(label);
        }
        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.ASC,"_id"));
        query.with(pageable);
        query.fields().exclude("response_object","request_parameters");
        return mongoMultiRepositoryDecorator.find(query, ActionMetaData.class);
    }

    public Long countBasicInfoByPageable(String condition, List<String> appCodes,Boolean needEsp,String label) {
        Criteria criteria = getActionIdOrActionNameCriteria(condition);
        criteria.andOperator(new Criteria().orOperator(Criteria.where("application").in(appCodes),Criteria.where("adpApplication").in(appCodes)));
        if (!needEsp){
            criteria.and("application").ne("espCommon");
        }
        if (!StringUtils.isEmpty(label)){
            criteria.and("label").is(label);
        }
        return mongoMultiRepositoryDecorator.count(new Query(criteria), ActionMetaData.class);
    }

    public <T> Long countBasicInfoByPageable(String condition, List<String> excludeActionIdList,  List<String> appCodes,Boolean needEsp,String label) {
        Criteria criteria = getActionIdOrActionNameCriteria(condition);
        criteria.andOperator(new Criteria().orOperator(Criteria.where("application").in(appCodes),Criteria.where("adpApplication").in(appCodes)));
        if (!excludeActionIdList.isEmpty()){
            criteria.and("actionId").not().in(excludeActionIdList);
        }
        if (!needEsp){
            criteria.and("application").ne("espCommon");
        }
        if (!StringUtils.isEmpty(label)){
            criteria.and("label").is(label);
        }
        return mongoMultiRepositoryDecorator.count(new Query(criteria), ActionMetaData.class);
    }

    public <T> List<ActionMetaData> selectBasicInfoByPageable(String condition, List<String> excludeActionIdList, List<String> appCodes,Boolean needEsp,String label, Pageable pageable) {
        Criteria criteria = getActionIdOrActionNameCriteria(condition);
        criteria.andOperator(new Criteria().orOperator(Criteria.where("application").in(appCodes),Criteria.where("adpApplication").in(appCodes)));

        if (!excludeActionIdList.isEmpty()) {
            criteria.and("actionId").not().in(excludeActionIdList);
        }

        if (!needEsp){
            criteria.and("application").ne("espCommon");
        }
        if (!StringUtils.isEmpty(label)){
            criteria.and("label").is(label);
        }
        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.ASC,"_id"));
        query.with(pageable);
        query.fields().exclude("response_object","request_parameters");
        return mongoMultiRepositoryDecorator.find(query, ActionMetaData.class);
    }

    public List<ActionMetaData> selectBasicInfoByApplication(String condition, String appCode) {
        Criteria criteria = getActionIdOrActionNameCriteria(condition);
        criteria.and("application").is(appCode);
        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.ASC,"_id"));
        query.fields().exclude("response_object","request_parameters");
        return mongoMultiRepositoryDecorator.find(query, ActionMetaData.class);
    }


    private Criteria matchDataName(String condition){
        condition = ".*" + condition + ".*";
        Criteria criteria = new Criteria();
        Criteria cri = Criteria.where("data_name").regex(condition, "six");
        criteria.orOperator(
                Criteria.where("response_object.field").elemMatch(cri)
                , Criteria.where(ModelDrivenConstant.MODEL_ACTIONID).regex(condition, "six")
                , Criteria.where("response_object.data_name").regex(condition, "six"));
        return criteria;
    }

    public List<ActionMetaData> selectActionByDataNamePageable(String condition,List<String> excludeActionIds, Pageable pageable) {
        Criteria criteria = matchDataName(condition);
        if (!excludeActionIds.isEmpty()){
            criteria.and("actionId").nin(excludeActionIds);
        }

        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.ASC,"_id"));
        query.with(pageable);

        return mongoMultiRepositoryDecorator.find(query,ActionMetaData.class);
    }

    public Long countActionByDataName(String condition,List<String> excludeActionIds) {
        Criteria criteria = matchDataName(condition);
        if (!excludeActionIds.isEmpty()){
            criteria.and("actionId").nin(excludeActionIds);
        }
        Query query = new Query(criteria);
        return mongoMultiRepositoryDecorator.count(query,ActionMetaData.class);
    }

    public List<ActionMetaData> selectBasicInfoByActionIds(List<String> actionIds) {
        Criteria criteria = Criteria.where("actionId").in(actionIds);
        Query query = new Query(criteria);
        query.fields().exclude("response_object","request_parameters");
        return mongoMultiRepositoryDecorator.find(query, ActionMetaData.class);
    }

    public void delete(ActionMetaData actionMetaData) {
        mongoMultiRepositoryDecorator.delete(actionMetaData);
    }

    public List<String> selectAllActionIdByApplication(String appCode) {
        Criteria criteria = Criteria.where("application").is(appCode);
        Query query = new Query(criteria);
        query.fields().include("actionId");
        return mongoMultiRepositoryDecorator.find(query,ActionMetaData.class).stream().map(ActionMetaData::getActionId).collect(Collectors.toList());
    }

    public List<ActionMetaData> selectActionByDataNameAndApplication(String condition, String individualCaseAppCode) {
        Criteria criteria = matchDataName(condition);
        criteria.and("application").is(individualCaseAppCode);
        return mongoMultiRepositoryDecorator.find(new Query(criteria),ActionMetaData.class);
    }

    public void save(ActionMetaData resActionMetaData) {
        mongoMultiRepositoryDecorator.save(resActionMetaData);
    }

    public void remove(ActionMetaData actionMetaData) {
        mongoMultiRepositoryDecorator.delete(actionMetaData);
    }

    public ActionMetaData findByActionIdAndAdpApplication(String actionId) {
        Criteria criteria = new Criteria();

        Criteria criteria1 = Criteria.where(ModelDrivenConstant.MODEL_ACTIONID).is(actionId).and("label").ne("EspAction");
        Criteria criteria2 = Criteria.where(ModelDrivenConstant.MODEL_ACTIONID).is(actionId).and("adpApplication").exists(true);
        criteria.orOperator(criteria1,criteria2);
        Query query = new Query(criteria);
        return mongoMultiRepositoryDecorator.findOne(query, ActionMetaData.class);
    }

    public List<ActionMetaData> selectActionIdByActionsExcludeEsp(Collection<String> actionIds) {
        Criteria criteria = Criteria.where("actionId").in(actionIds).and("label").ne("EspAction");
        Query query = new Query(criteria);
        query.fields().include("actionId").include("application");
        return mongoMultiRepositoryDecorator.find(query,ActionMetaData.class);
    }

    public Set<String> selectActionIdByActionIds(List<String> actionIds) {
        Criteria criteria = Criteria.where("actionId").in(actionIds);
        Query query = new Query(criteria);
        query.fields().include("actionId");
        return mongoMultiRepositoryDecorator.find(query, ActionMetaData.class).stream().map(ActionMetaData::getActionId).collect(Collectors.toSet());
    }
}
