package com.digiwin.athena.dao.mongodao;

import com.digiwin.athena.mongodb.domain.application.Application;
import com.digiwin.athena.mongodb.repository.MongoPrimaryRepositoryDecorator;
import com.digiwin.athena.utils.QueryUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.aggregation.Aggregation;
import org.springframework.data.mongodb.core.aggregation.AggregationResults;
import org.springframework.data.mongodb.core.aggregation.GroupOperation;
import org.springframework.data.mongodb.core.aggregation.MatchOperation;
import org.springframework.data.mongodb.core.aggregation.ProjectionOperation;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Field;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;
import org.springframework.util.CollectionUtils;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * <Description> <br>
 *
 * @author wang.xinyuan<br>
 * @version 1.0<br>
 * @CreateDate 2024/1/23 <br>
 */
@Slf4j
@Repository
public class ApplicationMongoDao {


    @Autowired
    private MongoPrimaryRepositoryDecorator mongoPrimaryRepositoryDecorator;

    public Application getApplication(String code) {
        return mongoPrimaryRepositoryDecorator.findOne(QueryUtils.findByCode(code), Application.class);
    }

    public Application getApplication(Query query) {
        return mongoPrimaryRepositoryDecorator.findOne(query, Application.class);
    }

    public List<Application> getApplications(Query query) {
        return mongoPrimaryRepositoryDecorator.find(query, Application.class);
    }

    public List<Application> selectByCodes(Collection<String> applicationCode) {
        Criteria criteria = Criteria.where("code").in(applicationCode);
        Query query = new Query(criteria);
        return mongoPrimaryRepositoryDecorator.find(query, Application.class);
    }

    public List<Application> selectByCodesExcludeTargetAppCodes(Collection<String> applicationCode,Collection<String> excludeCodes) {
        Criteria criteria = Criteria.where("code").in(applicationCode).nin(excludeCodes);
        Query query = new Query(criteria);
        return mongoPrimaryRepositoryDecorator.find(query, Application.class);
    }

    public List<Application> selectByPage(Query query, Integer pageNum, Integer pageSize) {
        Pageable pageable;
        if (pageNum != null && pageSize != null) {
            pageable = PageRequest.of(pageNum, pageSize);
            query.with(pageable);
        }
        query.with(Sort.by(Sort.Direction.ASC, "_id"));
        return mongoPrimaryRepositoryDecorator.find(query, Application.class);
    }

    public Long queryCountByQuery(Query query) {
        return mongoPrimaryRepositoryDecorator.count(query, Application.class);
    }

    public void saveNotSetEdit(Application application) {
        mongoPrimaryRepositoryDecorator.saveNotSetEdit(application);
    }

    public Set<String> selectApplicationCodeByCommonApp(Boolean isCommonApp) {
        Criteria criteria = Criteria.where("commonApp").is(isCommonApp);
        Query query = new Query(criteria);
        Field fields = query.fields();
        fields.include("code");

        List<Application> applications = mongoPrimaryRepositoryDecorator.find(query, Application.class);
        return applications.stream().map(Application::getCode).collect(Collectors.toSet());
    }

    public Application selectByCode(String code) {
        Query query = new Query();
        Criteria criteria = Criteria.where("code").is(code);
        query.addCriteria(criteria);
        return this.mongoPrimaryRepositoryDecorator.findOne(query, Application.class);
    }

    public List<Application> selectByCategory(String category) {
        Query query = new Query();
        Criteria criteria = Criteria.where("category").is(category);
        query.addCriteria(criteria);
        return this.mongoPrimaryRepositoryDecorator.find(query, Application.class);
    }

    public List<Application> selectByAppType(List<Integer> appTypes) {
        Query query = new Query();
        Criteria criteria = Criteria.where("appType").in(appTypes);
        query.addCriteria(criteria);
        return this.mongoPrimaryRepositoryDecorator.find(query, Application.class);
    }


    public List<Application> selectCheckUc(String applicationCode) {
        Query query = new Query();
        Criteria criteria = Criteria.where("sourceApplicationCode").is(applicationCode).and("individual").is(true);
        query.addCriteria(criteria);
        return this.mongoPrimaryRepositoryDecorator.find(query, Application.class);
    }


    public void insert(Application application) {
        mongoPrimaryRepositoryDecorator.insert(application);
    }

    public Long selectOnlyOneByIndividualAndSourceApplicationCodeAndTenantId(String standardAppCode, String tenantId) {

        Criteria criteria1 = new Criteria().orOperator(Criteria.where("belongTenantInfo.tenantId").is(tenantId), Criteria.where("useTenantInfos.tenantId").is(tenantId));

        Criteria criteria = Criteria.where("sourceApplicationCode").is(standardAppCode)
                .and("individual").is(true)
                .andOperator(criteria1);

        Query query = new Query(criteria);
        query.limit(1);
        return mongoPrimaryRepositoryDecorator.count(query, Application.class);
    }

    public List<Application> selectBySourceApplicationCodeAndIndividual(String sourceApplicationCode) {
        Criteria criteria = Criteria.where("sourceApplicationCode").is(sourceApplicationCode).and("individual").is(true);

        return mongoPrimaryRepositoryDecorator.find(new Query(criteria), Application.class);
    }

    public void save(Application application) {
        mongoPrimaryRepositoryDecorator.save(application);
    }

    public Long countStandardByTypeAndNameAndTenantIdAndCodes(String name, List<Integer> types, String tenantId,List<String> codes) {
        Criteria criteria = Criteria.where("tenantId").is(tenantId)
                .and("individual").ne(true)
                .and("code").in(codes);
        if (!StringUtils.isEmpty(name)){
            criteria.and("name").regex(".*" + name + ".*", "i");
        }

        if (!CollectionUtils.isEmpty(types)){
            criteria.and("appType").in(types);
        }

        return mongoPrimaryRepositoryDecorator.count(new Query(criteria), Application.class);
    }

    public List<Application> selectStandardByTypeAndNameAndTenantIdAndCodePageable(String name, List<Integer> types, String tenantId,List<String> codes, Pageable pageable) {
        Criteria criteria = Criteria.where("tenantId").is(tenantId)
                .and("individual").ne(true)
                .and("code").in(codes);

        if (!StringUtils.isEmpty(name)){
            criteria.and("name").regex(".*" + name + ".*", "i");
        }

        if (!CollectionUtils.isEmpty(types)){
            criteria.and("appType").in(types);
        }
        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.ASC, "_id"));
        query.with(pageable);
        return mongoPrimaryRepositoryDecorator.find(query, Application.class);
    }

    public List<Application> selectIndividualAppByTenantId(String tenantId) {
        Criteria criteria = Criteria.where("tenantId").is(tenantId).and("individual").is(true);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria), Application.class);
    }


    public List<Application> selectStandardAppByTenantId(String tenantId) {
        Criteria criteria = Criteria.where("tenantId").is(tenantId).and("individual").ne(true);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria), Application.class);
    }

    public List<Application> selectIndividualByTenantIdAndAppTypeAndDeleteStatus(String tenantId,List<Integer> excludeAppType,List<String> excludeDeleteStatus) {
        Query query = new Query(Criteria.where("tenantId").in(tenantId)
                .and("abandon").is(0)
                .and("individual").is(true)
                .and("deleteStatus").nin(excludeAppType)
                .and("appType").nin(excludeDeleteStatus));
        query.with(Sort.by(Sort.Direction.ASC, "_id"));
        return mongoPrimaryRepositoryDecorator.find(query, Application.class);
    }

    public List<Application> selectStandardByTenantIdAndAppTypeAndDeleteStatus(String tenantId,List<Integer> excludeAppType,List<String> excludeDeleteStatus) {

        Query query = new Query(Criteria.where("tenantId").in(tenantId)
                .and("abandon").is(0)
                .and("individual").ne(true)
                .and("deleteStatus").nin(excludeAppType)
                .and("appType").nin(excludeDeleteStatus));
        query.with(Sort.by(Sort.Direction.ASC, "_id"));

        return mongoPrimaryRepositoryDecorator.find(query, Application.class);
    }

    public List<String> selectStandardAppCodeByTenantIdAndCommonApp(String tenantId,Boolean commonApp) {
        Criteria criteria = Criteria.where("tenantId").is(tenantId).and("commonApp").is(commonApp).and("individual").ne(true);
        Query query = new Query(criteria);
        Field fields = query.fields();
        fields.include("code");
        return mongoPrimaryRepositoryDecorator.find(query, Application.class).stream().map(Application::getCode).collect(Collectors.toList());
    }

    public List<Application> selectCommonAppByTenantId(String tenantId) {
        Criteria criteria = Criteria.where("tenantId").is(tenantId).and("commonApp").is(true);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria), Application.class);
    }

    public List<Application> selectDistinctByType(Integer type,List<String> codes) {
        MatchOperation match = Aggregation.match(Criteria.where("appType").is(type).and("code").in(codes));
        GroupOperation group = Aggregation.group("code")
                .first("name").as("name")
                .first("code").as("code")
                .first("description").as("description")
                .first("appType").as("appType")
                .first("iconName").as("iconName")
                .first("iconBgcolor").as("iconBgcolor")
                .first("lang").as("lang");

        ProjectionOperation project = Aggregation.project()
                .andExclude("_id")
                .andInclude("name", "code","description","appType","iconName","iconBgcolor","lang");
        Aggregation aggregation = Aggregation.newAggregation(match, group, project);
        AggregationResults<Application> aggregate = mongoPrimaryRepositoryDecorator.aggregate(aggregation,"application", Application.class);
        return aggregate.getMappedResults();
    }

    public Boolean existBySourceApplicationCode(String code) {
        Criteria criteria = Criteria.where("sourceApplicationCode").is(code);
        Query query = new Query(criteria);
        query.limit(1);
        return mongoPrimaryRepositoryDecorator.count(query,Application.class)>0;
    }

    public List<String> selectAppCodeByTenantId(String tenantId) {
        Criteria criteria = Criteria.where("tenantId").is(tenantId).and("individual").ne(true);
        Query query = new Query(criteria);
        query.fields().include("code");
        return mongoPrimaryRepositoryDecorator.find(query,Application.class).stream().map(Application::getCode).collect(Collectors.toList());
    }

    public List<Application> selectByCodesAndIndividual(List<String> appCodes, Boolean individual) {
        Criteria criteria = Criteria.where("code").in(appCodes).and("individual").is(individual);
        Query query = new Query(criteria);
        Field fields = query.fields();
        fields.include("code");
        fields.include("appStatus");
        return mongoPrimaryRepositoryDecorator.find(query,Application.class);
    }

    public List<Application> findSelfCreatedApp(String userId, String currentTenantId, Integer appType, String condition) {
        Criteria criteria = Criteria.where("createBy").is(userId)
                .and("abandon").is(0)
                .and("tenantId").is(currentTenantId)
                .and("individual").ne(true);
        if (null != appType) {
            criteria.and("appType").is(appType);
        }

        addSearchCondition(condition, criteria);

        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.DESC,"_id"));
        return mongoPrimaryRepositoryDecorator.find(query,Application.class);
    }

    public List<Application> findSelfCreatedIndividualApp(String userId, String currentTenantId, Integer appType, String condition) {
        Criteria criteria = Criteria.where("createBy").is(userId)
                .and("abandon").is(0)
                .and("tenantId").is(currentTenantId)
                .and("individual").is(true);
        if (null != appType) {
            criteria.and("appType").is(appType);
        }

        addSearchCondition(condition, criteria);

        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.DESC,"_id"));
        return mongoPrimaryRepositoryDecorator.find(query,Application.class);
    }

    private static void addSearchCondition(String condition, Criteria criteria) {
        if (StringUtils.isNotBlank(condition)) {
            Criteria c1 = Criteria.where("code").regex(".*" + condition + ".*", "i");
            Criteria c2 = Criteria.where("name").regex(".*" + condition + ".*", "i");
            Criteria c3 = Criteria.where("description").regex(".*" + condition + ".*", "i");
            criteria.orOperator(c1, c2, c3);
        }
    }

    /**
     * 查询标准应用
     * @param codes
     * @param deleteStatus
     * @return
     */
    public List<Application> selectByCodeAndDeleteStatus(List<String> codes, List<String> deleteStatus) {
        Criteria criteria = Criteria.where("deleteStatus").nin(deleteStatus)
                .and("appType").ne(8)
                .and("individual").ne(true);
        if (!CollectionUtils.isEmpty(codes)){
            criteria.and("code").in(codes);
        }

        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.ASC, "_id"));
        return mongoPrimaryRepositoryDecorator.find(query,Application.class);
    }

    /**
     * 查询个案应用
     * @param codes
     * @param deleteStatus
     * @return
     */
    public List<Application> selectIndividualByCodeAndDeleteStatus(List<String> codes, List<String> deleteStatus) {
        Criteria criteria = Criteria.where("deleteStatus").nin(deleteStatus)
                .and("appType").ne(8)
                .and("individual").is(true);
        if (!CollectionUtils.isEmpty(codes)){
            criteria.and("code").in(codes);
        }

        Query query = new Query(criteria);
        query.with(Sort.by(Sort.Direction.ASC, "_id"));
        return mongoPrimaryRepositoryDecorator.find(query,Application.class);
    }

    public List<Integer> selectAppType(List<String> codes, List<Integer> solutionTypes) {
        Criteria criteria = Criteria.where("code").in(codes).and("appType").in(solutionTypes);
        Query query = new Query(criteria);
        query.fields().include("appType");
        return mongoPrimaryRepositoryDecorator.find(query, Application.class)
                .stream().map(Application::getAppType).distinct().collect(Collectors.toList());
    }

    public List<String> selectByQuery(Query query) {
        return mongoPrimaryRepositoryDecorator.find(query, Application.class)
                .stream().map(Application::getCode).distinct().collect(Collectors.toList());
    }

    public Application selectByCodeAndTenantId(String applicationCode, String tenantId) {
        return mongoPrimaryRepositoryDecorator.findOne(new Query(Criteria.where("code").is(applicationCode).and("tenantId").is(tenantId)), Application.class);
    }

    public List<Application> selectAllBusinessApplication(){
        Criteria criteria = Criteria.where("tag").exists(true).ne(null);
        Query query = new Query(criteria);
        return mongoPrimaryRepositoryDecorator.find(query, Application.class);
    }

    public List<Application> selectBasicInfoByCodesAndTenantId(Collection<String> codes,String tenantId) {
        Criteria criteria = Criteria.where("code").in(codes).and("tenantId").is(tenantId);
        Query query = new Query(criteria);
        query.fields().include("name","tenantId","lang");
        return mongoPrimaryRepositoryDecorator.find(query,Application.class);
    }

    public Long selectCountByCodesAndTenantId(Collection<String> appCodes, String tenantId) {
        Criteria criteria = Criteria.where("code").in(appCodes).and("tenantId").is(tenantId);
        return mongoPrimaryRepositoryDecorator.count(new Query(criteria),Application.class);
    }

    public List<Application> selectByCodesAndTenantId(Collection<String> appCodes, String tenantId) {
        Criteria criteria = Criteria.where("code").in(appCodes).and("tenantId").is(tenantId);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Application.class);
    }
    
    public List<Application> findIndividualAppsBySourceAppCode(String sourceApplicationCode){
        Criteria criteria = Criteria.where("sourceApplicationCode").is(sourceApplicationCode).and("individual").is(Boolean.TRUE);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Application.class);
    }

    public List<Application> findIndAppsBySrcAppCodeAndSrcBranch(String sourceApplicationCode, String individualComeFromBranch){
        Criteria criteria = Criteria.where("sourceApplicationCode").is(sourceApplicationCode).and("individual").is(Boolean.TRUE)
                .and("individualComeFromBranch").is(individualComeFromBranch);
        return mongoPrimaryRepositoryDecorator.find(new Query(criteria),Application.class);
    }

    public List<Application> selectBySourceAppCodeAndSourceBranchAndIndividual(String sourceAppCode, String sourceBranch) {
        Criteria criteria = Criteria.where("sourceApplicationCode").is(sourceAppCode).and("individualComeFromBranch").is(sourceBranch).and("individual").is(true);

        return mongoPrimaryRepositoryDecorator.find(new Query(criteria), Application.class);
    }

    public Collection<Application> selectCodeByCodesAndAppType(List<String> codes,Integer appType) {
        Criteria criteria = Criteria.where("code").in(codes).and("appType").is(appType);
        Query query = new Query(criteria);
        query.fields().include("code", "name", "lang");
        return mongoPrimaryRepositoryDecorator.find(query, Application.class);
    }

    public List<Application> findBasicInfoByCodes(List<String> codes){
        Criteria criteria = Criteria.where("code").in(codes);
        Query query = new Query(criteria);
        query.fields().include("code").include("name").include("lang");
        return mongoPrimaryRepositoryDecorator.find(query, Application.class);
    }

}