package com.digiwin.athena.service.applicationHandler.handler;

import com.alibaba.fastjson.JSON;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.base.ValidateException;
import com.digiwin.athena.constant.DeleteApplicationConstant;
import com.digiwin.athena.dao.mongodao.DeleteApplicationDetailMongoDao;
import com.digiwin.athena.enums.DeleteApplicationProcessEnum;
import com.digiwin.athena.mongodb.domain.application.Application;
import com.digiwin.athena.mongodb.domain.application.DeleteApplicationDetail;
import com.digiwin.athena.mongodb.repository.MongoSystemRepositoryDecorator;
import com.digiwin.athena.service.RedisService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Date;

@Component
public abstract class AbstractDeletionHandler {

    @Autowired
    protected RedisService redisService;

    @Autowired
    private MongoSystemRepositoryDecorator mongoSystemRepositoryDecorator;

    private DeleteApplicationDetailMongoDao deleteApplicationDetailMongoDao;

    @Autowired
    public void setDeleteApplicationDetailMongoDao(DeleteApplicationDetailMongoDao deleteApplicationDetailMongoDao) {
        this.deleteApplicationDetailMongoDao = deleteApplicationDetailMongoDao;
    }

    DeleteApplicationProcessEnum getDeleteType() {
        return null;
    }

    protected abstract void doDeleteExecute(Application application);

    protected abstract void doValidateExecute(Application application);

    protected abstract void doBackup(Application application);

    /**
     * 执行删除逻辑
     *
     * @param application 应用信息
     * @param deleteNo 删除编号，一次应用删除一个共同的编号
     */
    public final void deleteExecute(Application application, String deleteNo) {

        long startTime = System.currentTimeMillis();
        try {

            saveDeleteLogDetailPre(application, deleteNo);

            doDeleteExecute(application);

            saveValidateDetailPost(application, deleteNo, startTime);

            processExecute(application, deleteNo);
        } catch (Exception e) {
            failExecute(application, deleteNo, e, 3, startTime);
            throw new BusinessException(e);
        }

    }

    /**
     * 执行备份
     *
     * @param application 应用信息
     * @param deleteNo 删除编号，一次应用删除一个共同的编号
     */
    public final void backup(Application application, String deleteNo) {

        long startTime = System.currentTimeMillis();

        try {
            saveDeleteLogDetailPre(application, deleteNo);

            doBackup(application);

            saveValidateDetailPost(application, deleteNo, startTime);

            processExecute(application, deleteNo);
        } catch (Exception e) {
            failExecute(application, deleteNo, e, 2, startTime);
            throw new BusinessException(e);
        }

    }

    /**
     * 记录删除的进度 存放redis
     *
     * @param application 应用信息
     * @param deleteNo 删除编号，一次应用删除一个共同的编号
     */
    public void processExecute(Application application, String deleteNo) {

        String key = DeleteApplicationConstant.DELETE_APP_PROCESS_KEY + application.getCode()
            + DeleteApplicationConstant.COLON + deleteNo;

        // 设置进度
        redisService.set(key, calculateProgress().doubleValue(), DeleteApplicationConstant.EXPIRE_TIME);

    }

    /**
     * 校验逻辑处理
     *
     * @param application 应用信息
     * @param deleteNo 删除编号，一次应用删除一个共同的编号
     */
    public final void validate(Application application, String deleteNo) {

        long startTime = System.currentTimeMillis();

        try {
            saveDeleteLogDetailPre(application, deleteNo);

            doValidateExecute(application);

            saveValidateDetailPost(application, deleteNo, startTime);

            processExecute(application, deleteNo);
        } catch (Exception e) {
            // 记录操作日志
            failExecute(application, deleteNo, e, 1, startTime);
            throw new ValidateException(e.getMessage());
        }

    }

    /**
     * 执行之前记录操作日志入库
     *
     * @param application 应用信息
     * @param deleteNo 删除编号，一次应用删除一个共同的编号
     */
    protected void saveDeleteLogDetailPre(Application application, String deleteNo) {
        DeleteApplicationDetail deleteApplicationDetail = new DeleteApplicationDetail();

        deleteApplicationDetail.setApplicationCode(application.getCode()).setDeleteNo(deleteNo).setTime(new Date())
            .setResult(DeleteApplicationConstant.START)
            .setContent(DeleteApplicationConstant.START_CHE + getDeleteType().getContent());
        deleteApplicationDetailMongoDao.insert(deleteApplicationDetail);
    }

    /**
     * 执行之后记录操作日志入库
     *
     * @param application 应用信息
     * @param deleteNo 删除编号，一次应用删除一个共同的编号
     */
    public void saveValidateDetailPost(Application application, String deleteNo, long startTime) {
        DeleteApplicationDetail deleteApplicationDetail = new DeleteApplicationDetail();
        deleteApplicationDetail.setApplicationCode(application.getCode())
            .setContent(getDeleteType().getContent() + DeleteApplicationConstant.END_CHE).setDeleteNo(deleteNo)
            .setExecuteTimeMills(calculateExecutionTime(startTime)).setResult(DeleteApplicationConstant.SUCCESS)
            .setTime(new Date());
        deleteApplicationDetailMongoDao.insert(deleteApplicationDetail);
    }

    /**
     * 计算方法执行时间
     */
    protected long calculateExecutionTime(long startTime) {
        // 记录结束时间
        long endTime = System.currentTimeMillis();
        return endTime - startTime;
    }

    /**
     * 计算删除进度
     */
    protected BigDecimal calculateProgress() {
        return new BigDecimal(this.getDeleteType().ordinal() + 1)
            .divide(new BigDecimal(DeleteApplicationProcessEnum.values().length), 2, RoundingMode.FLOOR);
    }

    /**
     * 记录异常信息
     *
     * @param application 应用信息
     * @param deleteNo 删除编号，一次应用删除一个共同的编号
     * @param e 异常信息
     * @param type 1 校验 2 备份 3 删除
     * @param startTime 开始时间
     */
    public void failExecute(Application application, String deleteNo, Exception e, int type, long startTime) {

        String key = DeleteApplicationConstant.DELETE_APP_PROCESS_KEY + application.getCode()
            + DeleteApplicationConstant.COLON + deleteNo;

        DeleteApplicationDetail deleteApplicationDetail =
            new DeleteApplicationDetail().setApplicationCode(application.getCode()).setDeleteNo(deleteNo)
                .setExecuteTimeMills(calculateExecutionTime(startTime))
                .setErrorStack(JSON.parseArray(JSON.toJSONString(e.getStackTrace()))).setCause(e.toString())
                .setErrorMessage(e.getMessage()).setResult(DeleteApplicationConstant.FAIL);

        mongoSystemRepositoryDecorator.insert(deleteApplicationDetail);
        // 设置进度
        redisService.set(key, -1, DeleteApplicationConstant.EXPIRE_TIME);

        if (type == 3) {
            // 修改应用状态为删除失败
            application.setDeleteStatus(DeleteApplicationConstant.DELETE_FAILED);
            application.setDeleteOrder(DeleteApplicationConstant.DELETE_FAILED_ORDER);
            mongoSystemRepositoryDecorator.save(application);
        }
    }
}
