package com.digiwin.athena.controller.assembly;

import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.base.ErrorCodeConstants;
import com.digiwin.athena.bo.assembly.Designer;
import com.digiwin.athena.convertor.assembly.DesignerConvertor;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.digiwin.athena.dto.ResultDto;
import com.digiwin.athena.dto.assembly.designer.DesignerDetailDTO;
import com.digiwin.athena.dto.assembly.designer.DesignerListDTO;
import com.digiwin.athena.dto.assembly.designer.DesignerPageQo;
import com.digiwin.athena.dto.assembly.designer.DesignerSaveDTO;
import com.digiwin.athena.service.ManageBackgroundUserService;
import com.digiwin.athena.service.assembly.impl.DDesignerServiceImpl;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotBlank;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

/**
 * 设计器管理
 */
@Slf4j
@RestController
@RequestMapping("/assembly/designer/d/")
@RequiredArgsConstructor
@Validated
public class DDesignerController {

    private final DDesignerServiceImpl designerService;
    private final ManageBackgroundUserService manageBackgroundUserService;


    /**
     * 列表
     */
    @PostMapping("/list")
    public ResultDto<Pagination<DesignerListDTO>> list(@RequestBody PageReqCondition<DesignerPageQo> pageReqCondition) {
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }
        Pagination<DesignerListDTO> res = designerService.findList(pageReqCondition);
        return ResultDto.createSuccess(res);
    }

    /**
     * 保存
     */
    @PostMapping("/save")
    public ResultDto<DesignerDetailDTO> createDesignTimeDesigner(@Valid @RequestBody DesignerSaveDTO param) {
        log.info("收到创建设计时态设计器请求: {}", param);
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }
        Designer bo = designerService.save(param);
        return ResultDto.createSuccess(DesignerConvertor.INSTANCE.toDto(bo));
    }

    /**
     * 删除
     */
    @GetMapping("/{id}/delete")
    public ResultDto<Void> deleteDesignTimeDesigner(@NotBlank @PathVariable String id) {
        log.info("收到删除设计时态设计器请求: {}", id);
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }
        designerService.deleteById(id);
        return ResultDto.createSuccess(null);
    }

    /**
     * 详情
     */
    @GetMapping("/{id}")
    public ResultDto<DesignerDetailDTO> getDesignTimeDesignerDetail(@NotBlank @PathVariable String id) {
        log.debug("收到获取设计时态设计器详情请求: {}", id);
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }
        return ResultDto.createSuccess(designerService.findDetailById(id));
    }

    /**
     * 发布
     */
    @GetMapping("/{id}/publish")
    public ResultDto<Void> publishDesigner(@NotBlank @PathVariable String id) {
        log.info("收到发布设计器请求: {}", id);
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }
        designerService.publish(id);
        return ResultDto.createSuccess(null);
    }

    /**
     * 下架
     */
    @GetMapping("/{id}/offline")
    public ResultDto<Void> offlineDesigner(@NotBlank @PathVariable String id) {
        log.info("收到下架设计器请求: {}", id);
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }
        designerService.takeDown(id);
        return ResultDto.createSuccess(null);
    }

    /**
     * 上架
     */
    @GetMapping("/{id}/online")
    public ResultDto<Void> onlineDesigner(@NotBlank @PathVariable String id) {
        log.info("收到上架设计器请求: {}", id);
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }
        designerService.launch(id);
        return ResultDto.createSuccess(null);
    }

    /**
     * 验证设计器唯一性
     */
    @GetMapping("/validate/unique")
    public ResultDto<Boolean> validateDesignerCodeUnique(@RequestParam(required = false) String code,
                                                         @RequestParam(required = false) String name,
                                                         @RequestParam(required = false) String excludeObjectId) {
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }
        log.debug("收到验证设计器编码唯一性请求: {}", code);
        boolean isUnique = designerService.validateUnique(code, name, excludeObjectId);
        return ResultDto.createSuccess(isUnique);
    }

}