package com.digiwin.athena.dao.mongodao.assembly.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.bo.assembly.Designer;
import com.digiwin.athena.convertor.assembly.DesignerConvertor;
import com.digiwin.athena.dao.mongodao.assembly.DDesignerMongoDao;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.digiwin.athena.dto.assembly.designer.DesignerListDTO;
import com.digiwin.athena.dto.assembly.designer.DesignerPageQo;
import com.digiwin.athena.mongodb.domain.assembly.DesignerEntity;
import com.digiwin.athena.mongodb.repository.MongoPrimaryRepositoryDecorator;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 设计时态设计器DAO实现
 */
@Slf4j
@Repository
@RequiredArgsConstructor
public class DDesignerMongoDaoImpl implements DDesignerMongoDao {

    private static final DesignerConvertor CONVERTOR = DesignerConvertor.INSTANCE;
    private final MongoPrimaryRepositoryDecorator decorator;

    @Override
    public Designer selectById(String objectId) {
        DesignerEntity entity = decorator.findById(objectId, DesignerEntity.class);
        return CONVERTOR.toBO(entity);
    }

    @Override
    public Designer selectByCode(String code) {
        Query query = Query.query(Criteria.where(DesignerEntity.Fields.type).is(code));
        DesignerEntity entity = decorator.findOne(query, DesignerEntity.class);
        return CONVERTOR.toBO(entity);
    }

    @Override
    public Designer selectByName(String name) {
        Query query = Query.query(Criteria.where(DesignerEntity.Fields.name).is(name));
        DesignerEntity entity = decorator.findOne(query, DesignerEntity.class);
        return CONVERTOR.toBO(entity);
    }

    @Override
    public void save(Designer designer) {
        DesignerEntity entity = CONVERTOR.toPo(designer);
        entity.prepareSave();

        decorator.save(entity);

        designer.setId(entity.getId());
    }

    @Override
    public boolean deleteById(String objectId) {
        Query query = Query.query(Criteria.where(DesignerEntity.Fields.id).is(objectId));
        long count = decorator.remove(query, DesignerEntity.class);
        return count > 0;
    }

    @Override
    public boolean existsByKey(String code, String name, String excludeId) {
        Assert.isFalse(StrUtil.isAllBlank(code, name), "code or name can not be null");

        Query query = new Query();
        if (StrUtil.isNotBlank(code)) {
            query.addCriteria(Criteria.where(DesignerEntity.Fields.type).is(code));
        }

        if (StrUtil.isNotBlank(name)) {
            query.addCriteria(Criteria.where(DesignerEntity.Fields.name).is(name));
        }


        if (excludeId != null) {
            query.addCriteria(Criteria.where(DesignerEntity.Fields.id).ne(excludeId));
        }

        return decorator.count(query, DesignerEntity.class) > 0;
    }

    @Override
    public Pagination<DesignerListDTO> selectPage(PageReqCondition<DesignerPageQo> pageParam) {
        Query query = new Query();

        buildPageCondition(pageParam, query);
        return decorator
                .findWithPage(query, DesignerEntity.class, pageParam)
                .convert(DesignerConvertor.INSTANCE::toListDTO);
    }

    @Override
    public List<Designer> selectByRelatedAsset(String assetTypeId) {
        Criteria criteria = new Criteria().orOperator(Criteria.where("relatedAssetType.read.objId").is(assetTypeId),
                Criteria.where("relatedAssetType.write.objId").is(assetTypeId));
        return decorator.find(new Query(criteria),DesignerEntity.class).stream().map(CONVERTOR::toBO).collect(Collectors.toList());
    }

    @Override
    public List<Designer> selectByRelatedAsset(Collection<String> assetTypeIds) {
        Criteria criteria = new Criteria().orOperator(Criteria.where("relatedAssetType.read.objId").in(assetTypeIds),
                Criteria.where("relatedAssetType.write.objId").in(assetTypeIds));
        return decorator.find(new Query(criteria),DesignerEntity.class).stream().map(CONVERTOR::toBO).collect(Collectors.toList());
    }

    protected void buildPageCondition(PageReqCondition<DesignerPageQo> pageParam, Query query) {
        DesignerPageQo condition = ObjectUtil.defaultIfNull(pageParam.getCondition(), DesignerPageQo::new);
        DDesignerMongoDao.addSearchNameOrType(query, condition.getSearchContent());

        if (CollUtil.isNotEmpty(condition.getIntegrationTypeList())) {
            query.addCriteria(Criteria.where(DesignerEntity.Fields.integrationType).in(condition.getIntegrationTypeList()));
        }

        if (CollUtil.isNotEmpty(condition.getStatusList())) {
            query.addCriteria(Criteria.where(DesignerEntity.Fields.status).in(condition.getStatusList()));
        }
    }
}
