package com.digiwin.athena.controller.assembly;

import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.base.ErrorCodeConstants;
import com.digiwin.athena.bo.assembly.SolutionPlan;
import com.digiwin.athena.convertor.assembly.SolutionConvertor;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.digiwin.athena.dto.ResultDto;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanDetailDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanListDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanPageQo;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanSaveDTO;
import com.digiwin.athena.service.ManageBackgroundUserService;
import com.digiwin.athena.service.assembly.impl.DSolutionPlanServiceImpl;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotBlank;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

/**
 * 解决方案管理
 */
@Slf4j
@RestController
@RequestMapping("/assembly/solutionPlan/d/")
@RequiredArgsConstructor
@Validated
public class DSolutionPlanController {

    private final DSolutionPlanServiceImpl solutionService;
    private final ManageBackgroundUserService manageBackgroundUserService;


    /**
     * 列表
     */
    @PostMapping("/list")
    public ResultDto<Pagination<SolutionPlanListDTO>> list(@RequestBody PageReqCondition<SolutionPlanPageQo> pageReqCondition) {
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }

        // TODO list get请求转义字典及枚举多语言
        Pagination<SolutionPlanListDTO> res = solutionService.findList(pageReqCondition);
        return ResultDto.createSuccess(res);
    }


    /**
     * 保存
     */
    @PostMapping("/save")
    public ResultDto<SolutionPlanDetailDTO> save(@Valid @RequestBody SolutionPlanSaveDTO param) {
        log.info("收到创建设计时态设计器请求: {}", param);
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }

        SolutionPlan bo = solutionService.save(param);
        return ResultDto.createSuccess(SolutionConvertor.INSTANCE.toDto(bo));
    }

    /**
     * 删除
     */
    @GetMapping("/{id}/delete")
    public ResultDto<Void> deleteDesignTimeAssemblyApplicationConfig(@NotBlank @PathVariable String id) {
        log.info("收到删除设计时态设计器请求: {}", id);
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }
        solutionService.deleteById(id);
        return ResultDto.createSuccess(null);
    }

    /**
     * 详情
     */
    @GetMapping("/{id}")
    public ResultDto<SolutionPlanDetailDTO> getDesignTimeAssemblyApplicationConfigDetail(@NotBlank @PathVariable String id) {
        log.debug("收到获取设计时态设计器详情请求: {}", id);
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }
        return ResultDto.createSuccess(solutionService.findDetailById(id));
    }

    /**
     * 发布
     */
    @GetMapping("/{id}/publish")
    public ResultDto<Void> publishAssemblyApplicationConfig(@NotBlank @PathVariable String id) {
        log.info("收到发布设计器请求: {}", id);
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }
        solutionService.publish(id);
        return ResultDto.createSuccess(null);
    }

    /**
     * 下架
     */
    @GetMapping("/{id}/offline")
    public ResultDto<Void> offlineAssemblyApplicationConfig(@NotBlank @PathVariable String id) {
        log.info("收到下架设计器请求: {}", id);
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }
        solutionService.takeDown(id);
        return ResultDto.createSuccess(null);
    }

    /**
     * 上架
     */
    @GetMapping("/{id}/online")
    public ResultDto<Void> onlineAssemblyApplicationConfig(@NotBlank @PathVariable String id) {
        log.info("收到上架设计器请求: {}", id);
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }
        solutionService.launch(id);
        return ResultDto.createSuccess(null);
    }

    /**
     * 验证解决方案唯一性
     */
    @GetMapping("/validate/unique")
    public ResultDto<Boolean> validateAssemblyApplicationConfigCodeUnique(@RequestParam(required = false) String code,
                                                                          @RequestParam(required = false) String name,
                                                                          @RequestParam(required = false) String excludeObjectId) {
        if (!manageBackgroundUserService.isManager()) {
            throw new BusinessException(ErrorCodeConstants.Backend_Administrator_EXCEPTION, new Object[]{});
        }
        log.debug("收到验证设计器编码唯一性请求: {}", code);
        boolean isUnique = solutionService.validateUnique(code, name, excludeObjectId);
        return ResultDto.createSuccess(isUnique);
    }
}