package com.digiwin.athena.service.assembly.impl;

import cn.hutool.core.map.MapUtil;
import cn.hutool.http.HttpUtil;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.config.AssemblyProp;
import com.digiwin.athena.constant.GmcConstant;
import com.digiwin.athena.convertor.BaseConverter;
import com.digiwin.athena.dto.assembly.application.IamCreateApplicationResponse;
import com.digiwin.athena.dto.assembly.goods.CreateCommodityRequest;
import com.digiwin.athena.dto.assembly.goods.UpdateModulesRequest;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;

/**
 * GMC商品服务
 */
@Service
@RequiredArgsConstructor
public class GmcCommodityService {

    private final AssemblyProp assemblyProp;

    /**
     * 创建商品
     *
     * @param request 创建商品请求
     * @param appInfo 各区域应用信息
     * @see com.digiwin.athena.service.application.impl.ApplicationServiceImpl#createCommodity
     */
    public void createCommodity(CreateCommodityRequest request,
                                List<IamCreateApplicationResponse> appInfo) {

        for (IamCreateApplicationResponse response : appInfo) {
            AssemblyProp.EnvConfig envConfig = response.getEnvConfig();
            String iamToken = response.getToken();

            String gmcDomain = envConfig.getGmcDomain();
            String url = gmcDomain + GmcConstant.createCommodity;

            String responseBody = HttpUtil.createPost(url)
                    .body(BaseConverter.INSTANCE.serialize(request))
                    .header("digi-middleware-auth-user", iamToken)
                    .header("digi-middleware-auth-app", assemblyProp.getAppToken())
                    .execute()
                    .body();

            //noinspection unchecked
            Map<String, Object> responseObj = BaseConverter.INSTANCE.deserialize(responseBody, Map.class);
            Boolean success = MapUtil.getBool(responseObj, "success");

            if (!success) {
                throw new BusinessException("GMC failed to create the product, error message:"
                        + MapUtil.getStr(responseObj, "errorMessage"));
            }
        }
    }

    /**
     * 更新商品模组
     *
     * @param request 更新模组请求
     * @param appInfo 各区域应用信息
     * @see com.digiwin.athena.service.application.impl.ApplicationServiceImpl#updateCommodityModule
     */
    public void updateModules(UpdateModulesRequest request,
                              List<IamCreateApplicationResponse> appInfo) {

        for (IamCreateApplicationResponse response : appInfo) {
            AssemblyProp.EnvConfig envConfig = response.getEnvConfig();
            String token = response.getToken();

            String gmcUrl = envConfig.getGmcDomain() + GmcConstant.UPDATE_MODULES;

            try {
                String responseBody = HttpUtil.createPost(gmcUrl)
                        .header("digi-middleware-auth-user", token)
                        .header("digi-middleware-auth-app", assemblyProp.getAppToken())
                        .body(BaseConverter.INSTANCE.serialize(request))
                        .execute()
                        .body();

                //noinspection unchecked
                Map<String, Object> responseObj = BaseConverter.INSTANCE.deserialize(responseBody, Map.class);
                Integer code = MapUtil.getInt(responseObj, "code");

                if (code == null || code != 200) {
                    throw new BusinessException("更新商品模组失败!");
                }
            } catch (Exception e) {
                throw new BusinessException("更新商品模组失败!");
            }
        }
    }
}
