package com.digiwin.athena.service.assembly.impl;

import cn.hutool.core.map.MapUtil;
import cn.hutool.http.HttpUtil;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.config.AssemblyProp;
import com.digiwin.athena.constant.IamConstant;
import com.digiwin.athena.convertor.BaseConverter;
import com.digiwin.athena.dto.assembly.application.IamCreateApplicationRequest;
import com.digiwin.athena.dto.assembly.application.IamCreateApplicationResponse;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * IAM应用服务
 */
@Service
@RequiredArgsConstructor
public class IamApplicationService {

    private final AssemblyProp assemblyProp;

    /**
     * 创建应用（支持多区域）
     *
     * @param request  IAM创建应用请求
     * @param iamToken IAM用户token
     * @return 各区域创建应用响应列表
     * @see com.digiwin.athena.service.application.impl.ApplicationServiceImpl#iamCreateApplication
     */
    public List<IamCreateApplicationResponse> createApplication(IamCreateApplicationRequest request, String iamToken) {

        List<IamCreateApplicationResponse> responseList = new ArrayList<>();

        String url = assemblyProp.getIamDomain() + IamConstant.createApplication;

        String response = HttpUtil.createPost(url)
                .body(BaseConverter.INSTANCE.serialize(request))
                .header("digi-middleware-auth-user", iamToken)
                .header("digi-middleware-auth-app", assemblyProp.getAppToken())
                .execute()
                .body();

        //noinspection unchecked
        Map<String, Object> responseObj = BaseConverter.INSTANCE.deserialize(response, Map.class);
        Boolean success = MapUtil.getBool(responseObj, "success");

        Long appSid;
        if (success) {
            appSid = MapUtil.getLong(responseObj, "data");
            IamCreateApplicationResponse appResponse = new IamCreateApplicationResponse();
            appResponse.setAppSid(appSid);

            AssemblyProp.EnvConfig envConfig = new AssemblyProp.EnvConfig();
            envConfig.setIamDomain(assemblyProp.getIamDomain());
            envConfig.setGmcDomain(assemblyProp.getGmcDomain());
            appResponse.setEnvConfig(envConfig);
            appResponse.setToken(iamToken);
            responseList.add(appResponse);
        } else {
            throw new BusinessException("IAM failed to create the app, error message:"
                    + MapUtil.getStr(responseObj, "errorMessage"));
        }

        return responseList;
    }

    /**
     * 查询应用token
     *
     * @param appId 应用ID
     * @param iamToken IAM用户token
     * @return 应用token
     */
    public String queryAppToken(String appId, String iamToken) {
        String url = assemblyProp.getIamDomain() + IamConstant.queryAppToken2;
        url = url.replace("{id}", appId);

        String response = HttpUtil.createGet(url)
                .header("digi-middleware-auth-user", iamToken)
                .header("digi-middleware-auth-app", assemblyProp.getAppToken())
                .execute()
                .body();

        //noinspection unchecked
        Map<String, Object> responseObj = BaseConverter.INSTANCE.deserialize(response, Map.class);
        Boolean success = MapUtil.getBool(responseObj, "success");

        if (success) {
            @SuppressWarnings("unchecked")
            Map<String, Object> data = (Map<String, Object>) responseObj.get("data");
            if (data != null) {
                return MapUtil.getStr(data, "appToken");
            }
        }

        throw new BusinessException("查询应用token失败");
    }

    /**
     * 获取应用信息（支持多区域）
     * 用于launch等场景，不创建应用，只获取各区域的token
     *
     * @param iamToken IAM用户token
     * @return 各区域应用信息列表
     */
    public List<IamCreateApplicationResponse> getApplicationInfo(String iamToken) {
        List<IamCreateApplicationResponse> responseList = new ArrayList<>();

        // 当前区域
        IamCreateApplicationResponse appResponse = new IamCreateApplicationResponse();
        AssemblyProp.EnvConfig envConfig = new AssemblyProp.EnvConfig();
        envConfig.setIamDomain(assemblyProp.getIamDomain());
        envConfig.setGmcDomain(assemblyProp.getGmcDomain());
        appResponse.setEnvConfig(envConfig);
        appResponse.setToken(iamToken);
        responseList.add(appResponse);

        return responseList;
    }
}
