package com.digiwin.athena.service.assembly.impl;

import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.bo.assembly.SolutionPlan;
import com.digiwin.athena.constants.AssemblyErrorCode;
import com.digiwin.athena.convertor.assembly.SolutionConvertor;
import com.digiwin.athena.dao.mongodao.assembly.DSolutionMongoDao;
import com.digiwin.athena.dao.mongodao.assembly.RSolutionMongoDao;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanDetailDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanListDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanPageQo;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanSaveDTO;
import com.digiwin.athena.service.SyncRuntime;
import com.digiwin.athena.service.assembly.DSolutionPlanService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * 解决方案管理服务实现
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class DSolutionPlanServiceImpl extends SyncRuntime<SolutionPlan> implements DSolutionPlanService {

    private final DSolutionMongoDao dSolutionMongoDao;
    private final RSolutionMongoDao rSolutionMongoDao;
    private DSolutionPlanServiceImpl proxyThis;


    @Override
    public Pagination<SolutionPlanListDTO> findList(PageReqCondition<SolutionPlanPageQo> pageReqCondition) {
        // TODO pzz 查找设计器最新数据
        return dSolutionMongoDao.selectPage(pageReqCondition);
    }

    @Override
    public Map<String, Set<SolutionPlanListDTO>> findListByDesignerId(Iterable<String> designerIdColl) {
        return dSolutionMongoDao.findListByDesignerId(designerIdColl);
    }

    @Transactional(rollbackFor = Throwable.class)
    @Override
    public SolutionPlan save(SolutionPlanSaveDTO designer) {
        SolutionPlan bo = SolutionConvertor.INSTANCE.toBo(designer);

        save(bo);

        return bo;
    }

    @Transactional(rollbackFor = Throwable.class)
    @Override
    public void save(SolutionPlan updated) {
        log.info("开始创建设计时态解决方案: id:{},updatedName:{}", updated.getId(), updated.getName());

        // 验证唯一性
        checkUnique(updated);

        String objectId = updated.getId();
        if (objectId != null) {
            SolutionPlan existed = findExist(objectId);

            updated.removeUnmodifiable(existed.getStatus());
            SolutionConvertor.INSTANCE.merge(updated, existed);

            updated = existed;
        } else {
            updated.prepareNew();
        }

        dSolutionMongoDao.save(updated);

        log.info("成功创建设计时态解决方案: {}", updated.getId());
    }

    /**
     * 单纯保存，不做任何业务处理
     */
    protected void pureSave(SolutionPlan bo) {
        dSolutionMongoDao.save(bo);
    }

    protected void checkUnique(SolutionPlan bo) {
        if (!validateUnique(bo.getName(), null, bo.getId())) {
            throw new BusinessException(AssemblyErrorCode.NAME_CODE_NOT_UNIQUE, "编码不唯一！");
        }

        if (!validateUnique(null, bo.getName(), bo.getId())) {
            throw new BusinessException(AssemblyErrorCode.NAME_CODE_NOT_UNIQUE, "名称不唯一！");
        }
    }

    protected SolutionPlan findExist(String objectId) {
        return Optional
                .ofNullable(dSolutionMongoDao.selectById(objectId))
                .orElseThrow(() -> new BusinessException(AssemblyErrorCode.SOLUTION_NOT_EXIST, "解决方案不存在！"));
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void deleteById(String objectId) {
        log.info("开始删除设计时态解决方案: {}", objectId);

        // 检查解决方案是否存在
        SolutionPlan bo = findExist(objectId);
        bo.checkDelete(bo);

        // 删除
        dSolutionMongoDao.deleteById(objectId);
        log.info("成功删除设计时态解决方案: {}", objectId);
    }


    @Override
    public SolutionPlanDetailDTO findDetailById(String objectId) {
        // TODO pzz 查找设计器最新数据
        return SolutionConvertor.INSTANCE.toDto(findById(objectId));
    }

    @Override
    public SolutionPlan findById(String objectId) {
        log.debug("查询设计时态解决方案详情: {}", objectId);
        return dSolutionMongoDao.selectById(objectId);
    }


    @Override
    public boolean validateUnique(String code, String name, String excludeObjectId) {
        return !dSolutionMongoDao.existsByKey(code, name, excludeObjectId);
    }

    @Override
    protected void saveDesignerTimeData(SolutionPlan syncData) {
        pureSave(syncData);
    }

    @Override
    protected void insertRunTimeData(SolutionPlan syncData) {
        rSolutionMongoDao.save(syncData);
    }

    @Override
    protected void saveRunTimeData(SolutionPlan syncData) {
        rSolutionMongoDao.save(syncData);
    }

    @Override
    protected SolutionPlan getDesignerTimeData(String objId) {
        return proxyThis.findById(objId);
    }

    @Override
    protected SolutionPlan getRunTimeData(String objId) {
        return rSolutionMongoDao.selectById(objId);
    }

    @Override
    protected void deleteRuntimeData(String objId) {
        rSolutionMongoDao.deleteById(objId);
    }

    @Autowired
    @Lazy
    public void setProxyThis(DSolutionPlanServiceImpl proxyThis) {
        this.proxyThis = proxyThis;
    }

}