package com.digiwin.athena.datacollect.expression;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.env.Environment;
import org.springframework.expression.Expression;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * Spring EL表达式解析器
 * 支持从配置文件中读取值，替换${...}格式的表达式
 */
@Slf4j
@Component
public class ExpressionResolver {

    @Autowired
    private Environment environment;

    private final ExpressionParser parser = new SpelExpressionParser();

    /**
     * 解析字符串中的表达式
     * 支持格式: ${property.name} 或直接的SpEL表达式
     *
     * @param text 包含表达式的文本
     * @return 解析后的文本
     */
    public String resolve(String text) {
        if (text == null || text.isEmpty()) {
            return text;
        }

        try {
            // 创建SpEL上下文
            StandardEvaluationContext context = createEvaluationContext();

            // 解析表达式
            Expression exp = parser.parseExpression(text);
            Object value = exp.getValue(context);

            return value != null ? value.toString() : text;
        } catch (Exception e) {
            // 如果不是表达式，尝试从environment直接获取
            String value = environment.getProperty(text);
            if (value != null) {
                return value;
            }

            log.debug("无法解析表达式: {}, 保持原值", text);
            return text;
        }
    }

    /**
     * 解析Map中的表达式
     *
     * @param map 包含表达式的Map
     * @return 解析后的Map
     */
    public Map<String, Object> resolve(Map<String, Object> map) {
        if (map == null || map.isEmpty()) {
            return map;
        }

        Map<String, Object> result = new HashMap<>();
        for (Map.Entry<String, Object> entry : map.entrySet()) {
            String key = entry.getKey();
            Object value = entry.getValue();

            if (value instanceof String) {
                result.put(key, resolve((String) value));
            } else if (value instanceof Map) {
                @SuppressWarnings("unchecked")
                Map<String, Object> nestedMap = (Map<String, Object>) value;
                result.put(key, resolve(nestedMap));
            } else if (value instanceof List) {
                result.put(key, resolveList((List<?>) value));
            } else {
                result.put(key, value);
            }
        }

        return result;
    }

    /**
     * 解析List中的表达式
     *
     * @param list 包含表达式的List
     * @return 解析后的List
     */
    public List<Object> resolveList(List<?> list) {
        if (list == null || list.isEmpty()) {
            return (List<Object>) list;
        }

        return list.stream()
                .map(item -> {
                    if (item instanceof String) {
                        return resolve((String) item);
                    } else if (item instanceof Map) {
                        @SuppressWarnings("unchecked")
                        Map<String, Object> map = (Map<String, Object>) item;
                        return resolve(map);
                    } else {
                        return item;
                    }
                })
                .collect(Collectors.toList());
    }

    /**
     * 创建SpEL评估上下文
     * 将Environment作为变量注入，支持 #{env['property.name']} 格式
     */
    private StandardEvaluationContext createEvaluationContext() {
        StandardEvaluationContext context = new StandardEvaluationContext();

        // 注入Environment，支持 #{env['property.name']} 访问配置
        context.setVariable("env", new EnvironmentAccessor(environment));

        return context;
    }

    /**
     * Environment访问器
     * 用于在SpEL中访问配置属性
     */
    private static class EnvironmentAccessor {
        private final Environment environment;

        public EnvironmentAccessor(Environment environment) {
            this.environment = environment;
        }

        public String get(String key) {
            return environment.getProperty(key);
        }

        public String getOrDefault(String key, String defaultValue) {
            return environment.getProperty(key, defaultValue);
        }
    }
}
