package com.digiwin.athena.bo.assembly;

import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.constants.AssemblyErrorCode;
import com.digiwin.athena.constants.enums.assembly.IntegrationType;
import com.digiwin.athena.constants.enums.assembly.OnlineState;
import com.digiwin.athena.dto.action.MultiLanguageDTO;
import com.digiwin.athena.dto.assembly.designer.DesignerDetailDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanBaseDTO;
import com.digiwin.athena.mongodb.domain.PublishAble;
import lombok.AccessLevel;
import lombok.Data;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import java.time.LocalDateTime;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

/**
 * 设计器
 */
@Slf4j
@Data
public class Designer implements PublishAble, IVersion {

    /**
     * type字段正则：只允许大写字母和下划线
     */
    private static final Pattern TYPE_PATTERN = Pattern.compile("^[A-Z_]+$");

    /**
     * 主键ID
     */
    private String id;

    /**
     * 设计器编码，全局唯一
     */
    private String type;

    /**
     * 归属方
     */
    private String owner;

    /**
     * 设计器名称
     */
    private String name;

    /**
     * 版本号，service层暂时写死1.0
     */
    private String version = "1.0";

    /**
     * 版本排序，version根据`.`分割，{第一部分}*1000+{第二部分}
     */
    @Getter(AccessLevel.NONE)
    private Integer versionSort;

    /**
     * 状态
     *
     * @see OnlineState
     */
    private String status;

    /**
     * 描述
     */
    private String description;

    /**
     * 关联的资产类型
     */
    private DesignerDetailDTO.RelatedAssetTypeDTO relatedAssetType;

    /**
     * 关联解决方案
     */
    private List<SolutionPlanBaseDTO> relatedSolutionPlan;

    /**
     * API列表
     */
    private List<DesignerDetailDTO.ApiExecInfoDTO> apiList;

    /**
     * 集成方式：MF/microApp/Iframe
     *
     * @see IntegrationType#getCode()
     */
    private String integrationType;

    /**
     * 集成配置
     */
    private DesignerDetailDTO.IntegrationConfigDTO integrationConfig;

    /**
     * 多语言配置
     */
    private Map<String, MultiLanguageDTO<String>> lang;

    /**
     * 发布时间
     */
    private Date publishTime;

    /**
     * 发布人
     */
    private String publishBy;

    /**
     * 创建时间
     */
    private LocalDateTime createTime;

    /**
     * 更新时间
     */
    private LocalDateTime editTime;

    /**
     * 创建人
     */
    private String createBy;

    /**
     * 更新人
     */
    private String editBy;

    public void removeUnmodifiable(String status) {
        // 只有草稿态可以修改名称和编码
        if (!OnlineState.Draft.getCode().equals(status)) {
            setName(null);
            setType(null);
        }

        setCreateBy(null);
        setCreateTime(null);
    }

    public void prepareNew() {
        setStatus(OnlineState.Draft.getCode());
        setPublishTime(null);
        setPublishBy(null);
    }

    public void checkUpdate() {
        if (!OnlineState.Draft.getCode().equals(getStatus())) {
            throw new BusinessException(AssemblyErrorCode.NOT_DRAFT, "设计器状态不为草稿！", getName(), getStatus());
        }
    }

    public void checkDelete() {
        if (!OnlineState.Draft.getCode().equals(getStatus())) {
            throw new BusinessException(AssemblyErrorCode.NOT_DRAFT, "设计器状态不为草稿！", getName(), getStatus());
        }
    }

    /**
     * 校验type字段格式：只允许大写字母和下划线
     *
     * @throws BusinessException 如果type字段格式不符合要求
     */
    public void validateTypeFormat() {
        if (StrUtil.isBlank(type)) {
            throw new BusinessException(AssemblyErrorCode.INVALID_PARAM, "设计器编码不能为空！");
        }

        if (!TYPE_PATTERN.matcher(type).matches()) {
            throw new BusinessException(AssemblyErrorCode.INVALID_PARAM,
                    "设计器编码格式不正确！只允许大写字母和下划线，当前编码: " + type);
        }
    }
}
