package com.digiwin.athena.dao.mongodao.assembly.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.bo.assembly.SolutionPlan;
import com.digiwin.athena.convertor.assembly.SolutionConvertor;
import com.digiwin.athena.dao.mongodao.assembly.DDesignerMongoDao;
import com.digiwin.athena.dao.mongodao.assembly.DSolutionMongoDao;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanDetailDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanListDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanPageQo;
import com.digiwin.athena.mongodb.domain.assembly.AssemblyApplicationConfigEntity;
import com.digiwin.athena.mongodb.repository.MongoPrimaryRepositoryDecorator;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.mapping.FieldName;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;

import java.util.*;

/**
 * 设计时态解决方案DAO实现
 */
@Slf4j
@Repository
@RequiredArgsConstructor
public class DSolutionMongoDaoImpl implements DSolutionMongoDao {

    private static final SolutionConvertor CONVERTOR = SolutionConvertor.INSTANCE;
    private final MongoPrimaryRepositoryDecorator decorator;


    @Override
    public SolutionPlan selectById(String objectId) {
        AssemblyApplicationConfigEntity entity = decorator.findById(objectId, AssemblyApplicationConfigEntity.class);
        return CONVERTOR.toBO(entity);
    }

    @Override
    public SolutionPlan selectByCode(String code) {
        Query query = Query.query(Criteria.where(AssemblyApplicationConfigEntity.Fields.type).is(code));
        AssemblyApplicationConfigEntity entity = decorator.findOne(query, AssemblyApplicationConfigEntity.class);
        return CONVERTOR.toBO(entity);
    }

    @Override
    public SolutionPlan selectByName(String name) {
        Query query = Query.query(Criteria.where(AssemblyApplicationConfigEntity.Fields.name).is(name));
        AssemblyApplicationConfigEntity entity = decorator.findOne(query, AssemblyApplicationConfigEntity.class);
        return CONVERTOR.toBO(entity);
    }

    @Override
    public void save(SolutionPlan solution) {
        AssemblyApplicationConfigEntity entity = CONVERTOR.toEntity(solution);
        entity.prepareSave();

        Map<String, Object> mergeJson = solution.getMergeJson();
        // 如果mergeJson不为空，则将entity转为map，并合并mergeJson中的数据
        if (CollUtil.isNotEmpty(mergeJson)) {
            String collectionName = decorator.getCollectionNameFromDocumentAnnotedClass(AssemblyApplicationConfigEntity.class);
            Map<String, Object> entityMap = entity.mergeToEntityMap();

            decorator.save(entityMap, collectionName);
            solution.setId(MapUtil.getStr(entityMap, FieldName.ID.name()));
        } else {
            // 如果没有mergeJson，使用原有的保存方式
            decorator.save(entity);
            solution.setId(entity.getId());
        }
    }

    /**
     * 获取下一个appType值（当前集合中最大的appType+1）
     *
     * @return 下一个appType值，如果集合为空则返回1
     */
    @Override
    public Integer getNextAppType() {
        Query query = new Query();
        query.with(Sort.by(Sort.Direction.DESC, AssemblyApplicationConfigEntity.Fields.appType));
        query.limit(1);

        AssemblyApplicationConfigEntity maxEntity = decorator.findOne(query, AssemblyApplicationConfigEntity.class);

        if (maxEntity == null || maxEntity.getAppType() == null) {
            return 1;
        }

        return maxEntity.getAppType() + 1;
    }

    @Override
    public boolean deleteById(String objectId) {
        Query query = Query.query(Criteria.where(AssemblyApplicationConfigEntity.Fields.id).is(objectId));
        long count = decorator.remove(query, AssemblyApplicationConfigEntity.class);
        return count > 0;
    }

    @Override
    public boolean existsByKey(String code, String name, String excludeId) {
        Assert.isFalse(StrUtil.isAllBlank(code, name), "code or name can not be null");

        Query query = new Query();
        if (StrUtil.isNotBlank(code)) {
            query.addCriteria(Criteria.where(AssemblyApplicationConfigEntity.Fields.type).is(code));
        }

        if (StrUtil.isNotBlank(name)) {
            query.addCriteria(Criteria.where(AssemblyApplicationConfigEntity.Fields.name).is(name));
        }

        if (excludeId != null) {
            query.addCriteria(Criteria.where(AssemblyApplicationConfigEntity.Fields.id).ne(excludeId));
        }

        return decorator.count(query, AssemblyApplicationConfigEntity.class) > 0;
    }

    @Override
    public Pagination<SolutionPlanListDTO> selectPage(PageReqCondition<SolutionPlanPageQo> pageParam) {
        Query query = new Query();

        buildPageCondition(pageParam, query);

        return decorator
                .findWithPage(query, AssemblyApplicationConfigEntity.class, pageParam)
                .convert(SolutionConvertor.INSTANCE::entityToListDTO);
    }

    protected void buildPageCondition(PageReqCondition<SolutionPlanPageQo> pageParam, Query query) {
        SolutionPlanPageQo condition = ObjectUtil.defaultIfNull(pageParam.getCondition(), SolutionPlanPageQo::new);
        DDesignerMongoDao.addSearchNameOrType(query, condition.getSearchContent());

        if (CollUtil.isNotEmpty(condition.getAssemblyTypeList())) {
            query.addCriteria(Criteria
                    .where(AssemblyApplicationConfigEntity.Fields.assemblyType).in(condition.getAssemblyTypeList()));
        }

        if (CollUtil.isNotEmpty(condition.getSolutionStateList())) {
            query.addCriteria(Criteria
                    .where(AssemblyApplicationConfigEntity.Fields.status).in(condition.getSolutionStateList()));
        }
    }

    @Override
    public Map<String, Set<SolutionPlanListDTO>> findListByDesignerId(Iterable<String> designerIdColl) {
        Set<String> designerIdSet = new LinkedHashSet<>();
        designerIdColl.forEach(designerIdSet::add);

        // 查询包含指定设计器的解决方案
        Query query = Query.query(Criteria.where("designerList.id").in(designerIdSet));
        List<AssemblyApplicationConfigEntity> solutionList = decorator.find(query, AssemblyApplicationConfigEntity.class);


        return groupByDesignerId(solutionList, designerIdSet);
    }

    protected Map<String, Set<SolutionPlanListDTO>> groupByDesignerId(List<AssemblyApplicationConfigEntity> solutionList,
                                                                      Set<String> designerIdSet) {
        Map<String, Set<AssemblyApplicationConfigEntity>> designerSolutionMap = MapUtil.newHashMap(designerIdSet.size());
        for (AssemblyApplicationConfigEntity solution : solutionList) {
            // 遍历解决方案中的设计器列表，将每个设计器ID映射到解决方案DTO
            if (solution.getDesignerList() == null) {
                continue;
            }

            for (SolutionPlanDetailDTO.DesignerBaseInfoDTO designerInfo : solution.getDesignerList()) {
                if (designerIdSet.contains(designerInfo.getId())) {
                    designerSolutionMap
                            .computeIfAbsent(designerInfo.getId(), k -> new HashSet<>())
                            .add(solution);
                }
            }
        }

        return MapUtil.map(designerSolutionMap,
                (k, v) -> SolutionConvertor.INSTANCE.entitySetToListDTO(v));
    }
}
