package com.digiwin.athena.datacollect.collector.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.datacollect.api.dto.AssetCollectApiRequest;
import com.digiwin.athena.datacollect.api.dto.AssetCollectApiResponse;
import com.digiwin.athena.datacollect.collector.BaseAssetDataCollector;
import com.digiwin.athena.datacollect.context.CollectContext;
import com.digiwin.athena.datacollect.context.HttpScriptContext;
import com.digiwin.athena.datacollect.model.CollectExecutorProp;
import com.digiwin.athena.datacollect.model.PageResult;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.retry.RetryCallback;
import org.springframework.retry.backoff.ExponentialBackOffPolicy;
import org.springframework.retry.backoff.FixedBackOffPolicy;
import org.springframework.retry.policy.SimpleRetryPolicy;
import org.springframework.retry.support.RetryTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;

import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * TODO pzz 校验脚本执行结果，返回结果不能为null
 * TODO pzz refactor 解耦HttpCollector和Asset的关系，使用泛型或接口方式增强Collector的泛用性
 * TODO pzz refactor 使用工厂模式将组件与逻辑实现抽离，创建内部类，封装fetchPage逻辑，减少临时变量增强可读性
 * HTTP方式的资产数据采集器实现
 * 只负责HTTP请求的组装和发送，分页循环由父类控制
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class HttpAssetDataCollector extends BaseAssetDataCollector {

    protected static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    protected final RestTemplate restTemplate;

    @Override
    public String getType() {
        return "http";
    }

    /**
     * 分页查询（只查询一页）
     */
    @Override
    protected PageResult<AssetCollectApiResponse.AssetDataItem> fetchPage(CollectContext context) {
        CollectExecutorProp executor = context.getConfig().getExecutor();
        String url = executor.getUrl();

        try {
            // 1. 构建HTTP Headers
            Map<String, Object> headerMap = new HashMap<>(MapUtil.emptyIfNull(executor.getHeaders()));

            // 2. 构建请求体
            AssetCollectApiRequest request = buildRequest(context);

            // 3. 执行请求前置脚本处理
            RequestScriptResult scriptResult = processRequestScript(executor, context, url, request, headerMap);
            request = scriptResult.getRequest();
            headerMap = scriptResult.getHeaderMap();

            // 4. 构建最终的HTTP Headers
            HttpHeaders headers = buildHeaders(headerMap);
            HttpEntity<AssetCollectApiRequest> entity = new HttpEntity<>(request, headers);

            // 5. 发送HTTP请求（使用动态创建的RetryTemplate）
            RetryTemplate dynamicRetryTemplate = createRetryTemplate(executor);
            ResponseEntity<String> response = dynamicRetryTemplate.execute(request(url, entity));

            String responseBodyStr = response.getBody();
            if (log.isDebugEnabled()) {
                log.debug("HTTP请求响应： url:{},status:{},body:{}", response.getStatusCode(), url, responseBodyStr);
            }

            // 6. 解析响应
            if (response.getStatusCode().is2xxSuccessful() && StrUtil.isNotBlank(responseBodyStr)) {
                // 7. 执行响应后置脚本处理并反序列化
                AssetCollectApiResponse responseBody = processResponseScript(
                        executor, context, responseBodyStr, response.getHeaders());

                // 8. 转换为PageResult（返回原始数据项）
                return convertToPageResult(responseBody, context.getPageNo(), context.getPageSize());
            } else {
                log.error("HTTP请求失败: status={}, url={},body:{}", response.getStatusCode(), url, responseBodyStr);
                return new PageResult<>(new ArrayList<>(), context.getPageNo(), context.getPageSize());
            }

        } catch (Exception e) {
            log.error("HTTP数据采集失败: assetType={}, url={}, pageNo={}", context.getAssetType(), url, context.getPageNo(), e);
            throw e;
        }
    }

    protected RetryCallback<ResponseEntity<String>, RuntimeException> request(String url, HttpEntity<AssetCollectApiRequest> entity) {
        return retryContext -> {
            log.debug("发送HTTP请求: url={}, attempt={}", url, retryContext.getRetryCount() + 1);
            return restTemplate.postForEntity(url, entity, String.class);
        };
    }

    /**
     * 处理请求前置脚本（使用HttpScriptContext）
     */
    protected RequestScriptResult processRequestScript(CollectExecutorProp executor, CollectContext context,
                                                       String url, AssetCollectApiRequest request, Map<String, Object> headerMap) {
        if (executor.getRequestScript() != null && !executor.getRequestScript().trim().isEmpty()) {
            // 构建HttpScriptContext
            HttpScriptContext scriptContext = new HttpScriptContext(context);
            scriptContext.setUrl(url);
            scriptContext.setRequestBody(request);
            scriptContext.setRequestHeader(headerMap);

            // 执行脚本
            HttpScriptContext resultContext = scriptExecutor.execute(
                    executor.getRequestScript(),
                    // TODO pzz  scriptExecutor.execute将Map参数改为Object参数，使脚本可以直接访问context的属性
                    BeanUtil.beanToMap(scriptContext),
                    HttpScriptContext.class);

            return new RequestScriptResult(
                    resultContext.getRequestBody(),
                    resultContext.getRequestHeader());
        }
        return new RequestScriptResult(request, headerMap);
    }

    /**
     * 处理响应后置脚本并反序列化（使用HttpScriptContext）
     * 如果存在脚本，则将responseBody反序列化为Map传入脚本执行，脚本执行后再反序列化为目标类型
     * 否则直接使用scriptExecutor.convertResult进行反序列化
     */
    protected AssetCollectApiResponse processResponseScript(CollectExecutorProp executor, CollectContext context,
                                                            String responseBodyStr, HttpHeaders responseHeaders) {
        // 如果存在响应脚本
        if (executor.getResponseScript() != null && !executor.getResponseScript().trim().isEmpty()) {
            // 将响应体反序列化为Map
            //noinspection unchecked
            Map<String, Object> responseBodyMap = scriptExecutor.convertResult(responseBodyStr, Map.class);

            // 转换响应头为Map格式
            Map<String, String> responseHeaderMap = new HashMap<>();
            if (responseHeaders != null) {
                responseHeaders.forEach((key, values) -> {
                    if (values != null && !values.isEmpty()) {
                        responseHeaderMap.put(key, values.get(0));
                    }
                });
            }

            // 构建HttpScriptContext
            HttpScriptContext scriptContext = new HttpScriptContext(context);
            // TODO pzz 传入requestBody和requestHeader
            scriptContext.setResponseBody(responseBodyMap);
            scriptContext.setResponseHeader(responseHeaderMap);

            // 执行脚本
            HttpScriptContext resultContext = scriptExecutor.execute(
                    executor.getResponseScript(),
                    BeanUtil.beanToMap(scriptContext),
                    HttpScriptContext.class);

            if (resultContext != null) {
                // 从context中提取处理后的响应体，并反序列化为目标类型
                return scriptExecutor.convertResult(
                        resultContext.getResponseBody(), AssetCollectApiResponse.class);
            }
        }

        // 无脚本或脚本执行失败，直接反序列化
        return scriptExecutor.convertResult(responseBodyStr, AssetCollectApiResponse.class);
    }

    /**
     * 根据executor配置创建RetryTemplate
     * 支持配置重试次数、间隔和退避策略
     */
    protected RetryTemplate createRetryTemplate(CollectExecutorProp executor) {
        RetryTemplate template = new RetryTemplate();

        // 获取重试配置
        CollectExecutorProp.RetryConfig retryConfig = executor.getRetry();

        if (retryConfig == null || !Boolean.TRUE.equals(retryConfig.getEnabled())) {
            // 未配置重试或禁用重试，使用默认配置（不重试）
            SimpleRetryPolicy retryPolicy = new SimpleRetryPolicy();
            retryPolicy.setMaxAttempts(1);
            template.setRetryPolicy(retryPolicy);
            return template;
        }

        // 配置重试策略
        SimpleRetryPolicy retryPolicy = new SimpleRetryPolicy();
        retryPolicy.setMaxAttempts(retryConfig.getMaxAttempts() != null ? retryConfig.getMaxAttempts() : 3);
        template.setRetryPolicy(retryPolicy);

        // 配置退避策略
        if (retryConfig.getMultiplier() != null && retryConfig.getMultiplier() > 1.0) {
            // 使用指数退避策略
            ExponentialBackOffPolicy backOffPolicy = new ExponentialBackOffPolicy();
            backOffPolicy.setInitialInterval(retryConfig.getInterval() != null ? retryConfig.getInterval() : 2000L);
            backOffPolicy.setMaxInterval(retryConfig.getMaxInterval() != null ? retryConfig.getMaxInterval() : 10000L);
            backOffPolicy.setMultiplier(retryConfig.getMultiplier());
            template.setBackOffPolicy(backOffPolicy);

            log.debug("配置指数退避策略: initialInterval={}ms, maxInterval={}ms, multiplier={}",
                    backOffPolicy.getInitialInterval(),
                    backOffPolicy.getMaxInterval(),
                    backOffPolicy.getMultiplier());
        } else {
            // 使用固定间隔退避策略
            FixedBackOffPolicy backOffPolicy = new FixedBackOffPolicy();
            backOffPolicy.setBackOffPeriod(retryConfig.getInterval() != null ? retryConfig.getInterval() : 2000L);
            template.setBackOffPolicy(backOffPolicy);

            log.debug("配置固定退避策略: interval={}ms", backOffPolicy.getBackOffPeriod());
        }

        return template;
    }

    /**
     * 构建HTTP Headers
     */
    protected HttpHeaders buildHeaders(Map<String, Object> headersConfig) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        if (headersConfig != null && !headersConfig.isEmpty()) {
            for (Map.Entry<String, Object> entry : headersConfig.entrySet()) {
                headers.set(entry.getKey(), String.valueOf(entry.getValue()));
            }
        }

        return headers;
    }

    /**
     * 构建请求参数（使用Context中的参数）
     */
    protected AssetCollectApiRequest buildRequest(CollectContext context) {
        AssetCollectApiRequest request = new AssetCollectApiRequest();

        // 基本参数
        request.setAssetType(context.getAssetType().getType());
        request.setPageNo(context.getPageNo());
        request.setPageSize(context.getPageSize());

        // 时间范围（使用步进时间）
        if (context.getCurrentStepStartTime() != null) {
            request.setStartTime(context.getCurrentStepStartTime().format(DATE_TIME_FORMATTER));
        }
        if (context.getCurrentStepEndTime() != null) {
            request.setEndTime(context.getCurrentStepEndTime().format(DATE_TIME_FORMATTER));
        }

        // loopParam参数
        if (context.getLoopParam() != null && !context.getLoopParam().isEmpty()) {
            // 从loopParam中获取op参数
            Object op = context.getLoopParam().get("op");
            if (op != null) {
                request.setOp(String.valueOf(op));
            }
            // 可以扩展其他参数...
        }

        return request;
    }

    /**
     * 转换API响应为PageResult
     */
    protected PageResult<AssetCollectApiResponse.AssetDataItem> convertToPageResult(AssetCollectApiResponse apiResponse,
                                                                                    Integer pageNo, int pageSize) {
        if (apiResponse == null || !apiResponse.success()) {
            log.warn("HTTP接口返回非成功状态: code={}", apiResponse != null ? apiResponse.getCode() : null);
            return new PageResult<>(new ArrayList<>(), pageNo, pageSize);
        }

        List<AssetCollectApiResponse.AssetDataItem> dataItems = apiResponse.getData();
        if (CollectionUtils.isEmpty(dataItems)) {
            return new PageResult<>(new ArrayList<>(), pageNo, pageSize);
        }

        // 返回原始数据项，由上层负责转换
        boolean hasMore = dataItems.size() >= pageSize;

        PageResult<AssetCollectApiResponse.AssetDataItem> pageResult = new PageResult<>();
        pageResult.setData(dataItems);
        pageResult.setHasMore(hasMore);
        pageResult.setCurrentPage(pageNo);
        pageResult.setCurrentPageSize(dataItems.size());

        return pageResult;
    }

    /**
     * 请求脚本处理结果
     */
    @Data
    protected static class RequestScriptResult {
        private final AssetCollectApiRequest request;
        private final Map<String, Object> headerMap;

        public RequestScriptResult(AssetCollectApiRequest request, Map<String, Object> headerMap) {
            this.request = request;
            this.headerMap = headerMap;
        }
    }
}
