package com.digiwin.athena.datacollect.job;

import cn.hutool.core.date.LocalDateTimeUtil;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.convertor.assembly.DataCollectorConverter;
import com.digiwin.athena.dao.mongodao.assetType.RAssetTypeMongoDao;
import com.digiwin.athena.dao.mongodao.datacollect.JobExecutionRecordMongoDao;
import com.digiwin.athena.datacollect.collector.BaseAssetDataCollector;
import com.digiwin.athena.datacollect.consumer.CollectDataConsumer;
import com.digiwin.athena.datacollect.context.CollectContext;
import com.digiwin.athena.datacollect.model.CollectExecutorProp;
import com.digiwin.athena.datacollect.model.CollectResult;
import com.digiwin.athena.datacollect.model.JobBizData;
import com.digiwin.athena.datacollect.model.JobExecData;
import com.digiwin.athena.mongodb.domain.DataCollectConfig;
import com.digiwin.athena.mongodb.domain.assetType.AssetType;
import com.digiwin.athena.mongodb.domain.datacollect.JobExecutionRecord;
import com.digiwin.athena.service.asset.AssetDataCollectService;
import lombok.extern.slf4j.Slf4j;
import org.quartz.Job;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * TODO pzz 数据采集单独线程池，job异步执行，以防阻塞调度器
 * 资产数据采集Quartz任务
 * 负责任务调度、Context构建、结果记录
 */
@Slf4j
@Component
public class AssetDataCollectJob implements Job {

    private static final int DEFAULT_PAGE_SIZE = 100;
    private static final long DEFAULT_STEP_SIZE_MINUTES = 60 * 24 * 30; // 默认一个月

    @Autowired
    private RAssetTypeMongoDao rAssetTypeMongoDao;
    @Autowired
    private JobExecutionRecordMongoDao jobExecutionRecordMongoDao;
    @Autowired
    private AssetDataCollectService assetDataCollectService;
    @Autowired
    private List<BaseAssetDataCollector> collectors;
    private Map<String, BaseAssetDataCollector> collectorMap;

    @Override
    public void execute(JobExecutionContext context) throws JobExecutionException {
        // 初始化collector映射
        if (collectorMap == null) {
            collectorMap = collectors.stream()
                    .collect(Collectors.toMap(BaseAssetDataCollector::getType, c -> c));
        }

        String assetTypeId = context.getJobDetail().getJobDataMap().getString("assetTypeId");

        // 创建任务执行记录
        JobExecutionRecord record = initExecRecord(context, assetTypeId);
        CollectContext collectContext = null;

        log.info("开始执行资产数据采集任务: jobName={}, assetTypeId={}", record.getJobName(), assetTypeId);

        try {
            // 1. 查询AssetType配置（通过objId查询）
            AssetType assetType = rAssetTypeMongoDao.selectById(assetTypeId);
            if (assetType == null) {
                throw new JobExecutionException("资产类型不存在: " + assetTypeId);
            }

            DataCollectConfig config = JSONObject.parseObject(assetType.getDataCollectConfig().toJSONString(),DataCollectConfig.class);
            if (config == null || config.getExecutor() == null) {
                throw new JobExecutionException("资产类型未配置数据采集: " + assetType.getType());
            }

            // 2. 查询上次执行记录
            JobExecutionRecord lastRecord = jobExecutionRecordMongoDao.findLatestByBizKey(assetTypeId);
            JobExecData presetExecData = lastRecord.getNextExecData();

            // 3. 检测配置是否更新
            boolean configChanged = isConfigChanged(config, lastRecord);
            if (configChanged) {
                log.warn("检测到配置更新，结束当前任务并重新开始: assetTypeId={}, " +
                                "oldConfigTime={}, newConfigTime={}",
                        assetTypeId,
                        Optional.ofNullable(lastRecord.getBizData()).map(JobBizData::getConfigUpdateTime).orElse(null),
                        config.getConfigUpdateTime());

                // 清空lastRecord，从头开始
                presetExecData = null;
            }

            // 4. 构建CollectContext（传入任务触发时间）
            LocalDateTime fireTime = LocalDateTimeUtil.of(context.getFireTime());
            collectContext = buildCollectContext(assetType, config, presetExecData, fireTime);

            // 5. 选择对应的collector
            String executorType = config.getExecutor().getType();
            BaseAssetDataCollector collector = collectorMap.get(executorType);
            if (collector == null) {
                throw new JobExecutionException("不支持的采集器类型: " + executorType);
            }

            // 6. 创建DataConsumer
            CollectDataConsumer consumer = assetDataCollectService::batchProcessDataItems;

            // 7. 执行数据采集
            CollectResult result = collector.collect(collectContext, consumer);

            completeRecord(record, result, assetType, config, collectContext);

            log.info("资产数据采集任务执行完成: jobRecord：{}", record);

        } catch (Exception e) {
            failRecord(record, e.getMessage(), collectContext);

            log.error("资产数据采集任务执行失败: jobRecord{}", record, e);
            throw new JobExecutionException(e);
        } finally {
            // 保存执行记录
            jobExecutionRecordMongoDao.save(record);
        }
    }

    protected void failRecord(JobExecutionRecord record, String message, CollectContext collectContext) {
        record.setStatus(JobExecutionRecord.Status.FAILURE);
        record.setFinishTime(LocalDateTime.now());
        record.setErrorMessage(message);
        record.setDurationMs(ChronoUnit.MILLIS.between(record.getFireTime(), record.getFinishTime()));

        // 失败时保存Context以支持续查
        if (collectContext != null) {
            JobExecData failedNextData = buildFailedNextData(collectContext);
            record.setNextExecData(failedNextData);
        }
    }

    protected void completeRecord(JobExecutionRecord record, CollectResult result, AssetType assetType,
                                  DataCollectConfig config, CollectContext collectContext) {
        // 8. 记录执行结果
        record.setStatus(result.isSuccess() ?
                JobExecutionRecord.Status.SUCCESS : JobExecutionRecord.Status.FAILURE);
        record.setFinishTime(result.getEndTime());
        record.setDurationMs(result.getDurationMs());

        // 9. 设置业务数据
        JobBizData bizData = buildBizData(assetType, config, collectContext);
        record.setBizData(bizData);

        // 10. 设置下次执行数据
        JobExecData nextData = buildNextData(result, collectContext);
        record.setNextExecData(nextData);

        if (!result.isSuccess()) {
            record.setErrorMessage(result.getErrorMessage());
        }
    }

    protected JobExecutionRecord initExecRecord(JobExecutionContext context, String assetTypeId) {
        JobExecutionRecord record = new JobExecutionRecord();

        record.setJobName(context.getJobDetail().getKey().getName());
        record.setBizKey(assetTypeId);
        record.setFireTime(LocalDateTimeUtil.of(context.getFireTime()));
        record.setCreatedAt(LocalDateTime.now());
        record.setStatus(JobExecutionRecord.Status.RUNNING);

        return record;
    }

    /**
     * 构建CollectContext
     * 获取上次执行记录（不管成功失败），如果有nextData则合并到Context中
     *
     * @param fireTime 任务触发时间
     */
    protected CollectContext buildCollectContext(AssetType assetType,
                                                 DataCollectConfig config,
                                                 JobExecData presetExecData,
                                                 LocalDateTime fireTime) {
        CollectExecutorProp executor = config.getExecutor();
        // 分页参数（优先使用executor配置的pageSize，否则使用默认值）
        int pageSize = executor.getPageSize() != null ? executor.getPageSize() : DEFAULT_PAGE_SIZE;
        Long stepSizeMinutes = executor.getStepSizeMinutes() != null ? executor.getStepSizeMinutes() : DEFAULT_STEP_SIZE_MINUTES;

        CollectContext context = new CollectContext();
        // 基本信息
        context.setAssetTypeId(assetType.getObjId());
        context.setAssetType(assetType);
        context.setConfig(config);
        context.setStepSizeMinutes(stepSizeMinutes);

        context.init(pageSize);

        // 获取上次执行的nextData（不管成功失败）
        if (presetExecData != null) {
            DataCollectorConverter.INSTANCE.merge(presetExecData, context);
            log.info("续查任务:  assetTypeId={}, context:{}", assetType.getObjId(), context);
        } else {
            // 首次执行或无历史记录，从最小时间开始
            context.setStartTime(LocalDateTime.MIN);
            log.info("首次执行任务: assetTypeId={}, startTime={}", assetType.getObjId(), context.getStartTime());
        }

        // 使用任务触发时间作为endTime
        if (context.getEndTime() == null) {
            context.setEndTime(fireTime);
        }

        context.setFireTime(fireTime);

        return context;
    }

    /**
     * 检测配置是否更新
     */
    protected boolean isConfigChanged(DataCollectConfig config, JobExecutionRecord lastRecord) {
        if (config.getConfigUpdateTime() == null) {
            return false;
        }

        if (lastRecord == null || lastRecord.getBizData() == null) {
            return false;
        }

        LocalDateTime lastConfigTime = lastRecord.getBizData().getConfigUpdateTime();
        if (lastConfigTime == null) {
            return false;
        }

        return !config.getConfigUpdateTime().equals(lastConfigTime);
    }

    /**
     * 构建业务数据
     */
    protected JobBizData buildBizData(AssetType assetType, DataCollectConfig config, CollectContext context) {
        JobBizData bizData = new JobBizData();

        bizData.setAssetTypeId(assetType.getObjId());
        bizData.setAssetType(assetType.getType());
        bizData.setAssetTypeName(assetType.getName());
        bizData.setStatus(assetType.getStatus());
        bizData.setConfigUpdateTime(config.getConfigUpdateTime());
        bizData.setTotalCount(context.getTotalRecords());
        bizData.setConsumedCount(context.getConsumedRecords());
        bizData.setTotalPages(context.getTotalPages());
        bizData.setTotalSteps(context.getTotalSteps());

        return bizData;
    }

    /**
     * 构建下次执行数据
     */
    protected JobExecData buildNextData(CollectResult result, CollectContext context) {
        JobExecData nextData = new JobExecData();

        if (result.isSuccess()) {
            // 成功时保存最后采集时间和startTime
            nextData.setLastCollectTime(result.getEndTime());
            nextData.setStartTime(context.getCurrentStepEndTime());
        } else {
            // 失败时保存完整Context信息以支持续查
            nextData.setLastCollectTime(LocalDateTime.now());
            nextData.setStartTime(context.getStartTime());
            nextData.setEndTime(context.getEndTime());
            nextData.setPageNo(context.getPageNo());
            nextData.setLoopIndex(context.getLoopIndex());
            nextData.setStepStartTime(context.getCurrentStepStartTime());
            nextData.setStepEndTime(context.getCurrentStepEndTime());
        }

        return nextData;
    }

    /**
     * 构建失败续查数据（异常情况）
     */
    protected JobExecData buildFailedNextData(CollectContext context) {
        JobExecData nextData = new JobExecData();
        nextData.setLastCollectTime(LocalDateTime.now());

        if (context != null) {
            nextData.setStartTime(context.getStartTime());
            nextData.setEndTime(context.getEndTime());
            nextData.setPageNo(context.getPageNo());
            nextData.setLoopIndex(context.getLoopIndex());
            nextData.setStepStartTime(context.getCurrentStepStartTime());
            nextData.setStepEndTime(context.getCurrentStepEndTime());
        }

        return nextData;
    }
}
