package com.digiwin.athena.service.assembly.impl;

import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.bo.assembly.Designer;
import com.digiwin.athena.constants.AssemblyErrorCode;
import com.digiwin.athena.constants.enums.assembly.OnlineState;
import com.digiwin.athena.convertor.assembly.DesignerConvertor;
import com.digiwin.athena.convertor.assembly.SolutionConvertor;
import com.digiwin.athena.dao.mongodao.assembly.DDesignerMongoDao;
import com.digiwin.athena.dao.mongodao.assembly.RDesignerMongoDao;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.digiwin.athena.dto.assembly.designer.DesignerDetailDTO;
import com.digiwin.athena.dto.assembly.designer.DesignerListDTO;
import com.digiwin.athena.dto.assembly.designer.DesignerPageQo;
import com.digiwin.athena.dto.assembly.designer.DesignerSaveDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanListDTO;
import com.digiwin.athena.service.SyncRuntime;
import com.digiwin.athena.service.assembly.DDesignerService;
import com.digiwin.athena.service.assembly.DSolutionPlanService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;


/**
 * 设计器管理服务实现
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class DDesignerServiceImpl extends SyncRuntime<Designer> implements DDesignerService {

    private final DDesignerMongoDao dDesignerMongoDao;
    private final RDesignerMongoDao rDesignerMongoDao;
    private final DSolutionPlanService solutionPlanService;

    private DDesignerServiceImpl proxyThis;

    @Override
    public Pagination<DesignerListDTO> findList(PageReqCondition<DesignerPageQo> param) {
        Pagination<DesignerListDTO> pagination = dDesignerMongoDao.selectPage(param);

        // TODO pzz 根据assetId查找最新asset数据
        fillSolutionPlan(pagination);

        return pagination;
    }

    protected void fillSolutionPlan(Pagination<DesignerListDTO> pagination) {
        // 获取设计器ID列表
        List<String> designerIds = pagination.getData().stream()
                .map(DesignerListDTO::getId)
                .collect(Collectors.toList());

        // 使用solutionPlanService根据设计器ID查询相关解决方案
        if (designerIds.isEmpty()) {
            return;
        }

        Map<String, Set<SolutionPlanListDTO>> solutionMap = solutionPlanService.findListByDesignerId(designerIds);
        // 将解决方案信息添加到设计器DTO中
        for (DesignerListDTO designer : pagination.getData()) {
            Set<SolutionPlanListDTO> solutionPlan = solutionMap.get(designer.getId());
            designer.setRelatedSolutionPlan(SolutionConvertor.INSTANCE.toBaseDTO(solutionPlan));
        }
    }

    @Transactional(rollbackFor = Throwable.class)
    @Override
    public void save(Designer updated) {
        log.info("开始创建设计时态设计器: {}", updated.getName());

        // 验证type字段格式
        updated.validateTypeFormat();

        checkUnique(updated);

        String objectId = updated.getId();
        if (objectId != null) {
            Designer existingBo = findExist(objectId);

            updated.removeUnmodifiable(existingBo.getStatus());
            DesignerConvertor.INSTANCE.merge(updated, existingBo);
            updated = existingBo;
        } else {
            updated.prepareNew();
        }

        dDesignerMongoDao.save(updated);
        log.info("成功创建设计时态设计器: {}", updated.getId());
    }

    /**
     * 单纯保存，不做任何业务处理
     */
    protected void pureSave(Designer designer) {
        dDesignerMongoDao.save(designer);
    }

    @Transactional(rollbackFor = Throwable.class)
    @Override
    public Designer save(DesignerSaveDTO designer) {
        Designer bo = DesignerConvertor.INSTANCE.toBo(designer);

        save(bo);

        return bo;
    }


    @Transactional(rollbackFor = Throwable.class)
    @Override
    public void deleteById(String objectId) {
        log.info("开始删除设计时态设计器: {}", objectId);

        // 检查设计器是否存在
        Designer existingBo = findExist(objectId);
        // 检查状态是否为草稿
        existingBo.checkDelete();

        dDesignerMongoDao.deleteById(objectId);
        log.info("成功删除设计时态设计器: {}", objectId);
    }

    protected void checkUnique(Designer designer) {
        if (!validateUnique(designer.getType(), null, designer.getId())) {
            throw new BusinessException(AssemblyErrorCode.NAME_CODE_NOT_UNIQUE, "类型已存在！");
        }

        if (!validateUnique(null, designer.getName(), designer.getId())) {
            throw new BusinessException(AssemblyErrorCode.NAME_CODE_NOT_UNIQUE, "名称已存在！");
        }
    }

    protected Designer findExist(String objectId) {
        return Optional
                .ofNullable(dDesignerMongoDao.selectById(objectId))
                .orElseThrow(() -> new BusinessException(AssemblyErrorCode.DESIGNER_NOT_EXIST, "设计器不存在！", objectId));
    }

    @Override
    public DesignerDetailDTO findDetailById(String designerId) {
        // TODO pzz 根据assetId查找最新asset数据
        Map<String, Set<SolutionPlanListDTO>> solutionMap = solutionPlanService.findListByDesignerId(Collections.singleton(designerId));


        DesignerDetailDTO dto = DesignerConvertor.INSTANCE.toDto(findById(designerId));
        dto.setRelatedSolutionPlan(SolutionConvertor.INSTANCE.toBaseDTO(solutionMap.get(designerId)));

        return dto;
    }

    @Override
    public Designer findById(String objectId) {
        log.debug("查询设计时态设计器详情: {}", objectId);
        return dDesignerMongoDao.selectById(objectId);
    }


    @Override
    public boolean validateUnique(String code, String name, String excludeObjectId) {
        return !dDesignerMongoDao.existsByKey(code, name, excludeObjectId);
    }

    @Override
    public List<DesignerListDTO> findPublishedList() {
        PageReqCondition<DesignerPageQo> param = new PageReqCondition<>();
        param.setPageNum(1);
        param.setPageSize(9999);
        param.setCondition(new DesignerPageQo());

        // 设置状态列表为已上架和已下架
        param.getCondition().setStatusList(OnlineState.publishedCode());

        return findList(param).getData();
    }

    @Override
    protected void saveDesignerTimeData(Designer syncData) {
        pureSave(syncData);
    }

    @Override
    protected void insertRunTimeData(Designer syncData) {
        rDesignerMongoDao.save(syncData);
    }

    @Override
    protected void saveRunTimeData(Designer syncData) {
        rDesignerMongoDao.save(syncData);
    }

    @Override
    protected Designer getDesignerTimeData(String objId) {
        return proxyThis.findById(objId);
    }

    @Override
    protected Designer getRunTimeData(String objId) {
        return rDesignerMongoDao.selectById(objId);
    }

    @Override
    protected void deleteRuntimeData(String objId) {
        rDesignerMongoDao.deleteById(objId);
    }

    @Autowired
    @Lazy
    public void setProxyThis(DDesignerServiceImpl proxyThis) {
        this.proxyThis = proxyThis;
    }
}