package com.digiwin.athena.service.asset.impl;

import cn.hutool.core.collection.CollUtil;
import com.digiwin.athena.constant.Constant;
import com.digiwin.athena.convertor.assembly.AssetConverter;
import com.digiwin.athena.dao.mongodao.asset.AssetDefinitionMongoDao;
import com.digiwin.athena.dao.mongodao.asset.AssetMongoDao;
import com.digiwin.athena.datacollect.api.dto.AssetCollectApiResponse;
import com.digiwin.athena.datacollect.context.CollectContext;
import com.digiwin.athena.mongodb.domain.application.Asset;
import com.digiwin.athena.mongodb.domain.application.AssetDefinition;
import com.digiwin.athena.service.asset.AssetDataCollectService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.function.Supplier;
import java.util.stream.Collectors;

/**
 * 资产数据采集服务实现
 */
@Slf4j
@Service
public class AssetDataCollectServiceImpl implements AssetDataCollectService {

    public static final String EDIT_USER = Constant.SYSTEM;

    @Autowired
    private AssetMongoDao assetMongoDao;

    @Autowired
    private AssetDefinitionMongoDao assetDefinitionMongoDao;


    @Override
    public int batchUpsertAssets(List<Asset> assets) {
        throw new UnsupportedOperationException();
    }

    @Override
    public int batchProcessDataItems(CollectContext context, List<AssetCollectApiResponse.AssetDataItem> dataItems) {
        if (CollectionUtils.isEmpty(dataItems)) {
            return 0;
        }

        // 转换为Asset对象列表，同时生成按操作类型分类的assetId Map和definitionSupplierMap
        Map<String, Set<String>> assetIdsByOp = new HashMap<>();
        Map<String, Supplier<AssetDefinition>> definitionSupplierMap = new HashMap<>();
        List<Asset> assets = convertToAssets(dataItems, context, assetIdsByOp, definitionSupplierMap);

        // 调用重构后的batchUpsertAssets方法，传递操作类型Map和definitionSupplierMap
        return batchUpsertAssets(assets, assetIdsByOp, definitionSupplierMap);
    }

    /**
     * 批量处理资产（插入、更新、删除）
     *
     * @param assets                资产列表
     * @param assetIdsByOp          按操作类型分类的assetId
     *                              Map，key为操作类型(insert/update/delete/upsert)，value为assetId集合
     * @param definitionSupplierMap AssetDefinition的Supplier映射，key为assetId，value为创建AssetDefinition的Supplier
     * @return 成功处理的数量
     */
    public int batchUpsertAssets(List<Asset> assets,
                                 Map<String, Set<String>> assetIdsByOp,
                                 Map<String, Supplier<AssetDefinition>> definitionSupplierMap) {
        if (CollectionUtils.isEmpty(assets)) {
            return 0;
        }

        List<Asset> toInsert = new ArrayList<>();
        List<Asset> toUpdate = new ArrayList<>();
        List<String> toDelete = new ArrayList<>();
        List<AssetDefinition> toInsertDefinitions = new ArrayList<>();

        try {
            // 批量查询已存在的资产(根据assetId去重)
            List<String> assetIds = assets.stream()
                    .map(Asset::getAssetId)
                    .filter(StringUtils::hasText)
                    .distinct().collect(Collectors.toList());

            // 转换为Map便于查找,key为assetId
            Map<String, Asset> existingAssetMap = assetMongoDao.findByAssetIds(assetIds).stream()
                    .collect(Collectors.toMap(Asset::getAssetId, asset -> asset, (a, b) -> a));

            // 查询已存在的AssetDefinition，只查询assetId字段提升性能
            Set<String> existingDefinitionAssetIds = assetDefinitionMongoDao.selectExistAssetIds(assetIds);

            // 获取删除操作的assetId集合
            Set<String> deleteAssetIds = assetIdsByOp.getOrDefault("delete", Collections.emptySet());

            // 遍历待处理的资产，判断操作类型
            for (Asset asset : assets) {
                String assetId = asset.getAssetId();

                // 删除操作：从assetIdsByOp中判断
                if (deleteAssetIds.contains(assetId)) {
                    if (existingAssetMap.containsKey(assetId)) {
                        toDelete.add(assetId);
                    }
                    continue;
                }

                // 检查并准备创建AssetDefinition
                if (!existingDefinitionAssetIds.contains(assetId)) {
                    AssetDefinition assetDefinition = definitionSupplierMap.get(assetId).get();
                    prepareDefinitionForInsert(assetDefinition);

                    toInsertDefinitions.add(assetDefinition);
                    existingDefinitionAssetIds.add(assetId);
                }

                // 插入或更新操作
                Asset existingAsset = existingAssetMap.get(assetId);

                if (existingAsset == null) {
                    prepareAssetForInsert(asset);
                    // 不存在，执行插入
                    toInsert.add(asset);
                    existingAssetMap.put(assetId, asset);
                } else {
                    // 已存在，判断是否需要更新
                    if (shouldUpdate(asset, existingAsset)) {
                        prepareAssetForUpdate(asset, existingAsset);
                        toUpdate.add(existingAsset);
                    } else {
                        log.debug("资产无需更新: assetId={},existEditTime:{}, editTime:{}",
                                assetId, existingAsset.getEditDate(), asset.getEditDate());
                    }
                }
            }

            return persist(toInsertDefinitions, toInsert, toUpdate, toDelete);
        } catch (Exception e) {
            log.error("批量处理资产失败", e);
            throw new RuntimeException("批量处理资产失败: " + e.getMessage(), e);
        }

    }

    protected int persist(List<AssetDefinition> toInsertDefinitions, List<Asset> toInsert, List<Asset> toUpdate, List<String> toDelete) {
        int successCount = 0;

        // TODO pzz performance 修改为批量
        // 批量插入AssetDefinition
        if (CollUtil.isNotEmpty(toInsertDefinitions)) {
            assetDefinitionMongoDao.insertAll(toInsertDefinitions);
            log.info("批量插入AssetDefinition成功: count={}", toInsertDefinitions.size());
        }

        // 执行批量插入
        if (CollUtil.isNotEmpty(toInsert)) {
            assetMongoDao.insertAll(toInsert);
            successCount += toInsert.size();
            log.info("批量插入资产成功: count={}", toInsert.size());
        }

        // 执行批量更新
        if (CollUtil.isNotEmpty(toUpdate)) {
            for (Asset asset : toUpdate) {
                assetMongoDao.save(asset);
                successCount++;
            }
            log.info("批量更新资产成功: count={}", toUpdate.size());
        }

        // 执行批量删除
        if (CollUtil.isNotEmpty(toDelete)) {
            for (String assetId : toDelete) {
                assetMongoDao.delete(assetId);
                successCount++;
            }
            log.info("批量删除资产成功: count={}", toDelete.size());
        }

        return successCount;
    }

    /**
     * 创建AssetDefinition对象（用于批量插入）
     */
    protected AssetDefinition buildAssetDefinition(AssetCollectApiResponse.AssetData originData, Asset asset) {
        AssetDefinition definition = new AssetDefinition();
        definition.setAssetId(asset.getAssetId());
        definition.setCode(asset.getCode());
        definition.setType(asset.getType());
        definition.setAssetName(asset.getAssetName());
        definition.setSource(AssetDefinition.SD_SOURCE);
        definition.setStatus(AssetDefinition.AVAILABLE_STATUS);
        definition.setCreateDate(new Date());
        definition.setEditDate(new Date());

        // 设置租户ID
        if (StringUtils.hasText(originData.getAdpTenantId())) {
            definition.setAdpTenantId(originData.getAdpTenantId());
        }
        // 设置管理上下文
        if (asset.getManageContext() != null && !asset.getManageContext().isEmpty()) {
            definition.setManageContext(asset.getManageContext());
        }

        log.debug("创建AssetDefinition: assetId={}, type={}, code={}",
                asset.getAssetId(), asset.getType(), asset.getCode());

        return definition;
    }

    /**
     * 判断是否需要更新
     * 根据editDate字段值决定
     */
    protected boolean shouldUpdate(Asset newAsset, Asset existingAsset) {
        // 如果新资产有editDate，比较时间
        if (newAsset.getEditDate() != null && existingAsset.getEditDate() != null) {
            return newAsset.getEditDate().after(existingAsset.getEditDate());
        }
        // 默认更新
        return true;
    }

    protected void prepareDefinitionForInsert(AssetDefinition assetDefinition) {
        Date now = new Date();
        assetDefinition.setCreateBy(EDIT_USER);
        assetDefinition.setCreateDate(now);

        assetDefinition.setEditBy(EDIT_USER);
        assetDefinition.setEditDate(now);
    }

    /**
     * 准备插入的资产数据
     */
    protected void prepareAssetForInsert(Asset asset) {
        Date now = new Date();
        // 设置创建时间
        if (asset.getCreateDate() == null) {
            asset.setCreateBy(EDIT_USER);
            asset.setCreateDate(now);
        }

        // 设置修改时间
        if (asset.getEditDate() == null) {
            asset.setEditBy(EDIT_USER);
            asset.setEditDate(now);
        }

        // 设置状态为DRAFT（存储在manageContext中）
        if (asset.getManageContext() == null) {
            asset.setManageContext(new Asset.ManageContext());
        }
        asset.getManageContext().setAdpStatus(Asset.ADP_STATUS_DRAFT);

        // 设置source为独立资产
        asset.setSource(AssetDefinition.SD_SOURCE);
    }

    /**
     * 准备更新的资产数据,使用MapStruct merge只更新变更字段
     */
    protected void prepareAssetForUpdate(Asset newAsset, Asset existingAsset) {
        // 更新修改时间
        if (newAsset.getEditDate() == null) {
            newAsset.setEditBy(EDIT_USER);
            newAsset.setEditDate(new Date());
        }
        // 设置source为独立资产
        newAsset.setSource(AssetDefinition.SD_SOURCE);

        // 使用MapStruct merge方法,只更新非空字段
        AssetConverter.INSTANCE.merge(newAsset, existingAsset);
    }

    /**
     * 构建assetId（根据type和code生成唯一标识）
     */
    protected String buildAssetId(String type, String code) {
        return type + "_" + code;
    }

    /**
     * 将AssetCollectApiResponse.AssetDataItem转换为Asset对象
     *
     * @param dataItems             数据项列表
     * @param context               采集上下文
     * @param assetIdsByOp          输出参数，按操作类型分类的assetId Map
     * @param definitionSupplierMap 输出参数，AssetDefinition的Supplier映射
     * @return Asset对象列表
     */
    protected List<Asset> convertToAssets(List<AssetCollectApiResponse.AssetDataItem> dataItems,
                                          CollectContext context,
                                          Map<String, Set<String>> assetIdsByOp,
                                          Map<String, Supplier<AssetDefinition>> definitionSupplierMap) {
        List<Asset> assets = new ArrayList<>();

        for (AssetCollectApiResponse.AssetDataItem item : dataItems) {
            AssetCollectApiResponse.AssetData data = item.getData();
            if (data == null) {
                continue;
            }

            Asset asset = AssetConverter.INSTANCE.toAsset(data);
            // 生成并设置assetId
            String assetId = buildAssetId(data.getType(), data.getCode());
            asset.setAssetId(assetId);

            String op = item.obtainOp(context.getOpFromLoopParam(), assetId);
            assetIdsByOp
                    .computeIfAbsent(op.toLowerCase(), k -> new HashSet<>())
                    .add(assetId);

            // 创建AssetDefinition的Supplier，用于延迟创建
            if (!definitionSupplierMap.containsKey(assetId)) {
                definitionSupplierMap.put(assetId, () -> buildAssetDefinition(data, asset));
            }

            assets.add(asset);
        }

        return assets;
    }
}
