package com.digiwin.athena.service.assetType.impl;

import com.digiwin.athena.bo.assembly.Designer;
import com.digiwin.athena.dao.mongodao.assembly.DDesignerMongoDao;
import com.digiwin.athena.dao.mongodao.assetType.DAssetTypeMongoDao;
import com.digiwin.athena.dao.mongodao.assetType.RAssetTypeMongoDao;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.digiwin.athena.dto.assembly.designer.DesignerDetailDTO;
import com.digiwin.athena.dto.assetType.*;
import com.digiwin.athena.event.AssetTypeEvent;
import com.digiwin.athena.event.EventOperation;
import com.digiwin.athena.mongodb.domain.assetType.AssetType;
import com.digiwin.athena.service.SyncRuntime;
import com.digiwin.athena.service.assetType.DAssetTypeService;
import com.digiwin.athena.utils.user.UserHelper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Service
@Slf4j
public class DAssetTypeServiceImpl extends SyncRuntime<AssetType> implements DAssetTypeService {

    @Autowired
    private DAssetTypeMongoDao dAssetTypeMongoDao;

    @Autowired
    private RAssetTypeMongoDao rAssetTypeMongoDao;

    @Autowired
    private ApplicationEventPublisher applicationEventPublisher;
    @Autowired
    private DDesignerMongoDao dDesignerMongoDao;

    @Override
    public Pagination<AssetTypeListResDto> getAssetTypeList(PageReqCondition<AssetTypeListConditionReqDto> pageReqCondition) {

        Long count = dAssetTypeMongoDao.countAssetTypeList(pageReqCondition);
        List<AssetType> assetTypeList = dAssetTypeMongoDao.getAssetTypeList(pageReqCondition);
        List<AssetTypeListResDto> res = assetTypeList.stream().map(r -> AssetTypeListResDto.create(r)).collect(Collectors.toList());

        if (!res.isEmpty()){
            Map<String, AssetTypeListResDto> resMap = res.stream().collect(Collectors.toMap(AssetTypeListResDto::getObjId, a -> a));

            List<Designer> designers = dDesignerMongoDao.selectByRelatedAsset(resMap.keySet());
            for (Designer designer : designers) {

                DesignerDetailDTO.RelatedAssetTypeDTO relatedAssetType = designer.getRelatedAssetType();
                if (relatedAssetType!=null){
                    Map<String, RelationDesignerInfo> relationDesignerInfoMap = getStringRelationDesignerInfoMap(designer);

                    for (Map.Entry<String, RelationDesignerInfo> entrySet : relationDesignerInfoMap.entrySet()) {
                        String key = entrySet.getKey();

                        if (resMap.containsKey(key)){
                            AssetTypeListResDto assetTypeListResDto = resMap.get(key);
                            assetTypeListResDto.getRelationDesignerInfoList().add(entrySet.getValue());
                        }
                    }
                }
            }
        }

        return Pagination.buildPagination(pageReqCondition,res,count);
    }

    private static Map<String, RelationDesignerInfo> getStringRelationDesignerInfoMap(Designer designer) {
        Map<String,RelationDesignerInfo> relationDesignerInfoMap = new HashMap<>();
        DesignerDetailDTO.RelatedAssetTypeDTO relatedAssetType = designer.getRelatedAssetType();
        if (relatedAssetType == null){
            return relationDesignerInfoMap;
        }
        List<AssetTypeBaseDTO> write = relatedAssetType.getWrite();
        if (!CollectionUtils.isEmpty(write)){
            for (AssetTypeBaseDTO assetTypeBaseDTO : write) {
                if (relationDesignerInfoMap.containsKey(assetTypeBaseDTO.getObjId())){
                    RelationDesignerInfo relationDesignerInfo = relationDesignerInfoMap.get(assetTypeBaseDTO.getObjId());
                    relationDesignerInfo.setWrite(true);
                }else{
                    RelationDesignerInfo relationDesignerInfo = RelationDesignerInfo.create(designer);
                    relationDesignerInfo.setWrite(true);
                    relationDesignerInfoMap.put(assetTypeBaseDTO.getObjId(),relationDesignerInfo);
                }
            }
        }

        List<AssetTypeBaseDTO> read = relatedAssetType.getRead();
        if (!CollectionUtils.isEmpty(read)){
            for (AssetTypeBaseDTO assetTypeBaseDTO : read) {
                if (relationDesignerInfoMap.containsKey(assetTypeBaseDTO.getObjId())){
                    RelationDesignerInfo relationDesignerInfo = relationDesignerInfoMap.get(assetTypeBaseDTO.getObjId());
                    relationDesignerInfo.setRead(true);
                }else{
                    RelationDesignerInfo relationDesignerInfo = RelationDesignerInfo.create(designer);
                    relationDesignerInfo.setRead(true);
                    relationDesignerInfoMap.put(assetTypeBaseDTO.getObjId(),relationDesignerInfo);
                }
            }
        }
        return relationDesignerInfoMap;
    }

    @Override
    public AssetTypeResDto registerAssetType(AssetTypeReqDto assetTypeReqDto) {
        assetTypeReqDto.validateTypeFormat();

        if (StringUtils.isNotEmpty(assetTypeReqDto.getObjId())){
            AssetType oldAssetType = dAssetTypeMongoDao.selectById(assetTypeReqDto.getObjId());
            Assert.notNull(oldAssetType,"未查询到对应的资产类型");

            if (!assetTypeReqDto.getName().equals(oldAssetType.getName())){
                Long count = dAssetTypeMongoDao.checkExistByNameExcludeType(assetTypeReqDto.getName(),oldAssetType.getType());
                Assert.isTrue(count.equals(0L), "名称已存在");
            }

            if (!assetTypeReqDto.getType().equals(oldAssetType.getType())){
                Long count = dAssetTypeMongoDao.checkExistByType(assetTypeReqDto.getType());
                Assert.isTrue(count.equals(0L), "类型已存在");
            }

            BeanUtils.copyProperties(assetTypeReqDto,oldAssetType);
            UserHelper.fillEditInfo(oldAssetType);
            dAssetTypeMongoDao.save(oldAssetType);

            return AssetTypeResDto.create(oldAssetType);
        }else{
            Long count = dAssetTypeMongoDao.checkExistByType(assetTypeReqDto.getType());
            Assert.isTrue(count.equals(0L), "类型已经存在");
            count = dAssetTypeMongoDao.checkExistByNameExcludeType(assetTypeReqDto.getName(),assetTypeReqDto.getType());
            Assert.isTrue(count.equals(0L), "名称已存在");

            AssetType assetType = new AssetType();
            BeanUtils.copyProperties(assetTypeReqDto,assetType);

            UserHelper.fillCreateInfo(assetType);
            UserHelper.fillEditInfo(assetType);
            dAssetTypeMongoDao.insert(assetType);
            return AssetTypeResDto.create(assetType);
        }
    }

    @Override
    public AssetTypeResDto getAssetTypeDetail(String objId,String type) {
        AssetType assetType = null;
        if (!StringUtils.isEmpty(objId)){
            assetType = dAssetTypeMongoDao.selectById(objId);
        }else{
            assetType = dAssetTypeMongoDao.selectByType(type);
        }
        Assert.notNull(assetType,"未查询到对应的资产类型信息");
        AssetTypeResDto assetTypeResDto = AssetTypeResDto.create(assetType);

        List<Designer> designers =  dDesignerMongoDao.selectByRelatedAsset(objId);
        for (Designer designer : designers) {
            Map<String, RelationDesignerInfo> relationDesignerInfoMap = getStringRelationDesignerInfoMap(designer);
            if (relationDesignerInfoMap.containsKey(assetTypeResDto.getObjId())){
                assetTypeResDto.getRelationDesignerInfos().add(relationDesignerInfoMap.get(assetTypeResDto.getObjId()));
            }
        }
        return assetTypeResDto;
    }



    @Override
    public void assetTypePublish(String objId) {
        AssetType assetType = publish(objId);

        AssetTypeEvent assetTypeEvent = new AssetTypeEvent(assetType, EventOperation.UPDATE);
        applicationEventPublisher.publishEvent(assetTypeEvent);
    }

    @Override
    public void assetTypeTakeDown(String objId) {
        AssetType assetType = takeDown(objId);

        AssetTypeEvent assetTypeEvent = new AssetTypeEvent(assetType, EventOperation.DELETE);
        applicationEventPublisher.publishEvent(assetTypeEvent);
    }

    @Override
    public void assetTypeLaunch(String objId) {
        AssetType assetType = launch(objId);

        AssetTypeEvent assetTypeEvent = new AssetTypeEvent(assetType, EventOperation.UPDATE);
        applicationEventPublisher.publishEvent(assetTypeEvent);
    }

    @Override
    public void delete(String objId) {
        AssetType assetType = dAssetTypeMongoDao.selectById(objId);
        Assert.notNull(assetType,"未查询到对应的记录");
        Assert.isTrue(AssetType.STATUS_DRAFT.equals(assetType.getStatus()),"非草稿无法删除");
        dAssetTypeMongoDao.delete(assetType);
    }

    @Override
    public List<AssetTypeResDto> getPublishedList() {
        List<AssetType> assetTypeList = dAssetTypeMongoDao.selectByStatus(Arrays.asList(AssetType.STATUS_OFFLINE,AssetType.STATUS_ONLINE));
        List<AssetTypeResDto> res = assetTypeList.stream().map(a -> AssetTypeResDto.create(a)).collect(Collectors.toList());
        return res;
    }

    @Override
    protected void saveDesignerTimeData(AssetType syncData) {
        dAssetTypeMongoDao.save(syncData);
    }

    @Override
    protected void insertRunTimeData(AssetType syncData) {
        rAssetTypeMongoDao.insert(syncData);
    }

    @Override
    protected void saveRunTimeData(AssetType syncData) {
        rAssetTypeMongoDao.save(syncData);
    }

    @Override
    protected AssetType getDesignerTimeData(String objId) {
        return dAssetTypeMongoDao.selectById(objId);
    }

    @Override
    protected AssetType getRunTimeData(String objId) {
        return rAssetTypeMongoDao.selectById(objId);
    }

    @Override
    protected void deleteRuntimeData(String objId) {
        rAssetTypeMongoDao.deleteById(objId);
    }
}
