package com.digiwin.athena.datacollect.context;

import com.digiwin.athena.mongodb.domain.DataCollectConfig;
import com.digiwin.athena.mongodb.domain.assetType.AssetType;
import lombok.Data;

import java.time.LocalDateTime;
import java.util.Map;

/**
 * 数据采集执行上下文
 * 包含采集任务执行所需的所有参数和状态信息
 */
@Data
public class CollectContext {

    /**
     * 资产类型ID（AssetType的objId）
     */
    private String assetTypeId;

    /**
     * 资产类型（AssetType的type字段）
     */
    private AssetType assetType;

    /**
     * 任务开始时间
     */
    private LocalDateTime fireTime;

    /**
     * 数据采集配置
     */
    private DataCollectConfig config;

    /**
     * 查询开始时间
     */
    private LocalDateTime startTime;

    /**
     * 查询结束时间
     */
    private LocalDateTime endTime;

    /**
     * 当前页码（从1开始）
     */
    private Integer pageNo;

    /**
     * 每页大小
     */
    private Integer pageSize;

    /**
     * 查询时间步长（分钟）
     * 用于步进式查询，避免一次查询时间范围过大
     */
    private Long stepSizeMinutes;

    /**
     * 当前循环参数
     * 对应DataCollectConfig.Executor.loopParams中的一项
     */
    private Map<String, Object> loopParam;


    // ======= current =======

    /**
     * 循环索引（从0开始）
     */
    private Integer loopIndex;

    /**
     * 当前步骤开始时间（用于步进式查询）
     */
    private LocalDateTime currentStepStartTime;

    /**
     * 当前步骤结束时间（用于步进式查询）
     */
    private LocalDateTime currentStepEndTime;


    // ========== statistics ==========
    /**
     * 分页统计 - 已消费记录数
     */
    private Integer consumedRecords;

    /**
     * 总步数
     */
    private Integer totalSteps;

    /**
     * 分页统计 - 总页数
     */
    private Integer totalPages;

    /**
     * 分页统计 - 总记录数
     */
    private Integer totalRecords;


    /**
     * 初始化分页参数
     */
    public void init(int pageSize) {
        this.pageNo = 1;
        this.pageSize = pageSize;
        this.loopIndex = 0;
        this.totalSteps = 0;
        this.totalPages = 0;
        this.totalRecords = 0;
        this.consumedRecords = 0;
    }

    /**
     * 初始化当前步骤开始时间
     */
    public void begin() {
        LocalDateTime currentStepStart = this.currentStepStartTime;
        if (currentStepStart == null) {
            currentStepStart = this.startTime;

            this.currentStepStartTime = currentStepStart;
            this.currentStepEndTime = calculateStepEndTime(currentStepStart);
        }

        if (this.loopIndex == null) {
            this.loopIndex = 0;
        }
    }


    public boolean hasNext() {
        return currentStepStartTime.isBefore(endTime);
    }


    public void step() {
        this.totalSteps++;
        this.pageNo = 1;
        this.loopIndex = 0;

        this.loopParam = null;
        this.currentStepStartTime = this.currentStepEndTime;
        this.currentStepEndTime = calculateStepEndTime(this.currentStepStartTime);
    }

    /**
     * 增加页码
     */
    public void incrementPageNo() {
        this.pageNo++;
    }

    /**
     * 增加记录统计
     */
    public void addRecords(int fetchedCount, int consumedCount) {
        this.totalRecords += fetchedCount;
        this.consumedRecords += consumedCount;
        this.totalPages++;
    }

    /**
     * 从loopParam中获取op操作类型
     *
     * @return 操作类型: insert/update/delete/upsert，默认返回upsert
     */
    public String getOpFromLoopParam() {
        if (loopParam != null && loopParam.containsKey("op")) {
            Object op = loopParam.get("op");
            return op != null ? op.toString() : "upsert";
        }

        return "upsert";
    }

    /**
     * 计算步骤结束时间
     *
     * @param stepStart 当前步开始时间
     * @return 当前步结束时间
     */
    public LocalDateTime calculateStepEndTime(LocalDateTime stepStart) {
        LocalDateTime finalEndTime = getEndTime();
        if (stepSizeMinutes == null || stepSizeMinutes <= 0) {
            return finalEndTime;
        }

        LocalDateTime stepEnd = stepStart.plusMinutes(stepSizeMinutes);
        return stepEnd.isBefore(finalEndTime) ? stepEnd : finalEndTime;
    }


}
