package com.digiwin.athena.datacollect.listener;

import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.constants.enums.assembly.OnlineState;
import com.digiwin.athena.datacollect.job.QuartzJobManager;
import com.digiwin.athena.event.AssetTypeEvent;
import com.digiwin.athena.event.EventOperation;
import com.digiwin.athena.mongodb.domain.DataCollectConfig;
import com.digiwin.athena.mongodb.domain.assetType.AssetType;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Component;

/**
 * 数据采集事件监听器
 * 监听AssetTypeEvent，根据事件操作管理定时任务
 */
@Slf4j
@Component
public class DataCollectEventListener {

    @Autowired
    private QuartzJobManager quartzJobManager;

    /**
     * 处理资产类型事件
     */
    @EventListener
    public void handleAssetTypeEvent(AssetTypeEvent event) {
        if (event == null || event.getAssetType() == null) {
            return;
        }

        AssetType assetType = event.getAssetType();
        EventOperation operation = event.getOperation();
        String assetTypeId = assetType.getObjId();
        String type = assetType.getType();

        log.info("接收到资产类型事件: type={}, assetTypeId={}, operation={}, status={}",
                type, assetTypeId, operation, assetType.getStatus());

        try {
            switch (operation) {
                case CREATE:
                    handleCreateEvent(assetType);
                    break;
                case UPDATE:
                    handleUpdateEvent(assetType);
                    break;
                case DELETE:
                    handleDeleteEvent(assetType);
                    break;
                default:
                    log.warn("未知的事件操作类型: {}", operation);
            }
        } catch (Exception e) {
            log.error("处理资产类型事件失败: type={}, assetTypeId={}, operation={}",
                    type, assetTypeId, operation, e);
        }
    }

    /**
     * 处理创建事件
     */
    protected void handleCreateEvent(AssetType assetType) {
        String assetTypeId = assetType.getObjId();
        String type = assetType.getType();
        DataCollectConfig config = JSONObject.parseObject(assetType.getDataCollectConfig().toJSONString(),DataCollectConfig.class);

        // 检查是否配置了数据采集
        if (config == null || config.getTrigger() == null) {
            log.debug("资产类型未配置数据采集: type={}, assetTypeId={}", type, assetTypeId);
            return;
        }

        // 只有在online状态才创建定时任务
        if (OnlineState.Online.getCode().equals(assetType.getStatus())) {
            String cronExpression = config.getTrigger().getValue();
            boolean success = quartzJobManager.addOrUpdateJob(assetTypeId, cronExpression);
            if (success) {
                log.info("创建数据采集定时任务成功: type={}, assetTypeId={}, cron={}",
                        type, assetTypeId, cronExpression);
            } else {
                log.error("创建数据采集定时任务失败: type={}, assetTypeId={}", type, assetTypeId);
            }
        } else {
            log.debug("资产类型状态不是online，跳过创建定时任务: type={}, assetTypeId={}, status={}",
                    type, assetTypeId, assetType.getStatus());
        }
    }

    /**
     * 处理更新事件
     */
    protected void handleUpdateEvent(AssetType assetType) {
        String assetTypeId = assetType.getObjId();
        String type = assetType.getType();
        String status = assetType.getStatus();
        DataCollectConfig config = JSONObject.parseObject(assetType.getDataCollectConfig().toJSONString(),DataCollectConfig.class);

        // 检查是否配置了数据采集
        if (config == null || config.getTrigger() == null) {
            log.debug("资产类型未配置数据采集，删除可能存在的定时任务: type={}, assetTypeId={}",
                    type, assetTypeId);
            // 如果之前配置了数据采集，现在取消了，需要删除定时任务
            if (quartzJobManager.jobExists(assetTypeId)) {
                quartzJobManager.deleteJob(assetTypeId);
            }
            return;
        }

        String cronExpression = config.getTrigger().getValue();

        // 根据状态决定操作
        if (OnlineState.Online.getCode().equals(status)) {
            // online状态：启动或更新定时任务
            boolean success = quartzJobManager.addOrUpdateJob(assetTypeId, cronExpression);
            if (success) {
                log.info("启动/更新数据采集定时任务成功: type={}, assetTypeId={}, cron={}",
                        type, assetTypeId, cronExpression);
            } else {
                log.error("启动/更新数据采集定时任务失败: type={}, assetTypeId={}",
                        type, assetTypeId);
            }
        } else if (OnlineState.Offline.getCode().equals(status) || OnlineState.Draft.getCode().equals(status)) {
            // offline/draft状态：暂停定时任务
            if (quartzJobManager.jobExists(assetTypeId)) {
                boolean success = quartzJobManager.pauseJob(assetTypeId);
                if (success) {
                    log.info("暂停数据采集定时任务成功: type={}, assetTypeId={}", type, assetTypeId);
                } else {
                    log.error("暂停数据采集定时任务失败: type={}, assetTypeId={}", type, assetTypeId);
                }
            }
        }
    }

    /**
     * 处理删除事件
     */
    protected void handleDeleteEvent(AssetType assetType) {
        String assetTypeId = assetType.getObjId();
        String type = assetType.getType();

        // 删除定时任务
        if (quartzJobManager.jobExists(assetTypeId)) {
            boolean success = quartzJobManager.deleteJob(assetTypeId);
            if (success) {
                log.info("删除数据采集定时任务成功: type={}, assetTypeId={}", type, assetTypeId);
            } else {
                log.error("删除数据采集定时任务失败: type={}, assetTypeId={}", type, assetTypeId);
            }
        } else {
            log.debug("定时任务不存在，无需删除: type={}, assetTypeId={}", type, assetTypeId);
        }
    }
}
