package com.digiwin.athena.controller.assembly;

import com.digiwin.athena.dto.ResultDto;
import com.digiwin.athena.dto.assembly.application.ApplicationDataRespDTO;
import com.digiwin.athena.dto.assembly.application.ApplicationDataSaveDTO;
import com.digiwin.athena.service.assembly.ApplicationDataService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.NotBlank;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

/**
 * 应用管理
 */
@Slf4j
@RestController
@RequestMapping("/assembly/application/")
@RequiredArgsConstructor
@Validated
@Tag(name = "应用管理", description = "提供应用数据的保存和查询接口")
public class AssemblyApplicationController {

    private final ApplicationDataService applicationDataService;

    /**
     * 保存应用数据
     */
    @PostMapping("/data/save")
    @Operation(summary = "保存应用数据", description = "保存或更新应用的解决方案数据")
    public ResultDto<Void> save(@Valid @RequestBody ApplicationDataSaveDTO saveDTO) {
        log.info("收到保存应用数据请求: appCode={}", saveDTO.getAppCode());
        applicationDataService.save(saveDTO);
        return ResultDto.createSuccess(null);
    }

    /**
     * 根据应用编码查询应用数据
     */
    @GetMapping("/data/byCode/{appCode}")
    @Operation(summary = "查询应用数据", description = "根据应用编码查询应用的解决方案数据，不存在时返回null")
    public ResultDto<ApplicationDataRespDTO> getByCode(@NotBlank @PathVariable String appCode) {
        log.debug("收到查询应用数据请求: appCode={}", appCode);
        ApplicationDataRespDTO respDTO = applicationDataService.getByAppCode(appCode);
        return ResultDto.createSuccess(respDTO);
    }
}
