package com.digiwin.athena.datacollect.script;

import cn.hutool.core.convert.Convert;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;
import java.util.Map;

/**
 * 脚本执行器
 * 只负责执行脚本，参数封装和结果解析由外部处理
 */
@Slf4j
@Component
public class ScriptExecutor {

    private final ScriptEngineManager scriptEngineManager;
    private final ObjectMapper objectMapper;

    public ScriptExecutor(ObjectMapper objectMapper) {
        this.scriptEngineManager = new ScriptEngineManager();
        this.objectMapper = objectMapper;
    }

    /**
     * 通用脚本执行方法
     * 支持泛型返回类型，自动处理类型转换
     *
     * @param script      脚本内容
     * @param params      脚本参数（key-value形式）
     * @param returnClass 期望的返回类型
     * @param <T>         返回类型泛型
     * @return 脚本执行结果
     */
    public <T> T execute(String script, Map<String, Object> params, Class<T> returnClass) {
        if (script == null || script.trim().isEmpty()) {
            return null;
        }

        try {
            ScriptEngine engine = createEngine();

            // 设置脚本输入参数
            if (params != null) {
                for (Map.Entry<String, Object> entry : params.entrySet()) {
                    engine.put(entry.getKey(), entry.getValue());
                }
            }

            engine.setContext(null);
            // TODO pzz performance 添加scriptId，优化脚本执行性能，比如缓存引擎实例等
            // TODO pzz performance 优化调试体验，输出脚本到文件中再执行/attach源码，以此支持报错时返回行号
            // 执行脚本
            Object result = engine.eval(script);

            // 类型转换
            return convertResult(result, returnClass);

        } catch (ScriptException e) {
            log.error("执行脚本失败", e);
            throw new RuntimeException("执行脚本失败: " + e.getMessage(), e);
        }
    }

    /**
     * 转换脚本执行结果
     * 如果返回string则尝试ObjectMapper进行json反序列化，否则使用Convert转换类型
     */
    @SuppressWarnings("unchecked")
    public <T> T convertResult(Object result, Class<T> targetClass) {
        if (result == null) {
            return null;
        }

        // 如果结果已经是目标类型，直接返回
        if (targetClass.isInstance(result)) {
            return (T) result;
        }

        // 如果结果是String，尝试JSON反序列化
        if (result instanceof String) {
            String strResult = (String) result;
            try {
                return objectMapper.readValue(strResult, targetClass);
            } catch (Exception e) {
                log.debug("JSON反序列化失败，尝试使用Convert转换: {}", e.getMessage());
                // JSON反序列化失败，使用Convert转换
                return Convert.convert(targetClass, result);
            }
        }

        // 使用Convert进行类型转换
        return Convert.convert(targetClass, result);
    }

    /**
     * 创建脚本引擎
     * 优先使用Groovy，如果不可用则使用JavaScript
     */
    private ScriptEngine createEngine() {
        // 尝试使用Groovy引擎
        ScriptEngine engine = scriptEngineManager.getEngineByName("groovy");
        if (engine != null) {
            return engine;
        }

        // 回退到JavaScript引擎（如果可用）
        engine = scriptEngineManager.getEngineByName("javascript");
        if (engine != null) {
            return engine;
        }

        // 尝试Nashorn（JDK 8-14）
        engine = scriptEngineManager.getEngineByName("nashorn");
        if (engine != null) {
            return engine;
        }

        throw new IllegalStateException("没有可用的脚本引擎。请添加Groovy或JavaScript引擎依赖。");
    }

}
