package com.digiwin.athena.dao.mongodao.assembly.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import com.digiwin.athena.bo.assembly.SolutionPlan;
import com.digiwin.athena.constants.enums.assembly.AssemblyType;
import com.digiwin.athena.constants.enums.assembly.OnlineState;
import com.digiwin.athena.convertor.assembly.SolutionConvertor;
import com.digiwin.athena.dao.mongodao.assembly.DDesignerMongoDao;
import com.digiwin.athena.dao.mongodao.assembly.RSolutionMongoDao;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.digiwin.athena.dto.assembly.solution.RSolutionPlanPageQo;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanListDTO;
import com.digiwin.athena.mongodb.domain.application.ApplicationConfig;
import com.digiwin.athena.mongodb.repository.MongoRuntimeRepositoryDecorator;
import lombok.RequiredArgsConstructor;
import org.springframework.data.mongodb.core.mapping.FieldName;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Repository;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

@RequiredArgsConstructor
@Repository
public class RSolutionMongoDaoImpl implements RSolutionMongoDao {

    private static final SolutionConvertor CONVERTOR = SolutionConvertor.INSTANCE;
    private final MongoRuntimeRepositoryDecorator decorator;

    @Override
    public SolutionPlan selectById(String objectId) {
        ApplicationConfig entity = decorator.findById(objectId, ApplicationConfig.class);
        return CONVERTOR.toBO(entity);
    }

    @Override
    public Pagination<SolutionPlanListDTO> selectOnlineSolutionPage(PageReqCondition<RSolutionPlanPageQo> pageParam) {
        Query query = new Query();

        // 添加online状态条件
        query.addCriteria(Criteria.where(ApplicationConfig.Fields.status).is(OnlineState.Online.getCode()));
        // 未整合已有解决方案设计器前 先进行筛选，用来区分
        commonCondition(query);

        // 添加keyword查询条件
        RSolutionPlanPageQo condition = ObjectUtil.defaultIfNull(pageParam.getCondition(), RSolutionPlanPageQo::new);
        if (condition.getIsDynamic() != null) {
            Criteria dynamic = Criteria.where(ApplicationConfig.Fields.assemblyType);
            if (condition.getIsDynamic()) {
                dynamic.is(AssemblyType.Dynamic.getCode());
            } else {
                dynamic.is(AssemblyType.Static.getCode());
            }

            query.addCriteria(dynamic);
        }

        DDesignerMongoDao.addSearchNameOrType(query, condition.getSearchContent());

        return decorator
                .findWithPage(query, ApplicationConfig.class, pageParam)
                .convert(SolutionConvertor.INSTANCE::entityToListDTO);
    }

    @Override
    public void save(SolutionPlan solution) {
        ApplicationConfig entity = CONVERTOR.toEntity(solution);
        entity.prepareSave();

        decorator.save(entity);

        solution.setId(entity.getId());
    }

    @Override
    public boolean deleteById(String objectId) {
        Query query = Query.query(Criteria.where(FieldName.ID.name()).is(objectId));
        long count = decorator.remove(query, ApplicationConfig.class);
        return count > 0;
    }

    @Override
    public List<SolutionPlanListDTO> selectByAppType(Collection<Integer> appTypeList) {
        Query query = new Query();

        if (CollUtil.isNotEmpty(appTypeList)) {
            query.addCriteria(Criteria.where(ApplicationConfig.Fields.appType).in(appTypeList));
        }

        commonCondition(query);

        return decorator.find(query, ApplicationConfig.class).stream()
                .map(CONVERTOR::entityToListDTO)
                .collect(Collectors.toList());
    }

    protected void commonCondition(Query query) {
        // 未整合已有解决方案设计器前 先进行筛选，用来区分
        query.addCriteria(Criteria.where(ApplicationConfig.Fields.assemblyType).exists(true).ne(null));
    }
}
