package com.digiwin.athena.datacollect.job;

import lombok.extern.slf4j.Slf4j;
import org.quartz.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 *
 * Quartz任务管理器
 * 封装Quartz API，提供任务的动态管理
 */
@Slf4j
@Component
public class QuartzJobManager {

    private static final String JOB_GROUP = "AssetDataCollect";
    @Autowired
    private Scheduler scheduler;

    /**
     * 添加或更新定时任务
     *
     * @param assetTypeId    资产类型ID
     * @param cronExpression cron表达式
     * @return 是否成功
     */
    public boolean addOrUpdateJob(String assetTypeId, String cronExpression) {
        try {
            String jobName = buildJobName(assetTypeId);
            JobKey jobKey = JobKey.jobKey(jobName, JOB_GROUP);

            // 检查任务是否已存在
            if (scheduler.checkExists(jobKey)) {
                // 任务已存在,更新cron表达式
                return updateJobCron(assetTypeId, cronExpression);
            } else {
                // 任务不存在,创建新任务
                return addJob(assetTypeId, cronExpression);
            }
        } catch (Exception e) {
            log.error("添加或更新定时任务失败: assetTypeId={}, cron={}", assetTypeId, cronExpression, e);
            return false;
        }
    }

    /**
     * 添加定时任务
     *
     * @param assetTypeId    资产类型ID
     * @param cronExpression cron表达式
     * @return 是否成功
     */
    public boolean addJob(String assetTypeId, String cronExpression) {
        try {
            String jobName = buildJobName(assetTypeId);

            // 创建JobDetail
            JobDetail jobDetail = JobBuilder.newJob(AssetDataCollectJob.class)
                    .withIdentity(jobName, JOB_GROUP)
                    .usingJobData("assetTypeId", assetTypeId)
                    .storeDurably()
                    .build();

            // 创建Trigger
            CronTrigger trigger = TriggerBuilder.newTrigger()
                    .withIdentity(buildTriggerName(assetTypeId), JOB_GROUP)
                    .withSchedule(CronScheduleBuilder.cronSchedule(cronExpression))
                    .build();

            // 调度任务
            scheduler.scheduleJob(jobDetail, trigger);

            log.info("添加定时任务成功: assetTypeId={}, cron={}", assetTypeId, cronExpression);
            return true;
        } catch (Exception e) {
            log.error("添加定时任务失败: assetTypeId={}, cron={}", assetTypeId, cronExpression, e);
            return false;
        }
    }

    /**
     * 更新任务的cron表达式
     *
     * @param assetTypeId    资产类型ID
     * @param cronExpression 新的cron表达式
     * @return 是否成功
     */
    public boolean updateJobCron(String assetTypeId, String cronExpression) {
        try {
            TriggerKey triggerKey = TriggerKey.triggerKey(buildTriggerName(assetTypeId), JOB_GROUP);

            // 获取现有trigger
            CronTrigger oldTrigger = (CronTrigger) scheduler.getTrigger(triggerKey);
            if (oldTrigger == null) {
                log.warn("Trigger不存在,无法更新: assetTypeId={}", assetTypeId);
                return false;
            }

            // 创建新的trigger
            CronTrigger newTrigger = TriggerBuilder.newTrigger()
                    .withIdentity(triggerKey)
                    .withSchedule(CronScheduleBuilder.cronSchedule(cronExpression))
                    .build();

            // 重新调度
            scheduler.rescheduleJob(triggerKey, newTrigger);

            log.info("更新定时任务cron成功: assetTypeId={}, cron={}", assetTypeId, cronExpression);
            return true;
        } catch (Exception e) {
            log.error("更新定时任务cron失败: assetTypeId={}, cron={}", assetTypeId, cronExpression, e);
            return false;
        }
    }

    /**
     * 暂停任务
     *
     * @param assetTypeId 资产类型ID
     * @return 是否成功
     */
    public boolean pauseJob(String assetTypeId) {
        try {
            JobKey jobKey = JobKey.jobKey(buildJobName(assetTypeId), JOB_GROUP);
            scheduler.pauseJob(jobKey);

            log.info("暂停定时任务成功: assetTypeId={}", assetTypeId);
            return true;
        } catch (Exception e) {
            log.error("暂停定时任务失败: assetTypeId={}", assetTypeId, e);
            return false;
        }
    }

    /**
     * 恢复任务
     *
     * @param assetTypeId 资产类型ID
     * @return 是否成功
     */
    public boolean resumeJob(String assetTypeId) {
        try {
            JobKey jobKey = JobKey.jobKey(buildJobName(assetTypeId), JOB_GROUP);
            scheduler.resumeJob(jobKey);

            log.info("恢复定时任务成功: assetTypeId={}", assetTypeId);
            return true;
        } catch (Exception e) {
            log.error("恢复定时任务失败: assetTypeId={}", assetTypeId, e);
            return false;
        }
    }

    /**
     * 删除任务
     *
     * @param assetTypeId 资产类型ID
     * @return 是否成功
     */
    public boolean deleteJob(String assetTypeId) {
        try {
            JobKey jobKey = JobKey.jobKey(buildJobName(assetTypeId), JOB_GROUP);
            scheduler.deleteJob(jobKey);

            log.info("删除定时任务成功: assetTypeId={}", assetTypeId);
            return true;
        } catch (Exception e) {
            log.error("删除定时任务失败: assetTypeId={}", assetTypeId, e);
            return false;
        }
    }

    /**
     * 立即执行任务(不影响原定时计划)
     *
     * @param assetTypeId 资产类型ID
     * @return 是否成功
     */
    public boolean triggerJob(String assetTypeId) {
        try {
            JobKey jobKey = JobKey.jobKey(buildJobName(assetTypeId), JOB_GROUP);
            scheduler.triggerJob(jobKey);

            log.info("手动触发任务成功: assetTypeId={}", assetTypeId);
            return true;
        } catch (Exception e) {
            log.error("手动触发任务失败: assetTypeId={}", assetTypeId, e);
            return false;
        }
    }

    /**
     * 检查任务是否存在
     *
     * @param assetTypeId 资产类型ID
     * @return 是否存在
     */
    public boolean jobExists(String assetTypeId) {
        try {
            JobKey jobKey = JobKey.jobKey(buildJobName(assetTypeId), JOB_GROUP);
            return scheduler.checkExists(jobKey);
        } catch (Exception e) {
            log.error("检查任务是否存在失败: assetTypeId={}", assetTypeId, e);
            return false;
        }
    }

    /**
     * 获取任务状态
     *
     * @param assetTypeId 资产类型ID
     * @return 任务状态 (NORMAL, PAUSED, COMPLETE, ERROR, BLOCKED, NONE)
     */
    public String getJobState(String assetTypeId) {
        try {
            TriggerKey triggerKey = TriggerKey.triggerKey(buildTriggerName(assetTypeId), JOB_GROUP);
            Trigger.TriggerState state = scheduler.getTriggerState(triggerKey);
            return state.name();
        } catch (Exception e) {
            log.error("获取任务状态失败: assetTypeId={}", assetTypeId, e);
            return "ERROR";
        }
    }

    /**
     * 构建Job名称
     */
    private String buildJobName(String assetTypeId) {
        return "AssetDataCollect_" + assetTypeId;
    }

    /**
     * 构建Trigger名称
     */
    private String buildTriggerName(String assetTypeId) {
        return "Trigger_" + assetTypeId;
    }
}
