package com.digiwin.athena.service.asset.impl;

import cn.hutool.core.util.IdUtil;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.actuator.HttpActuator;
import com.digiwin.athena.actuator.HttpExecuteConfig;
import com.digiwin.athena.actuator.ReqPlaceholder;
import com.digiwin.athena.base.BaseConstant;
import com.digiwin.athena.config.AssetCenterRegisterConfig;
import com.digiwin.athena.config.BranchConfig;
import com.digiwin.athena.constant.Constant;
import com.digiwin.athena.dao.mongodao.ApplicationMongoDao;
import com.digiwin.athena.dao.mongodao.asset.AssetDefinitionMongoDao;
import com.digiwin.athena.dao.mongodao.asset.AssetDepInfoMongoDao;
import com.digiwin.athena.dao.mongodao.asset.AssetMongoDao;
import com.digiwin.athena.dao.mongodao.asset.AssetRelationMongoDao;
import com.digiwin.athena.dao.mongodao.assetType.RAssetTypeMongoDao;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.digiwin.athena.domain.pagedesign.TenantInfo;
import com.digiwin.athena.dto.asset.AssetBatchOpenReqDto;
import com.digiwin.athena.dto.asset.AssetCenterInfoResDto;
import com.digiwin.athena.dto.asset.AssetDetailResDto;
import com.digiwin.athena.dto.asset.AssetInfoDto;
import com.digiwin.athena.dto.asset.AssetInfoResDto;
import com.digiwin.athena.dto.asset.AssetOpenReqDto;
import com.digiwin.athena.dto.asset.AssetRelationAppInfoResDto;
import com.digiwin.athena.dto.asset.AssetRelationInfoDto;
import com.digiwin.athena.dto.asset.DeleteAssetReqDto;
import com.digiwin.athena.dto.asset.DependentAssetResDto;
import com.digiwin.athena.dto.asset.DesignerAssetListReqDto;
import com.digiwin.athena.dto.asset.CreateAssetReqDto;
import com.digiwin.athena.dto.asset.DepAssetResDto;
import com.digiwin.athena.dto.asset.UseAssetListReqDto;
import com.digiwin.athena.enums.VersionStatusEnum;
import com.digiwin.athena.http.asset.DataPermissionRequestInfoDto;
import com.digiwin.athena.http.asset.PermissionConditionResDto;
import com.digiwin.athena.http.asset.PermissionResDto;
import com.digiwin.athena.http.asset.ThirdPartyApiHelper;
import com.digiwin.athena.mongodb.domain.DependencyRule;
import com.digiwin.athena.mongodb.domain.ExecuteConfig;
import com.digiwin.athena.mongodb.domain.application.Application;
import com.digiwin.athena.mongodb.domain.application.Asset;
import com.digiwin.athena.mongodb.domain.application.AssetDefinition;
import com.digiwin.athena.mongodb.domain.application.AssetDepInfo;
import com.digiwin.athena.mongodb.domain.application.AssetRelation;
import com.digiwin.athena.mongodb.domain.assetType.AssetType;
import com.digiwin.athena.mysql.domain.user.UserPO;
import com.digiwin.athena.service.AssetInit;
import com.digiwin.athena.service.asset.AssetBusinessDataAction;
import com.digiwin.athena.service.asset.AssetCenter;
import com.digiwin.athena.service.codeStandard.BuildGeneratorCodeUtil;
import com.digiwin.athena.utils.CurThreadInfoUtils;
import com.digiwin.athena.utils.JSUtil;
import com.digiwin.athena.utils.user.UserHelper;
import jakarta.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.stream.Collectors;

@Service
@Slf4j
public class AssetCenterImpl implements AssetCenter {

    private AssetMongoDao assetMongoDao;
    private BuildGeneratorCodeUtil buildGeneratorCodeUtil;

    private AssetRelationMongoDao assetRelationMongoDao;

    private AssetDefinitionMongoDao assetDefinitionMongoDao;

    private AssetCenterRegisterConfig assetCenterRegisterConfig;
    @Resource
    private ThreadPoolExecutor assetCenterPool;
    private ApplicationMongoDao applicationMongoDao;
    @Autowired
    private RAssetTypeMongoDao rAssetTypeMongoDao;

    @Autowired
    @Lazy
    private AssetInit assetInit;
    @Autowired
    private ThirdPartyApiHelper thirdPartyApiHelper;
    @Autowired
    private AssetDepInfoMongoDao assetDepInfoMongoDao;
    @Autowired
    private HttpActuator httpActuator;

    @Autowired
    public void setApplicationMongoDao(ApplicationMongoDao applicationMongoDao) {
        this.applicationMongoDao = applicationMongoDao;
    }

    @Autowired
    public void setAssetCenterRegisterConfig(AssetCenterRegisterConfig assetCenterRegisterConfig) {
        this.assetCenterRegisterConfig = assetCenterRegisterConfig;
    }

    @Autowired
    public void setAssetDefinitionMongoDao(AssetDefinitionMongoDao assetDefinitionMongoDao) {
        this.assetDefinitionMongoDao = assetDefinitionMongoDao;
    }

    @Autowired
    public void setAssetRelationMongoDao(AssetRelationMongoDao assetRelationMongoDao) {
        this.assetRelationMongoDao = assetRelationMongoDao;
    }

    @Autowired
    public void setBuildGeneratorCodeUtil(BuildGeneratorCodeUtil buildGeneratorCodeUtil) {
        this.buildGeneratorCodeUtil = buildGeneratorCodeUtil;
    }

    @Autowired
    public void setAssetMongoDao(AssetMongoDao assetMongoDao) {
        this.assetMongoDao = assetMongoDao;
    }

    @Override
    public String saveDTDAsset(AssetInfoDto assetInfoDto) {
        Assert.isTrue(assetInfoDto.judgeSaveRequiredFields(), "需要补充必填字段");

        //判断资产定义表是否存在对应的资产
        AssetDefinition oldAssetDefinition = assetDefinitionMongoDao.selectByApplicationAndCode(assetInfoDto.getCode(), assetInfoDto.getApplication());

        if (oldAssetDefinition == null) {
            //兼容已经存在的asset数据
            Asset existAsset = assetMongoDao.selectAssetIdByCodeAndApplication(assetInfoDto.getCode(), assetInfoDto.getApplication());

            oldAssetDefinition = assetInfoDto.convertAssetDefinition();
            oldAssetDefinition.setAssetId(existAsset != null ? existAsset.getAssetId() : buildGeneratorCodeUtil.createAssetId(assetInfoDto.getCode(), AssetDefinition.AssetTypeEnum.DTD.getCode()));
            oldAssetDefinition.setType(AssetDefinition.AssetTypeEnum.DTD.getCode());
            oldAssetDefinition.setPublisher(CurThreadInfoUtils.getCurTokenTenant().getTenantId());
            oldAssetDefinition.setPublisherName(CurThreadInfoUtils.getCurTokenTenant().getTenantName());

            UserHelper.fillCreateInfo(oldAssetDefinition);
            UserHelper.fillEditInfo(oldAssetDefinition);
            assetDefinitionMongoDao.insert(oldAssetDefinition);
        }

        //如果当前的版本是生效版本，先将其他的版本置为失效
        if (VersionStatusEnum.RUN_STATE == assetInfoDto.assetAdpStatus()) {
            assetMongoDao.updateDraftAdpStatusByAssetId(UserHelper.getCurrentUser().getName(), oldAssetDefinition.getAssetId());
        }

        Asset asset = assetMongoDao.selectByAssetIdAndSourceBranchAndAdpVersion(oldAssetDefinition.getAssetId(), assetInfoDto.getSourceBranch(), assetInfoDto.getAdpVersion());
        if (asset == null) {
            asset = assetInfoDto.convertAsset();
            asset.setAssetId(oldAssetDefinition.getAssetId());
            asset.setType(oldAssetDefinition.getType());

            UserHelper.fillCreateInfo(asset);
            UserHelper.fillEditInfo(asset);

            assetMongoDao.insert(asset);
        } else {
            asset.setContent(assetInfoDto.getContent());
            if (assetInfoDto.getBusinessData() != null) {
                asset.setBusinessData(JSONObject.toJSONString(assetInfoDto.getBusinessData()));
            }

            asset.getManageContext().setAdpStatus(assetInfoDto.assetAdpStatus().getCode());
            asset.getShowFields().setAdpRemarkNumber(assetInfoDto.getAdpRemarkNumber());
            UserHelper.fillEditInfo(asset);
            assetMongoDao.save(asset);
        }

        AssetType assetType = rAssetTypeMongoDao.selectByType(AssetDefinition.AssetTypeEnum.DTD.getCode());
        resolveDepInfo(asset,assetType, null);

        return oldAssetDefinition.getAssetId();
    }

    @Override
    public String saveWorkbenchAsset(AssetInfoDto assetInfoDto) {
        Assert.isTrue(assetInfoDto.judgeSaveRequiredFields(), "需要补充必填字段");

        // 判断资产定义表是否存在对应的资产
        AssetDefinition oldAssetDefinition = assetDefinitionMongoDao.selectByApplicationAndCode(assetInfoDto.getCode(), assetInfoDto.getApplication());

        if (oldAssetDefinition == null) {
            //兼容已经存在的asset数据
            Asset existAsset = assetMongoDao.selectAssetIdByCodeAndApplication(assetInfoDto.getCode(), assetInfoDto.getApplication());

            oldAssetDefinition = assetInfoDto.convertAssetDefinition();
            oldAssetDefinition.setAssetId(
                    existAsset != null ? existAsset.getAssetId() : buildGeneratorCodeUtil.createAssetId(
                            assetInfoDto.getCode(), AssetDefinition.AssetTypeEnum.WORKBENCH.getCode()));
            oldAssetDefinition.setType(AssetDefinition.AssetTypeEnum.WORKBENCH.getCode());
            oldAssetDefinition.setPublisher(CurThreadInfoUtils.getCurTokenTenant().getTenantId());
            oldAssetDefinition.setPublisherName(CurThreadInfoUtils.getCurTokenTenant().getTenantName());

            UserHelper.fillCreateInfo(oldAssetDefinition);
            UserHelper.fillEditInfo(oldAssetDefinition);
            // 新增资产定义
            assetDefinitionMongoDao.insert(oldAssetDefinition);
        }

        // 如果当前的版本是生效版本，先将其他的版本置为失效
        if (VersionStatusEnum.RUN_STATE.getCode().equals(assetInfoDto.assetAdpStatus())) {
            assetMongoDao.updateDraftAdpStatusByAssetId(UserHelper.getCurrentUser().getName(), oldAssetDefinition.getAssetId());
        }

        // 查询资产表
        Asset asset = assetMongoDao.selectByAssetIdAndSourceBranchAndAdpVersion(oldAssetDefinition.getAssetId(), assetInfoDto.getSourceBranch(), assetInfoDto.getAdpVersion());
        if (asset == null) {
            asset = assetInfoDto.convertAsset();
            asset.setAssetId(oldAssetDefinition.getAssetId());
            asset.setType(oldAssetDefinition.getType());

            UserHelper.fillCreateInfo(asset);
            UserHelper.fillEditInfo(asset);

            assetMongoDao.insert(asset);
        } else {
            asset.setContent(assetInfoDto.getContent());
            asset.setAssetName(assetInfoDto.getAssetName());
            if (assetInfoDto.getBusinessData() != null) {
                asset.setBusinessData(JSONObject.toJSONString(assetInfoDto.getBusinessData()));
            }

            asset.getManageContext().setAdpStatus(assetInfoDto.assetAdpStatus().getCode());
            asset.getShowFields().setAdpRemarkNumber(assetInfoDto.getAdpRemarkNumber());
            UserHelper.fillEditInfo(asset);
            assetMongoDao.save(asset);
        }

        return oldAssetDefinition.getAssetId();
    }

    @Override
    public void bindRelation(AssetRelationInfoDto assetRelationInfoDto) {
        Assert.isTrue(assetRelationInfoDto.judgeRequiredFields(), "需要补充必填字段");

        List<Asset> assets = assetMongoDao.selectByAssetId(assetRelationInfoDto.getAssetId());
        Assert.isTrue(!assets.isEmpty(), "资产没有版本，无法引用");
        Asset asset = assets.get(0);

        //检查这个版本是否已经绑定过这个资产
        AssetRelation existAssetRelation = assetRelationMongoDao.selectByAssetIdAndAdpVersionAndUserBranchAndApplication(assetRelationInfoDto);
        Assert.isNull(existAssetRelation,"已经引用过这个资产");

        AssetDefinition assetDefinition = assetDefinitionMongoDao.selectByAssetId(assetRelationInfoDto.getAssetId());
        //兼容已经存在asset老数据，补数据
        if (assetDefinition == null) {
            Application application = applicationMongoDao.selectByCode(asset.getManageContext().getApplication());
            assetDefinition = AssetInfoDto.create(asset).convertAssetDefinition();
            assetDefinition.setAssetId(asset.getAssetId());
            assetDefinition.setType(AssetDefinition.AssetTypeEnum.DTD.getCode());
            assetDefinition.setPublisherName(Constant.SYSTEM);
            assetDefinition.setPublisher(Constant.SYSTEM);
            assetDefinition.setAdpTenantId(application == null && StringUtils.isEmpty(application.getTenantId()) ? Constant.SYSTEM : application.getTenantId());

            UserHelper.fillCreateInfo(assetDefinition);
            UserHelper.fillEditInfo(assetDefinition);
            assetDefinitionMongoDao.insert(assetDefinition);
        }
        Assert.isTrue(assetDefinition != null, assetRelationInfoDto.getAssetId() + "不存在");

        AssetRelation assetRelation = assetRelationInfoDto.convertAssetRelation();
        assetRelation.setBizType(assetDefinition.getType());
        assetRelation.setType(asset.getManageContext().getApplication().equals(assetRelationInfoDto.getApplication()) ?
                AssetRelation.BelongTypeEnum.OWN_TYPE.getType() : AssetRelation.BelongTypeEnum.REF_TYPE.getType());

        assetRelationMongoDao.insert(assetRelation);
        if (!asset.getManageContext().getApplication().equals(assetRelationInfoDto.getApplication())) {
//            Long count = assetRelationMongoDao.selectRefCountByAssetIdAndApplication(assetRelationInfoDto.getAssetId(), assetRelationInfoDto.getApplication());
//            assetDefinition.setReferencedCount(count.intValue());
            assetDefinition.setReferencedCount(assetDefinition.getReferencedCount()+1);
            assetDefinitionMongoDao.save(assetDefinition);
        }
    }

    @Override
    public List<Asset> getAssetList( String application,  String adpVersion,  String branch) {
        return assetMongoDao.selectByApplicationAndAdpVersionAndBranch(application, adpVersion, branch);
    }

    @Override
    public List<Asset> getRefDTDAssetList( String application,  String adpVersion,  String branch) {
        AssetRelationInfoDto assetRelationInfoDto = AssetRelationInfoDto.create(null, application, adpVersion, branch);
        List<AssetRelation> bindDTDRefRelations = getBindDTDRefRelation(assetRelationInfoDto);
        List<String> assetIds = bindDTDRefRelations.stream().map(AssetRelation::getAssetId).collect(Collectors.toList());

        if (assetIds.isEmpty()) {
            return new ArrayList<>();
        } else {
            return getEffectAssets(assetIds);
        }
    }

    

    @Override
    public Boolean removeRefRelation(List<String> assetIds, String adpVersion, String branch, String application) {
        if (CollectionUtils.isEmpty(assetIds)) {
            return true;
        }
        assetRelationMongoDao.deleteByAssetIdsAndVersionAndBranchAndApp(assetIds, adpVersion, branch, application);
        return true;
    }

    @Override
    public List<AssetRelation> getBindDTDRefRelation(AssetRelationInfoDto assetRelationInfoDto) {
        List<AssetRelation> assetRelations = assetRelationMongoDao.select(assetRelationInfoDto, AssetDefinition.AssetTypeEnum.DTD);
        return assetRelations;
    }

    @Override
    public Boolean removeReferenceRel(AssetRelationInfoDto assetRelationInfoDto) {

        assetRelationMongoDao.delete(assetRelationInfoDto);

        return true;
    }

    @Override
    public Asset getEffectAsset(String assetId) {
        return assetMongoDao.selectEffectByAssetId(assetId);
    }


    @Override
    public List<Asset> getEffectAssets(List<String> assetIds) {
        return assetMongoDao.selectEffectByAssetIds(assetIds);
    }

    @Override
    public Map<String,Asset> getEffectAssetVersion(List<String> assetIds) {
        Map<String, Asset> res = new HashMap<>();
        if (CollectionUtils.isEmpty(assetIds)){
            return res;
        }
        List<Asset> assets = assetMongoDao.selectEffectVersionByAssetIds(assetIds);
        for (Asset asset : assets) {
            res.put(asset.getAssetId(),asset);
        }
        return res;
    }
    @Override
    public Asset getHistoryAsset(String assetId, String sourceBranch, String adpVersion) {

        return assetMongoDao.selectByAssetIdAndSourceBranchAndAdpVersion(assetId, sourceBranch, adpVersion);
    }

    @Override
    public Boolean availableAsset(AssetInfoDto assetInfoDto) {
        String assetId = saveDTDAsset(assetInfoDto);
        updateStatus(AssetDefinition.AVAILABLE_STATUS, assetId);

        return true;
    }

    @Override
    public Boolean updateAssetDefinition(AssetOpenReqDto assetOpenReqDto) {
        if (assetOpenReqDto.getCanReference()){
            Asset asset = assetMongoDao.selectEffectByAssetId(assetOpenReqDto.getAssetId());
            Assert.notNull(asset,"该资产没有生效版本，无法设置引用");
        }

        AssetDefinition assetDefinition = assetDefinitionMongoDao.selectByAssetId(assetOpenReqDto.getAssetId());
        Assert.notNull(assetDefinition, assetDefinition.getAssetId() + "不存在对应的资产");
        BeanUtils.copyProperties(assetOpenReqDto, assetDefinition);
        assetDefinition.getPermission().setTenantInfos(assetOpenReqDto.getTenantInfos());

        UserHelper.fillEditInfo(assetDefinition);

        assetDefinitionMongoDao.save(assetDefinition);

        try {
            if (AssetDefinition.AVAILABLE_STATUS.equals(assetDefinition.getStatus())&& !assetDefinition.getCopyableItems().isEmpty()){
                List<Asset> assetBasicInfoList = assetMongoDao.selectBasicInfoByObjectId(assetDefinition.getCopyableItems());

                updateAssetData(assetBasicInfoList);
            }
        }catch (InterruptedException e){
            Thread.currentThread().interrupt();
            throw new RuntimeException("同步可复制版本异常:"+e.getMessage(),e);
        }
        catch (Exception e) {
            throw new RuntimeException("同步可复制版本异常:"+e.getMessage(),e);
        }

        return true;
    }

    @Override
    public Boolean availableAsset(AssetOpenReqDto assetOpenReqDto) {
        try {
            if (assetOpenReqDto.getCanReference()){
                Asset asset = assetMongoDao.selectEffectByAssetId(assetOpenReqDto.getAssetId());
                Assert.notNull(asset,"该资产没有生效版本，无法设置引用");
            }

            AssetDefinition assetDefinition = assetDefinitionMongoDao.selectByAssetId(assetOpenReqDto.getAssetId());
            Assert.notNull(assetDefinition, assetDefinition.getAssetId() + "不存在对应的资产");

            BeanUtils.copyProperties(assetOpenReqDto, assetDefinition);
            assetDefinition.setStatus(AssetDefinition.AVAILABLE_STATUS);
            UserHelper.fillEditInfo(assetDefinition);

            assetDefinitionMongoDao.save(assetDefinition);
            if (!assetOpenReqDto.getCopyableItems().isEmpty()) {
                Map<String, AssetBusinessDataAction> assetBusinessDataActions = assetCenterRegisterConfig.getAssetBusinessDataActions();
                if (assetBusinessDataActions.containsKey(assetDefinition.getType())) {

                    List<Asset> assetBasicInfoList = assetMongoDao.selectBasicInfoByObjectId(assetOpenReqDto.getCopyableItems());

                    updateAssetData(assetBasicInfoList);

                } else {
                    log.warn(assetDefinition.getType() + "资产，没有注册资产数据执行逻辑");
                }

            }
        } catch (Exception e) {
            log.error("availableAsset exception:"+e.getMessage(),e);
            Thread.currentThread().interrupt();
            throw new RuntimeException(e.getMessage());
        }
        return true;
    }

    @Override
    public Boolean batchAvailableAsset(AssetBatchOpenReqDto assetBatchOpenReqDto) {
        List<String> assetIds = assetBatchOpenReqDto.getAssetIds();
        List<AssetDefinition> assetDefinitions = assetDefinitionMongoDao.selectByAssetIds(assetIds);

        if (assetBatchOpenReqDto.getCanReference()){
            for (AssetDefinition assetDefinition : assetDefinitions) {
                Asset asset = assetMongoDao.selectEffectByAssetId(assetDefinition.getAssetId());
                Assert.notNull(asset,assetDefinition.getAssetId()+"该资产没有生效版本，无法设置引用");
            }
        }

        Assert.isTrue(!assetDefinitions.isEmpty(), "没有查询到资产定义数据");
        try {
            for (AssetDefinition assetDefinition : assetDefinitions) {
                BeanUtils.copyProperties(assetBatchOpenReqDto, assetDefinition);
                assetDefinition.setStatus(AssetDefinition.AVAILABLE_STATUS);


                if (assetDefinition.getCopyableItems().isEmpty()) {
                    Asset asset = assetMongoDao.selectLatestVersionByAssetIdAndSourceBranchAndApplication(assetDefinition.getAssetId(), BranchConfig.DEV_BRANCH_NAME, assetDefinition.getManageContext().getApplication());
                    if (asset != null) {
                        assetDefinition.getCopyableItems().add(asset.getObjectId());
                    }
                }else{

                }

                UserHelper.fillEditInfo(assetDefinition);
                assetDefinitionMongoDao.save(assetDefinition);
            }

            if (assetBatchOpenReqDto.getCanCopy()) {
                List<String> allAssetObjId = new ArrayList<>();
                for (AssetDefinition assetDefinition : assetDefinitions) {
                    List<String> copyableItems = assetDefinition.getCopyableItems();
                    if (!copyableItems.isEmpty()) {
                        allAssetObjId.addAll(copyableItems);
                    }
                }

                if (!allAssetObjId.isEmpty()) {
                    List<Asset> assetBasicInfoList = assetMongoDao.selectBasicInfoByObjectId(allAssetObjId);

                    Map<String, AssetBusinessDataAction> assetBusinessDataActions = assetCenterRegisterConfig.getAssetBusinessDataActions();
                    for (Asset asset : assetBasicInfoList) {
                        if (assetBusinessDataActions.containsKey(asset.getType())) {
                            updateAssetData(assetBasicInfoList);
                        } else {
                            log.warn(asset.getType() + "资产，没有注册资产数据执行逻辑");
                        }
                    }
                }
            }
        } catch (Exception e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }

        return true;
    }

    private void updateAssetData(List<Asset> assetBasicInfoList) throws InterruptedException, ExecutionException {
        List<CompletableFuture<Boolean>> completableFutures = new ArrayList<>();
        //更新对应的版本的业务数据
        for (Asset asset : assetBasicInfoList) {
            CompletableFuture<Boolean> future = asyncUpdateAssetData(asset);
            completableFutures.add(future);
        }
        CompletableFuture[] array = completableFutures.toArray(new CompletableFuture[0]);
        CompletableFuture.allOf(array);

        List<String> errorAssetIds = new ArrayList<>();
        for (int i = 0; i < completableFutures.size(); i++) {
            if (!completableFutures.get(i).get()) {
                Asset asset = assetBasicInfoList.get(i);
                errorAssetIds.add(asset.getAssetId());
            }
        }
        Assert.isTrue(errorAssetIds.isEmpty(), String.join(BaseConstant.COMMA_SEPARATOR, errorAssetIds) + "资产同步失败");
    }

    private CompletableFuture<Boolean> asyncUpdateAssetData(Asset asset) {
        UserPO currentUser = UserHelper.getCurrentUser();

        Map<String, AssetBusinessDataAction> assetBusinessDataActions = assetCenterRegisterConfig.getAssetBusinessDataActions();

        return CompletableFuture.supplyAsync(() -> {

            try {
                AssetBusinessDataAction assetBusinessDataAction = assetBusinessDataActions.get(asset.getType());
                AssetInfoDto assetBusinessDataDto = assetBusinessDataAction.loadBusinessData(AssetInfoDto.create(asset));

                if (assetBusinessDataDto != null) {
                    asset.setContent(assetBusinessDataDto.getContent());
                    asset.setBusinessData(JSONObject.toJSONString(assetBusinessDataDto.getBusinessData()));

                    asset.setEditBy(currentUser.getName());
                    asset.setEditDate(new Date());
                    assetMongoDao.save(asset);

                    return true;
                } else {
                    log.warn(String.format("资产objectId:%s,返回的业务对象为null", asset.getObjectId()));
                    return false;
                }
            } catch (Exception e) {
                log.error("syncAssetData error:" + e.getMessage(), e);
                return false;
            }
        }, assetCenterPool);

    }

    private void updateStatus(Integer status, String assetId) {
        assetDefinitionMongoDao.updateStatus(status, UserHelper.getCurrentUser().getName(), assetId);
    }

    @Override
    public Boolean removedAsset(String code, String application) {
        AssetDefinition assetDefinition = assetDefinitionMongoDao.selectByApplicationAndCode(code, application);
        updateStatus(AssetDefinition.REMOVED_STATUS, assetDefinition.getAssetId());
        return true;
    }

    @Override
    public Boolean removedAsset(String assetId) {
        updateStatus(AssetDefinition.REMOVED_STATUS, assetId);
        return true;
    }

    @Override
    public void deleteVersionAsset(AssetInfoDto assetInfoDto) {
        Assert.isTrue(assetInfoDto.judgeDeleteRequiredFields(), "需要补充必填字段");
        List<Asset> assets = assetMongoDao.selectByCodeAndApplicationAndSourceBranchAndAdpVersion(assetInfoDto.convertAsset());
        for (Asset asset : assets) {
            //如果当前生效版本的资产被删除，需要将业务数据给冗余下来，用来做持久化
            backupBusinessData(asset);
            assetMongoDao.delete(asset);
            modifySyncDefinition(asset.getAssetId());
        }
    }

    private void backupBusinessData(Asset asset) {
        if (VersionStatusEnum.RUN_STATE.getCode().equals(asset.getManageContext().getAdpStatus())) {
            Map<String, AssetBusinessDataAction> assetBusinessDataActions = assetCenterRegisterConfig.getAssetBusinessDataActions();
            AssetBusinessDataAction assetBusinessDataAction = assetBusinessDataActions.get(asset.getType());

            AssetInfoDto assetBusinessDataDto = assetBusinessDataAction.loadBusinessData(AssetInfoDto.create(asset));

            AssetDefinition assetDefinition = assetDefinitionMongoDao.selectByAssetId(asset.getAssetId());
            assetDefinition.setBusinessData(JSONObject.toJSONString(assetBusinessDataDto.getBusinessData()));
            assetDefinitionMongoDao.save(assetDefinition);
        }
    }

    @Override
    public void deleteAsset(Asset asset) {
        backupBusinessData(asset);
        assetMongoDao.delete(asset);
        modifySyncDefinition(asset.getAssetId());
    }

    /**
     * 有资产版本被删除了，需要同步修改下资产定义的一些信息
     * @param assetId
     */
    private void modifySyncDefinition(String assetId) {
        AssetDefinition assetDefinition = assetDefinitionMongoDao.selectByAssetId(assetId);

        Map<String, Asset> assetAllVersion = getAssetAllVersion(assetId);
        //没有一个资产版本，就将这个资产下架
        if (assetAllVersion.isEmpty()){
            assetDefinition.setCanReference(false);
            assetDefinition.setCanCopy(false);
            assetDefinition.setStatus(AssetDefinition.REMOVED_STATUS);
        }else{
            boolean hasEffect = false;
            for (Map.Entry<String, Asset> entry : assetAllVersion.entrySet()) {
                Asset value = entry.getValue();
                hasEffect = hasEffect || VersionStatusEnum.RUN_STATE.getCode().equals(value.getManageContext().getAdpStatus());
            }

            //可复制的资产版本如果被删除，需要从列表中移除
            if (!CollectionUtils.isEmpty(assetDefinition.getCopyableItems())){
                Iterator<String> iterator = assetDefinition.getCopyableItems().iterator();
                while (iterator.hasNext()){
                    String oId = iterator.next();
                    if (!assetAllVersion.containsKey(oId)){
                        iterator.remove();
                    }
                }
            }

            //生效的资产版本如果被删除，需要从列表中移除
            assetDefinition.setCanReference(assetDefinition.getCanReference() && hasEffect);
            if (assetDefinition.getCopyableItems().isEmpty()){
                assetDefinition.setCanCopy(false);
            }
            //既没有可复制版本也没有可引用版本，则需要将资产下架
            if (!assetDefinition.getCanCopy() && !assetDefinition.getCanReference()) {
                assetDefinition.setStatus(AssetDefinition.REMOVED_STATUS);
            }
        }
        UserHelper.fillEditInfo(assetDefinition);
        assetDefinitionMongoDao.save(assetDefinition);
    }

    @Override
    public String getAssetDefinitionBusinessData(String code, String application) {
        AssetDefinition assetDefinition = assetDefinitionMongoDao.selectByApplicationAndCode(code, application);

        return assetDefinition.getBusinessData();
    }

    @Override
    public Pagination<AssetCenterInfoResDto> getAssetCenterList(PageReqCondition<String> pageReqCondition,String type) {

        List types = type == null ? List.of() : List.of(type);

        PermissionResDto permissionResDto = null;
        if (type!=null){
            AssetType assetType = rAssetTypeMongoDao.selectByType(type);

            JSONObject rowDataPermissionJo = assetType.getRowDataPermission();
            if (rowDataPermissionJo!=null){
                ExecuteConfig rowDataPermission = JSONObject.parseObject(rowDataPermissionJo.toJSONString(),ExecuteConfig.class);

                HttpExecuteConfig httpExecuteConfig = new HttpExecuteConfig();
                httpExecuteConfig.setUrl(rowDataPermission.getUrl());
                if (!StringUtils.isEmpty(rowDataPermission.getMethod())){
                    httpExecuteConfig.setMethod(rowDataPermission.getMethod());
                }

                httpExecuteConfig.setRequestScript(rowDataPermission.getRequestScript());
                httpExecuteConfig.setResponseScript(rowDataPermission.getResponseScript());
                httpExecuteConfig.setHeaders(rowDataPermission.getHeaders());

                permissionResDto = (PermissionResDto) httpActuator.execute(httpExecuteConfig, new HashMap<>() , ReqPlaceholder.createParams(null), PermissionResDto.class);
            }
        }
        if (!permissionResDto.getDenyAll()){
            return buildPageAssetCenterInfo(pageReqCondition, new ArrayList<>(), 0L, true);
        }
        PermissionConditionResDto rowPermission = permissionResDto.getRowPermission();
        Long count = assetDefinitionMongoDao.countPagination(pageReqCondition, CurThreadInfoUtils.getCurTokenTenant().getTenantId(), types,rowPermission);
        List<AssetDefinition> assetDefinitions = assetDefinitionMongoDao.selectPagination(pageReqCondition, CurThreadInfoUtils.getCurTokenTenant().getTenantId(), types,rowPermission);
        List<AssetCenterInfoResDto> assetCenterInfoResDtoList = convertAssetCenterInfoResDtos(assetDefinitions);

        Pagination<AssetCenterInfoResDto> res = buildPageAssetCenterInfo(pageReqCondition, assetCenterInfoResDtoList, count,true);
        return res;
    }

    
    private List<AssetCenterInfoResDto> convertAssetCenterInfoResDtos(List<AssetDefinition> assetDefinitions) {
        List<String> assetIds = assetDefinitions.stream().map(AssetDefinition::getAssetId).collect(Collectors.toList());
        Map<String, Asset> effectAssetVersionMap = getEffectAssetVersion(assetIds);
        Set<String> existAssetIds = assetMongoDao.selectExistAssetId(assetIds);
        Map<String, Asset> lastVersionAssetVersionMap = getLastVersionAssetVersion(assetIds);

        List<AssetCenterInfoResDto> assetCenterInfoResDtoList = assetDefinitions.stream().map(a -> {
            AssetCenterInfoResDto assetCenterInfoResDto = AssetCenterInfoResDto.create(a);
            Asset effecAsset = effectAssetVersionMap.get(assetCenterInfoResDto.getAssetId());
            if (effecAsset!=null) {
                assetCenterInfoResDto.setEffectAdpVersion(effecAsset.getManageContext().getAdpVersion());
                assetCenterInfoResDto.setEffectAdpRemarkNumber(effecAsset.getShowFields().getAdpRemarkNumber());
            }

            assetCenterInfoResDto.setLatestVersionAsset(lastVersionAssetVersionMap.get(assetCenterInfoResDto.getAssetId()));

            assetCenterInfoResDto.setHasResult(existAssetIds.contains(assetCenterInfoResDto.getAssetId()));


            return assetCenterInfoResDto;
        }).collect(Collectors.toList());
        return assetCenterInfoResDtoList;
    }

    private Map<String, Asset> getLastVersionAssetVersion(List<String> assetIds) {
        Map<String, Asset> res = new HashMap<>();
        List<Asset> lastVersionAssetList = assetMongoDao.selectVersionInfo(assetIds);
        for (Asset asset : lastVersionAssetList) {
            if(res.containsKey(asset.getAssetId())){
                continue;
            }
            res.put(asset.getAssetId(),asset);
        }
        return res;
    }

    @Override
    public AssetDetailResDto getAsset(String assetId) {
        AssetDefinition assetDefinition = assetDefinitionMongoDao.selectByAssetId(assetId);

        AssetDetailResDto assetDetailResDto = AssetDetailResDto.create(assetDefinition);

        return assetDetailResDto;
    }

    @Override
    public List<AssetRelationAppInfoResDto> getAssetRefRelation(String assetId,Boolean isAssetCenter) {
        List<AssetRelationAppInfoResDto> res = new ArrayList<>();
        List<AssetRelation> relations = assetRelationMongoDao.selectByAssetIdAndType(assetId,AssetRelation.BelongTypeEnum.REF_TYPE.getType());
        if (!relations.isEmpty()) {

            Map<String, List<AssetRelation>> relationMap = relations.stream().collect(Collectors.groupingBy(a->a.getSource().getApplication()));
            List<Application> applications;
            if (isAssetCenter != null && isAssetCenter) {
                applications = applicationMongoDao.selectByCodesAndTenantId(relationMap.keySet(),CurThreadInfoUtils.getCurTokenTenant().getTenantId());
            }else{
                applications = applicationMongoDao.selectByCodes(relationMap.keySet());
            }

            for (Map.Entry<String, List<AssetRelation>> entry : relationMap.entrySet()) {
                AssetRelation assetRelation = entry.getValue().get(0);

                AssetRelationAppInfoResDto assetRelationAppInfoResDto = new AssetRelationAppInfoResDto();
                assetRelationAppInfoResDto.setAssetId(assetRelation.getAssetId());
                for (Application application : applications) {
                    if (application.getCode().equals(assetRelation.getSource().getApplication())) {
                        assetRelationAppInfoResDto.setName(application.getName());
                        assetRelationAppInfoResDto.setApplication(application.getCode());
                        assetRelationAppInfoResDto.setTenantId(application.getTenantId());
                        assetRelationAppInfoResDto.setLang(application.getLang());
                        break;
                    }
                }
                res.add(assetRelationAppInfoResDto);
            }
        }
        return res;
    }

    @Override
    public Integer assetErrorCount(String appCode) {
        List<String> assetIdsRelation = assetRelationMongoDao.selectAssetIdByAppCode(appCode, AssetDefinition.AssetTypeEnum.DTD.getCode());
        List<String> assetIds = assetDefinitionMongoDao.selectAssetIdAvailableStatusByApplication(appCode, AssetDefinition.AssetTypeEnum.DTD.getCode());

        Long count = assetMongoDao.selectEffectCount(assetIdsRelation);
        Set<String> existAssetIds = assetMongoDao.selectExistAssetId(assetIds);
        return (assetIdsRelation.size() - count.intValue()) + (assetIds.size() - existAssetIds.size());
    }

    @Override
    public Pagination<AssetCenterInfoResDto> appAssetList(PageReqCondition<String> pageReqCondition, List<String> appCodes, List<String> types) {
        Long count = assetDefinitionMongoDao.countApplicationPagination(pageReqCondition, appCodes, types);
        List<AssetDefinition> assetDefinitions = assetDefinitionMongoDao.selectApplicationPagination(pageReqCondition, appCodes, types);
        List<AssetCenterInfoResDto> assetCenterInfoResDtoList = convertAssetCenterInfoResDtos(assetDefinitions);

        return buildPageAssetCenterInfo(pageReqCondition, assetCenterInfoResDtoList, count,false);
    }

    @Override
    public Pagination<AssetDetailResDto> designerAssetList(PageReqCondition<DesignerAssetListReqDto> pageReqCondition) {
        Long count = assetMongoDao.countDesignerAssetPagination(pageReqCondition);
        List<Asset> assets = assetMongoDao.selectDesignerAssetPagination(pageReqCondition);

        List<AssetCenterInfoResDto> res = assets.stream().map(a -> AssetDetailResDto.create(a)).collect(Collectors.toList());
        return buildPageAssetCenterInfo(pageReqCondition, res, count,false);
    }

    @Override
    public List<DepAssetResDto> getDepAssets(String objId) {
        Asset asset = assetMongoDao.selectByObjectId(objId);
        Assert.notNull(asset,"没有该资产信息");

        List<AssetDepInfo> assetDepInfos = assetDepInfoMongoDao.selectBySourceId(asset.getObjectId());
        Map<String, AssetDepInfo> assetDepInfoMap = assetDepInfos.stream().collect(Collectors.toMap(AssetDepInfo::getTargetAssetId, a -> a));

        List<DepAssetResDto> res = new ArrayList<>();
        if (!assetDepInfos.isEmpty()){
            List<AssetDefinition> assetDefinitions = assetDefinitionMongoDao.selectByAssetIds(assetDepInfoMap.keySet());

            if (assetDefinitions.isEmpty()){
                return res;
            }
            Map<String, AssetDefinition> assetDefinitionMap = assetDefinitions.stream().collect(Collectors.toMap(AssetDefinition::getAssetId, a -> a));

            for (Map.Entry<String, AssetDepInfo> entry : assetDepInfoMap.entrySet()) {
                String key = entry.getKey();
                if (assetDefinitionMap.containsKey(key)){
                    res.add(DepAssetResDto.create(entry.getValue(),assetDefinitionMap.get(entry.getKey())));
                }
            }
        }
        return res;
    }

    @Override
    public void assetInit(String appCode, String type) {
        if ("ALL".equalsIgnoreCase(type) || AssetDefinition.AssetTypeEnum.DTD.getCode().equals(type)) {
            assetInit.dtdAssetInit(appCode);
        }

        Application application = applicationMongoDao.getApplication(appCode);
        UserHelper.fillEditInfo(application);
        if ("ALL".equalsIgnoreCase(type)){
            application.getAssetInitFlag().put(AssetDefinition.AssetTypeEnum.DTD.getCode(), true);
            application.getAssetInitFlag().put(type, true);
        }else{
            application.getAssetInitFlag().put(type, true);
        }
        applicationMongoDao.save(application);
    }


    @Override
    public Pagination<DependentAssetResDto> getDependAssetInfoList(PageReqCondition<String> pageReqCondition, String appCode, String type) {
        Long count = assetRelationMongoDao.selectCountPagination(pageReqCondition, appCode,type);
        List<AssetRelation> relations = assetRelationMongoDao.selectPagination(pageReqCondition, appCode,type);
        Set<String> assetIds = relations.stream().map(AssetRelation::getAssetId).collect(Collectors.toSet());

        //查找存在生效版本的资产
        List<Asset> assets = assetMongoDao.selectEffectByAssetIds(assetIds);
        List<AssetDefinition> assetDefinitions = assetDefinitionMongoDao.selectByAssetIds(assetIds);

        List<DependentAssetResDto> res = new ArrayList<>();
        for (AssetRelation re : relations) {
            DependentAssetResDto dependentAssetResDto = new DependentAssetResDto();
            dependentAssetResDto.setObjectId(re.getObjectId());
            dependentAssetResDto.setAssetId(re.getAssetId());

            Asset existAsset = assets.stream().filter(a -> a.getAssetId().equals(re.getAssetId())).findFirst().orElse(null);
            if (existAsset != null) {
                dependentAssetResDto.setReferenceInvalid(false);
                dependentAssetResDto.setAdpVersion(existAsset.getManageContext().getAdpVersion());
                dependentAssetResDto.setAdpRemarkNumber(existAsset.getShowFields().getAdpRemarkNumber());
            }

            AssetDefinition existAssetDefinition = assetDefinitions.stream().filter(a -> a.getAssetId().equals(re.getAssetId())).findFirst().orElse(null);
            if (existAssetDefinition != null) {
                dependentAssetResDto.setAssetName(existAssetDefinition.getAssetName());
                dependentAssetResDto.setAssetDesc(existAssetDefinition.getAssetDesc());
                dependentAssetResDto.setPublisher(existAssetDefinition.getPublisher());
                dependentAssetResDto.setPublisherName(existAssetDefinition.getPublisherName());
                dependentAssetResDto.setApplication(existAssetDefinition.getManageContext().getApplication());
                dependentAssetResDto.setType(existAssetDefinition.getType());
                dependentAssetResDto.setCode(existAssetDefinition.getCode());
            }
            dependentAssetResDto.setAssetRelation(re);
            res.add(dependentAssetResDto);
        }
        return Pagination.buildPagination(pageReqCondition, res, count);
    }



    @Override
    public List<AssetInfoResDto> assetVersionList(String assetId) {
        List<Asset> assets = assetMongoDao.selectByAssetId(assetId);

        return assets.stream().map(a -> AssetInfoResDto.create(a)).collect(Collectors.toList());
    }

    @Override
    public Pagination<AssetDetailResDto> getSameTenantDtdAsset(PageReqCondition<String> pageReqCondition) {

        Long count = assetDefinitionMongoDao.countSameTenantDtdAsset(pageReqCondition, CurThreadInfoUtils.getCurTokenTenant().getTenantId(), AssetDefinition.AssetTypeEnum.DTD.getCode());
        List<AssetDefinition> assetDefinitions = assetDefinitionMongoDao.selectSameTenantDtdAsset(pageReqCondition, CurThreadInfoUtils.getCurTokenTenant().getTenantId(), AssetDefinition.AssetTypeEnum.DTD.getCode());

        List<AssetDetailResDto> assetDetailResDtoList = assetDefinitions.stream().map(a -> AssetDetailResDto.create(a)).collect(Collectors.toList());

        return Pagination.buildPagination(pageReqCondition, assetDetailResDtoList, count);
    }

    @Override
    public Pagination<AssetDetailResDto> getPlatformDtdAsset(PageReqCondition<String> pageReqCondition) {
        Long count = assetDefinitionMongoDao.countPlatformDtdAsset(pageReqCondition, CurThreadInfoUtils.getCurTokenTenant().getTenantId(), AssetDefinition.AssetTypeEnum.DTD.getCode());
        List<AssetDefinition> assetDefinitions = assetDefinitionMongoDao.selectPlatformDtdAsset(pageReqCondition, CurThreadInfoUtils.getCurTokenTenant().getTenantId(), AssetDefinition.AssetTypeEnum.DTD.getCode());

        List<AssetDetailResDto> assetDetailResDtoList = assetDefinitions.stream().map(a -> AssetDetailResDto.create(a)).collect(Collectors.toList());

        return Pagination.buildPagination(pageReqCondition, assetDetailResDtoList, count);
    }



    @Override
    public Boolean removedAssetOpenTenant(String assetId) {
        AssetDefinition assetDefinition = assetDefinitionMongoDao.selectByAssetId(assetId);
        if (AssetDefinition.ALL_RANGE.equals(assetDefinition.getOpenRange())){
            throw new RuntimeException("该资产开放范围为平台级，不能直接下架");
        }
        if (AssetDefinition.CUR_TENANT_RANGE.equals(assetDefinition.getOpenRange())){
            assetDefinition.setStatus(AssetDefinition.REMOVED_STATUS);
        }
        if (AssetDefinition.TENANT_RANGE.equals(assetDefinition.getOpenRange())){
            List<TenantInfo> tenantInfos = assetDefinition.getPermission().getTenantInfos();
            String tenantId = CurThreadInfoUtils.getCurTokenTenant().getTenantId();

            Iterator<TenantInfo> iterator = tenantInfos.iterator();
            while (iterator.hasNext()){
                TenantInfo tenantInfo = iterator.next();
                if (tenantId.equals(tenantInfo.getTenantId())){
                    iterator.remove();
                }
            }

            if (tenantInfos.isEmpty()){
                assetDefinition.setStatus(AssetDefinition.REMOVED_STATUS);
            }
        }
        UserHelper.fillEditInfo(assetDefinition);
        assetDefinitionMongoDao.save(assetDefinition);
        return true;
    }



    private Pagination buildPageAssetCenterInfo(PageReqCondition pageReqCondition, List<AssetCenterInfoResDto> assetCenterInfoResDtoList, Long count,Boolean isAssetCenter) {

        for (AssetCenterInfoResDto assetCenterInfoResDto : assetCenterInfoResDtoList) {

            //查询这个资产当前被引用的次数
            Collection<String> appCodes = assetRelationMongoDao.selectRefAppAppByAssetId(assetCenterInfoResDto.getAssetId());
            Integer refCount = appCodes.size();
            if (!appCodes.isEmpty()) {
                //如果是资产中心查询，需要只统计当前租户下的
                if (isAssetCenter){
                    refCount = applicationMongoDao.selectCountByCodesAndTenantId(appCodes, CurThreadInfoUtils.getCurTokenTenant().getTenantId()).intValue();
                }
            }
            assetCenterInfoResDto.setCurReferenceCount(refCount);
        }
        Pagination res = Pagination.buildPagination(pageReqCondition, assetCenterInfoResDtoList, count);

        return res;
    }

    @Override
    public void copyAppRef(String application, String fromBranch, String toBranch){
        List<AssetRelation> relations = assetRelationMongoDao.selectByApplicationAndBranch(application,fromBranch);
        assetRelationMongoDao.deleteByApplicationAndBranch(application,toBranch);

        if (!relations.isEmpty()){
            for (AssetRelation relation : relations) {
                relation.setObjectId(null);
                relation.getSource().setUserBranch(toBranch);
            }

            assetRelationMongoDao.insertAll(relations);
        }

        List<Asset> fromAssetList = assetMongoDao.selectByApplicationAndBranch(application,fromBranch);
        if (!fromAssetList.isEmpty()){

            for (Asset asset : fromAssetList) {
                asset.setObjectId(null);
                asset.getManageContext().setBranch(toBranch);
                if (!BranchConfig.DEV_BRANCH_NAME.equals(toBranch)){
                    asset.getManageContext().setAdpStatus(VersionStatusEnum.DESIGN_STATE.getCode());
                }
            }
            assetMongoDao.deleteByApplicationAndBranch(application,toBranch);
            assetMongoDao.insertAll(fromAssetList);
        }
    }


    @Override
    public void copyRelationMultipleVersions(String oldVersion,String newVersion,String branch,String appCode){
        List<AssetRelation> assetRelations = assetRelationMongoDao.selectByUserBranchAndAdpVersionAndApp(oldVersion,branch,appCode);

        if (!assetRelations.isEmpty()){
            for (AssetRelation assetRelation : assetRelations) {
                assetRelation.setObjectId(null);
                assetRelation.getSource().setAdpVersion(newVersion);
            }
            assetRelationMongoDao.insertAll(assetRelations);
        }
    }

    @Override
    public void removeRelationMultipleVersions(String version,String branch,String appCode){
        assetRelationMongoDao.deleteByAdpVersionAndBranchAndApplication(version,branch,appCode);
    }

    @Override
    public Boolean batchSyncAsset(List<String> assetIds) {
        List<AssetDefinition> assetDefinitions = assetDefinitionMongoDao.selectByAssetIdsAndStatus(assetIds,AssetDefinition.AVAILABLE_STATUS);

        try {
            for (AssetDefinition assetDefinition : assetDefinitions) {
                if (assetDefinition.getCopyableItems().isEmpty()){
                    continue;
                }
                List<Asset> assetBasicInfoList = assetMongoDao.selectBasicInfoByObjectId(assetDefinition.getCopyableItems());

                updateAssetData(assetBasicInfoList);
            }
        } catch (Exception e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException("同步可复制版本异常",e);
        }
        return true;
    }

    @Override
    public AssetDefinition findAssetDefinition(String assetId) {
        return assetDefinitionMongoDao.selectByAssetId(assetId);
    }

    @Override
    public void save(AssetDefinition assetDefinition) {
        assetDefinitionMongoDao.save(assetDefinition);
    }

    @Override
    public void draftDtdAllVersion(String application) {
        assetMongoDao.updateDraftByApplicationAndType(application,AssetDefinition.AssetTypeEnum.DTD);
    }



    @Override
    public Pagination<AssetCenterInfoResDto> appUseAssetList(PageReqCondition<UseAssetListReqDto> pageReqCondition) {
        Long count = assetDefinitionMongoDao.countAppUseAssetList(pageReqCondition);
        List<AssetDefinition> assetDefinitions = assetDefinitionMongoDao.selectAppUseAssetList(pageReqCondition);
        List<AssetCenterInfoResDto> assetCenterInfoResDtoList = convertAssetCenterInfoResDtos(assetDefinitions);

        Pagination<AssetCenterInfoResDto> res = Pagination.buildPagination(pageReqCondition, assetCenterInfoResDtoList, count);
        return res;
    }

    @Override
    public void unbindingRelationship(String taskCodes,String type, String appCode) {
        List<String> taskCodeArr = Arrays.asList(taskCodes.split(BaseConstant.COMMA_SEPARATOR));
        List<AssetDefinition> assetDefinitions = assetDefinitionMongoDao.selectByCodesAndType(taskCodeArr,type);

        if (taskCodeArr.size() > assetDefinitions.size()){
            Set<String> existsCodes = assetDefinitions.stream().map(AssetDefinition::getCode).collect(Collectors.toSet());
            for (String code : taskCodeArr) {
                if (!existsCodes.contains(code)){
                    throw new RuntimeException(code+"没有查询到对应的资产信息");
                }
            }
        }
        //查找是否存在生效版本的资产
        for (AssetDefinition assetDefinition : assetDefinitions) {
            AssetRelation assetRelation = assetRelationMongoDao.selectByAssetIdAndAdpVersionAndUserBranchAndApplication(assetDefinition.getAssetId(), CurThreadInfoUtils.getAdpVersion(), CurThreadInfoUtils.getBranchKey(), appCode);
            Assert.notNull(assetRelation, "没有查询到引用此资产的关系");

            assetRelationMongoDao.deleteByObjectId(assetRelation.getObjectId());
        }

    }

    @Override
    public List<AssetInfoResDto> assetCanCopyVersionList(String assetId) {
        AssetDefinition assetDefinition = assetDefinitionMongoDao.selectByAssetId(assetId);
        Assert.notNull(assetDefinition,"不存在对应的资产");
        List<String> copyableItems = assetDefinition.getCopyableItems();
        if (!CollectionUtils.isEmpty(copyableItems)){
            List<Asset> assets = assetMongoDao.selectByIds(copyableItems);
            return assets.stream().map(a -> AssetInfoResDto.create(a)).collect(Collectors.toList());
        }

        return new ArrayList<>();
    }

    @Override
    public AssetCenterInfoResDto getAssetDefinitionInfo(String code, String type) {
        AssetDefinition assetDefinition = assetDefinitionMongoDao.selectByCodeAndType(code, type);
        Assert.notNull(assetDefinition,"没有查询到对应的资产");
        AssetCenterInfoResDto res = AssetCenterInfoResDto.create(assetDefinition);
        Application application = applicationMongoDao.selectByCode(res.getApplication());

        Assert.notNull(application,res.getApplication()+"应用不存在");
        res.setApplicationName(application.getName());
        res.getLang().put("applicationName",application.getLang().get("name"));

        Map<String, Asset> effectAssetVersionMap = getEffectAssetVersion(Arrays.asList(assetDefinition.getAssetId()));
        Asset asset = effectAssetVersionMap.get(res.getAssetId());
        if (asset!=null) {
            res.setEffectAdpVersion(asset.getManageContext().getAdpVersion());
        }
        return res;
    }

    @Override
    public Map<String, Asset> getAssetAllVersion(String assetId) {
        List<Asset> assets = assetMongoDao.selectBasicInfoByAssetId(assetId);
        Map<String, Asset> res = new HashMap<>();
        for (Asset asset : assets) {
            res.put(asset.getObjectId(),asset);
        }
        return res;
    }

    @Override
    public AssetDetailResDto saveAsset(CreateAssetReqDto createAssetReqDto, com.digiwin.athena.dto.TenantInfo tenantInfo) {
        AssetType assetType = rAssetTypeMongoDao.selectByType(createAssetReqDto.getType());
        Assert.notNull(assetType,"不存在对应的已发布数据");

        Asset.ManageContext manageContext = createAssetReqDto.getManageContext();
        if (StringUtils.isEmpty(manageContext.getDataId())){
            manageContext.setDataId(createAssetReqDto.getCode());
        }

        //接口没有传assetId，需要根据业务code判断是否已经存在对应的资产
        AssetDefinition oldAssetDefinition = null;
        if (StringUtils.isEmpty(createAssetReqDto.getAssetId())){
            if (!StringUtils.isEmpty(createAssetReqDto.getCode())) {
                oldAssetDefinition = assetDefinitionMongoDao.selectByCodeAndType(createAssetReqDto.getCode(), createAssetReqDto.getType());
            }
        }else{
            oldAssetDefinition = assetDefinitionMongoDao.selectByAssetId(createAssetReqDto.getAssetId());
            Assert.notNull(oldAssetDefinition,"未查询到对应的资产实例");
        }
        //更新
        if (oldAssetDefinition != null) {
            oldAssetDefinition.setAssetName(createAssetReqDto.getAssetName());
            oldAssetDefinition.setAssetDesc(createAssetReqDto.getAssetDesc());
            oldAssetDefinition.setLang(createAssetReqDto.getLang());
            oldAssetDefinition.setBusinessData(createAssetReqDto.getBusinessData());
            oldAssetDefinition.setManageContext(createAssetReqDto.getManageContext());
            UserHelper.fillEditInfo(oldAssetDefinition);

            assetDefinitionMongoDao.save(oldAssetDefinition);

            Asset asset = assetMongoDao.selectByAssetIdAndManageContext(oldAssetDefinition.getAssetId(), createAssetReqDto.getManageContext());
            //资产实例多版本 为null，新增
            if (asset == null) {
                asset = createAssetReqDto.convertAsset();
                asset.setAssetId(createAssetReqDto.getAssetId());

                UserHelper.fillCreateInfo(asset);
                UserHelper.fillEditInfo(asset);

                assetMongoDao.insert(asset);
            } else {
                asset.setDepResolveError(null);
                asset.setAssetName(createAssetReqDto.getAssetName());
                asset.setBusinessData(createAssetReqDto.getBusinessData());
                asset.setLang(createAssetReqDto.getLang());
                UserHelper.fillEditInfo(asset);
                assetMongoDao.save(asset);
            }
            resolveDepInfo(asset,assetType, createAssetReqDto.getAssetDepInfos());

            AssetDetailResDto assetDetailResDto = AssetDetailResDto.create(oldAssetDefinition);
            assetDetailResDto.setManageContext(asset.getManageContext());
            assetDetailResDto.setShowFields(asset.getShowFields());
            return assetDetailResDto;
        }else {
            AssetDefinition assetDefinition = createAssetReqDto.convertAssetDefinition();
            String code = StringUtils.isEmpty(assetDefinition.getCode())? IdUtil.fastSimpleUUID(): assetDefinition.getCode();

            String assetId = buildGeneratorCodeUtil.createAssetId(code, createAssetReqDto.getType());
            assetDefinition.setAssetId(assetId);

            assetDefinition.setCanReference(assetType.getCanReference());
            assetDefinition.setCanCopy(assetType.getCanCopy());
            assetDefinition.setStatus(AssetDefinition.AVAILABLE_STATUS);
            assetDefinition.getManageContext().putAll(createAssetReqDto.getManageContext());
            com.digiwin.athena.dto.TenantInfo curTokenTenant = tenantInfo != null ? tenantInfo : CurThreadInfoUtils.getCurTokenTenant();

            assetDefinition.setAdpTenantId(curTokenTenant.getTenantId());
            assetDefinition.setPublisherName(curTokenTenant.getTenantName());
            assetDefinition.setPublisher(curTokenTenant.getTenantId());

            UserHelper.fillCreateInfo(assetDefinition);
            UserHelper.fillEditInfo(assetDefinition);
            assetDefinitionMongoDao.insert(assetDefinition);

            Asset asset = createAssetReqDto.convertAsset();
            asset.setAssetId(assetId);
            UserHelper.fillCreateInfo(asset);
            UserHelper.fillEditInfo(asset);
            assetMongoDao.insert(asset);

            resolveDepInfo(asset,assetType, createAssetReqDto.getAssetDepInfos());
            AssetDetailResDto assetDetailResDto = AssetDetailResDto.create(assetDefinition);
            assetDetailResDto.setManageContext(asset.getManageContext());
            assetDetailResDto.setShowFields(asset.getShowFields());
            return assetDetailResDto;
        }
    }

    /**
     * 解析资产的依赖信息
     * @param asset
     * @param assetType
     * @param assetDepInfos
     */
    private void resolveDepInfo(Asset asset,AssetType assetType,List<AssetDepInfo> assetDepInfos) {
        try {
            List<JSONObject> dependencyRule = assetType.getDependencyRule();
            if (CollectionUtils.isEmpty(dependencyRule) && CollectionUtils.isEmpty(assetDepInfos)) {
                assetDepInfoMongoDao.deleteBySourceId(asset.getObjectId());
            }
            if (CollectionUtils.isEmpty(assetDepInfos)){
                assetDepInfos = new ArrayList<>();
            }

            if (!CollectionUtils.isEmpty(dependencyRule)){
                for (JSONObject jsonObject : dependencyRule) {
                    DependencyRule dependency = JSONObject.parseObject(jsonObject.toJSONString(), DependencyRule.class);

                    if (dependency.getType().equals(DependencyRule.SCRIPT_TYPE)){
                        Object depInfo = JSUtil.evalDependencyRule(dependency.getScript(),JSONObject.toJSONString(asset),Map.class);

                        List<AssetDepInfo> allAnalysisDepInfo = new ArrayList<>();
                        if (depInfo instanceof Collection){
                            allAnalysisDepInfo = JSONObject.parseArray(JSONObject.toJSONString(depInfo), AssetDepInfo.class);
                        }else{
                            allAnalysisDepInfo.add(JSONObject.parseObject(JSONObject.toJSONString(depInfo), AssetDepInfo.class));
                        }
                        for (AssetDepInfo assetDepInfo : allAnalysisDepInfo) {

                            Asset targetAsset = assetMongoDao.selectByCodeAndTypeAndManageContext(assetDepInfo.getTargetType(), assetDepInfo.getTargetCode(), assetDepInfo.getTargetManageContext());
                            if (targetAsset == null) {
                                continue;
                            }

                            assetDepInfo.setTargetAssetId(targetAsset.getAssetId());
                            assetDepInfo.setSourceId(asset.getObjectId());
                            assetDepInfos.add(assetDepInfo);
                        }

                    }
                }
            }

            Map<String, List<AssetDepInfo>> depInfoMap = assetDepInfos.stream().collect(Collectors.groupingBy(a -> a.getTargetType() + a.getTargetCode() + a.getTargetAssetId()));

            if (!assetDepInfos.isEmpty()){
                List<AssetDepInfo> oldDepInfos = assetDepInfoMongoDao.selectBySourceId(asset.getObjectId());
                Map<String, List<AssetDepInfo>> oldDepInfoMap = oldDepInfos.stream().collect(Collectors.groupingBy(a -> a.getTargetType() + a.getTargetCode() + a.getTargetAssetId()));

                //过滤需要删除的关系
                List<AssetDepInfo> needDeleteDepInfo = oldDepInfos.stream().filter(o -> {
                    String key = o.getTargetType() + o.getTargetCode() + o.getTargetAssetId();
                    if (depInfoMap.containsKey(key)) {
                        List<AssetDepInfo> newAssetDepInfos = depInfoMap.get(key);
                        for (AssetDepInfo newAssetDepInfo : newAssetDepInfos) {
                            if (newAssetDepInfo.getTargetManageContext().equals(o.getTargetManageContext())) {
                                return false;
                            }
                        }
                    }
                    return true;
                }).collect(Collectors.toList());

                if (!needDeleteDepInfo.isEmpty()){
                    needDeleteDepInfo.forEach(n->assetDepInfoMongoDao.delete(n));
                }
                //过滤需要新增的关系
                List<AssetDepInfo> needInsertAssetDepInfos = assetDepInfos.stream().filter(a -> {
                    String key = a.getTargetType() + a.getTargetCode() + a.getTargetAssetId();

                    if (oldDepInfoMap.containsKey(key)) {
                        List<AssetDepInfo> oldAssetDepInfos = oldDepInfoMap.get(key);
                        for (AssetDepInfo oldAssetDepInfo : oldAssetDepInfos) {
                            if (oldAssetDepInfo.getTargetManageContext().equals(a.getTargetManageContext())) {
                                return false;
                            }
                        }
                    }
                    return true;
                }).collect(Collectors.toList());
                if (!needInsertAssetDepInfos.isEmpty()) {
                    assetDepInfoMongoDao.insert(needInsertAssetDepInfos);
                }

            }else{
                assetDepInfoMongoDao.deleteBySourceId(asset.getObjectId());
            }
        } catch (Exception e) {
            asset.setDepResolveError(e.getMessage());
            assetMongoDao.save(asset);
            throw new RuntimeException("资产解析依赖异常");
        }
    }

    @Override
    public void deleteAsset(DeleteAssetReqDto deleteAssetReqDto) {
        AssetDefinition assetDefinition = null;
        if (!StringUtils.isEmpty(deleteAssetReqDto.getAssetId())){
            assetDefinition = assetDefinitionMongoDao.selectByAssetId(deleteAssetReqDto.getAssetId());
        }else{
            assetDefinition = assetDefinitionMongoDao.selectByCodeAndType(deleteAssetReqDto.getCode(), deleteAssetReqDto.getType());
        }
        Assert.notNull(assetDefinition,"没有对应的资产信息");

        Asset.ManageContext manageContext = deleteAssetReqDto.getManageContext();
        if (StringUtils.isEmpty(manageContext.getDataId())){
            manageContext.setDataId(deleteAssetReqDto.getCode());
        }

        Asset asset = assetMongoDao.selectByAssetIdAndManageContext(assetDefinition.getAssetId(), deleteAssetReqDto.getManageContext());
        if (asset != null){
            assetDepInfoMongoDao.deleteBySourceId(asset.getObjectId());
            assetMongoDao.delete(asset);
        }
        Set<String> existAssetId = assetMongoDao.selectExistAssetId(List.of(assetDefinition.getAssetId()));
        if (existAssetId.isEmpty()){
            assetDefinitionMongoDao.delete(assetDefinition);
        }
    }

}
