package com.digiwin.athena.config;

import cn.hutool.extra.spring.SpringUtil;
import jakarta.annotation.PostConstruct;
import jakarta.validation.constraints.Min;
import lombok.Data;
import org.hibernate.validator.constraints.Range;
import org.springframework.boot.context.properties.NestedConfigurationProperty;

import java.time.Duration;
import java.time.LocalDateTime;

/**
 * Assembly模块配置属性
 */
@Data
public class AssemblyProp {

    /**
     * 全局应用token
     * `@Value("${rest-url.appToken:}")`
     */
    private String appToken;

    /**
     * IAM域名
     * `@Value("${module.iam.domain:}")`
     */
    private String iamDomain;

    /**
     * GMC域名
     * `@Value("${module.gmc.domain:}")`
     */
    private String gmcDomain;

    /**
     * 解决方案logo url
     */
    @NestedConfigurationProperty
    private SolutionProp solution = new SolutionProp();

    @NestedConfigurationProperty
    private DataCollector collector = new DataCollector();

    @NestedConfigurationProperty
    private JobProp job = new JobProp();

    /**
     * 商品类别代码（ODM/lcdp）
     */
    private String productCode = "lcdp";

    @PostConstruct
    public void init() {
        if (appToken == null) {
            appToken = SpringUtil.getProperty("rest-url.appToken");
        }

        if (iamDomain == null) {
            iamDomain = SpringUtil.getProperty("module.iam.domain");
        }

        if (gmcDomain == null) {
            gmcDomain = SpringUtil.getProperty("module.gmc.domain");
        }

        if (solution.gmcLogoImgUrl == null) {
            solution.gmcLogoImgUrl = SpringUtil.getProperty("module.iam.app-logo");
        }
    }


    @Data
    public static class SolutionProp {
        /**
         * `@Value("${module.iam.app-logo}")`
         */
        private String gmcLogoImgUrl;
    }

    /**
     * 环境配置
     */
    @Data
    public static class EnvConfig {

        /**
         * IAM域名
         */
        private String iamDomain;

        /**
         * GMC域名
         */
        private String gmcDomain;
    }


    @Data
    public static class DataCollector {
        /**
         * 分页查询大小
         */
        @Range(min = 1, max = 10 * 1000)
        private int pageSize = 100;

        /**
         * 步进查询步长
         */
        @Range(min = 1)
        private Duration stepSpan = Duration.ofHours(10);

        /**
         * 同步开始时间
         */
        private LocalDateTime startTime;
    }

    /**
     * Job配置
     */
    @Data
    public static class JobProp {
        private Boolean enabled = true;
        /**
         * 最大线程数
         */
        @Range(min = 1, max = 100)
        private int maxThreads = 5;

        /**
         * 最大任务数，超出拒绝
         */
        @Min(0)
        private int maxRunJob = 100;

        /**
         * 任务执行超时时间
         */
        private Duration timeout = Duration.ofMinutes(30);
    }


}
