package com.digiwin.athena.bo.assembly;

import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.constants.AssemblyConstants;
import com.digiwin.athena.constants.AssemblyErrorCode;
import com.digiwin.athena.constants.enums.assembly.AssemblyType;
import com.digiwin.athena.constants.enums.assembly.OnlineState;
import com.digiwin.athena.dto.action.MultiLanguageDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanDetailDTO;
import com.digiwin.athena.mongodb.domain.PublishAble;
import lombok.Data;

import java.time.LocalDateTime;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * 可组装解决方案配置
 */
@Data
public class SolutionPlan implements PublishAble, IVersion {

    public static final int BASE_APP_TYPE = 10000;

    /**
     * 主键ID
     */
    private String id;

    /**
     * 解决方案编码，全局唯一
     */
    private String type;

    /**
     * 解决方案名称
     */
    private String name;

    /**
     * 版本号，service层暂时写死1.0
     */
    private String version = "1.0";

    /**
     * 版本排序，version根据`.`分割，{第一部分}*1000+{第二部分}
     */
    private Integer versionSort;

    /**
     * 描述
     */
    private String description;

    /**
     * 组装方式：dynamic(可组装)/static(不可组装)
     *
     * @see AssemblyType#getCode()
     */
    private String assemblyType;

    /**
     * 解决方案状态：Draft(草稿)/Online(已上架)/Offline(未上架)
     *
     * @see OnlineState#getCode()
     */
    private String status;

    /**
     * 归属方
     */
    private String owner;

    /**
     * 应用类型，新增时自动分配
     */
    private Integer appType;

    /**
     * 关联设计器列表
     */
    private List<SolutionPlanDetailDTO.DesignerBaseInfoDTO> designerList;

    /**
     * 布局配置
     */
    private Map<String, Object> canvas;

    /**
     * 布局方式
     */
    private String layoutMode;

    /**
     * SSO配置(不可组装页面配置)
     */
    private SolutionPlanDetailDTO.SsoConfigDTO ssoConfig;

    /**
     * 扩展配置字段 - 应用创建回调
     */
    private List<SolutionPlanDetailDTO.AppCallBackDTO> appCreateCallback;

    /**
     * 扩展配置字段 - 应用更新回调
     */
    private List<SolutionPlanDetailDTO.AppCallBackDTO> appUpdateCallback;

    /**
     * 扩展配置字段 - 编译回调
     */
    private List<SolutionPlanDetailDTO.AppCallBackDTO> compile;

    /**
     * 扩展配置字段 - 发布回调
     */
    private List<SolutionPlanDetailDTO.AppCallBackDTO> publish;

    /**
     * true走标准发布，对应的应用类型会在运维中心体现
     */
    private Boolean isStdPublish;

    /**
     * 合并JSON字段，用于动态添加额外数据
     * <br/>
     * 后期若与原功能合并，则将Map类型限定为{@link com.digiwin.athena.mongodb.domain.application.ApplicationConfig}
     *
     * @see com.digiwin.athena.mongodb.domain.application.ApplicationConfig
     */
    private Map<String, Object> mergeJson;

    /**
     * 多语言配置
     */
    private Map<String, MultiLanguageDTO<String>> lang;

    /**
     * 发布时间
     */
    private Date publishTime;

    /**
     * 发布人
     */
    private String publishBy;

    /**
     * 创建时间
     */
    private LocalDateTime createTime;

    /**
     * 更新时间
     */
    private LocalDateTime editTime;

    /**
     * 创建人
     */
    private String createBy;

    /**
     * 更新人
     */
    private String editBy;


    public void overwriteUnmodifiable(SolutionPlan updated) {
        // 只有草稿态可以修改名称和编码
        if (!OnlineState.Draft.getCode().equals(status)) {
            updated.setName(getName());
            updated.setType(getType());
        }

        updated.setStatus(getStatus());
        updated.setAppType(getAppType());
        updated.setCreateBy(getCreateBy());
        updated.setCreateTime(getCreateTime());
        updated.setPublishBy(getPublishBy());
        updated.setPublishTime(getPublishTime());
    }

    public void prepareNew() {
        setStatus(OnlineState.Draft.getCode());
        setPublishTime(null);
        setPublishBy(null);
    }

    public void checkUpdate(SolutionPlan exist) {
        if (!OnlineState.Draft.getCode().equals(exist.getStatus())) {
            throw new BusinessException(AssemblyErrorCode.NOT_DRAFT, "状态不为草稿！", exist.getName(), exist.getStatus());
        }
    }

    public void checkDelete(SolutionPlan exist) {
        if (!OnlineState.Draft.getCode().equals(exist.getStatus())) {
            throw new BusinessException(AssemblyErrorCode.NOT_DRAFT, "状态不为草稿！", exist.getName(), exist.getStatus());
        }
    }

    public boolean isFirstPublish() {
        return getPublishTime() == null;
    }

    /**
     * 校验type字段格式
     *
     * @throws BusinessException 如果type字段格式不符合要求
     */
    public void validateTypeFormat() {
        if (StrUtil.isBlank(type)) {
            throw new BusinessException(AssemblyErrorCode.INVALID_PARAM, "解决方案编码不能为空！");
        }

        if (!AssemblyConstants.TYPE_PATTERN.matcher(type).matches()) {
            throw new BusinessException(AssemblyErrorCode.INVALID_PARAM,
                    StrUtil.format("解决方案编码格式不正确！{}，当前编码: {}", AssemblyConstants.TYPE_PATTERN_TIP, type));
        }
    }

    public static boolean isAssemblyApp(Integer appType) {
        if (appType == null) {
            return false;
        }

        return appType >= SolutionPlan.BASE_APP_TYPE;
    }
}
