package com.digiwin.athena.integration.assembly.facade;

import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.bo.assembly.SolutionPlan;
import com.digiwin.athena.config.AssemblyProp;
import com.digiwin.athena.constant.GmcConstant;
import com.digiwin.athena.http.iam.service.AppTokenService;
import com.digiwin.athena.integration.assembly.facade.dto.CommodityMultiEnvSyncDTO;
import com.digiwin.athena.integration.assembly.gmc.api.GmcCommodityApi;
import com.digiwin.athena.integration.assembly.gmc.dto.commodity.CreateCommodityRequest;
import com.digiwin.athena.integration.assembly.gmc.dto.commodity.GmcResultDto;
import com.digiwin.athena.integration.assembly.gmc.dto.commodity.UpdateModulesRequest;
import com.digiwin.athena.service.assembly.CommodityService;
import com.digiwin.athena.utils.user.UserHelper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;

/**
 * GMC商品服务
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class GmcCommodityFacadeService implements CommodityService {

    private final AssemblyProp assemblyProp;
    private final AppTokenService appTokenService;
    private final IamApplicationFacadeService iamApplicationService;
    private final GmcCommodityApi commodityApi;

    /**
     * 同步解决方案计划
     * <br/>
     * 在GMC上创建应用，并设置基础信息
     *
     * @param firstPublish 是否首次发布
     * @param solutionPlan 解决方案计划对象
     */
    @Override
    public void syncSolutionPlan(boolean firstPublish, SolutionPlan solutionPlan) {
        try {
            // 获取IAM Token
            String iamToken = UserHelper.getCurrentIamToken();

            if (firstPublish) {
                createApplicationAndCommodity(solutionPlan, iamToken);
            }

            updateCommodity(iamToken, solutionPlan);
        } catch (Exception e) {
            log.error("解决方案[{}]发布时创建应用或商品失败: {}", solutionPlan.getName(), e.getMessage());
            throw new BusinessException("发布失败：" + e.getMessage(), e);
        }
    }

    protected void createApplicationAndCommodity(SolutionPlan solutionPlan, String iamToken) {
        // 2. 调用创建应用服务
        Long appSid = iamApplicationService.createApplication(iamToken, solutionPlan);

        // 只同步本区GMC
        List<CommodityMultiEnvSyncDTO> multiEnvSyncList = CommodityMultiEnvSyncDTO.buildList(
                assemblyProp.getGmcDomain(), iamToken, appSid);
        log.info("解决方案[{}]创建应用成功，共{}个区域", solutionPlan.getName(), multiEnvSyncList.size());

        // 3. 构建创建商品请求
        CreateCommodityRequest commodityRequest = CreateCommodityRequest
                .build(solutionPlan.getType(), assemblyProp.getProductCode());

        // 4. 调用创建商品服务
        invokeCreateCommodity(commodityRequest, multiEnvSyncList);
        log.info("解决方案[{}]创建商品成功", solutionPlan.getName());
    }


    protected void updateCommodity(String iamToken, SolutionPlan solutionPlan) {
        String appToken = appTokenService.findAppTokenByCode(assemblyProp.getAppToken(), iamToken, solutionPlan.getType());
        // 2. 获取各区域应用信息
        // 只同步本区GMC
        List<CommodityMultiEnvSyncDTO> appInfo = CommodityMultiEnvSyncDTO.buildList(assemblyProp.getGmcDomain(), iamToken);

        // 3. 构建更新模组请求（即使没有module，也更新描述）
        UpdateModulesRequest updateRequest = UpdateModulesRequest.build(
                solutionPlan.getType(),
                solutionPlan.getName(),
                appToken,
                new ArrayList<>()  // 空的模组列表
        );

        // 4. 调用更新商品模组服务
        invokeUpdateCommodity(updateRequest, appInfo);

        log.info("解决方案[{}]发布并更新商品成功", solutionPlan.getName());
    }

    /**
     * 创建商品
     *
     * @param request 创建商品请求
     * @param envList 各区域应用信息
     * @see com.digiwin.athena.service.application.impl.ApplicationServiceImpl#createCommodity
     */
    protected void invokeCreateCommodity(CreateCommodityRequest request,
                                         List<CommodityMultiEnvSyncDTO> envList) {

        for (CommodityMultiEnvSyncDTO envDto : envList) {
            AssemblyProp.EnvConfig envConfig = envDto.getEnvConfig();
            String iamToken = envDto.getToken();

            String gmcDomain = envConfig.getGmcDomain();
            String url = gmcDomain + GmcConstant.createCommodity;

            GmcResultDto<Boolean> responseObj = commodityApi.createCommodity(
                    url, assemblyProp.getAppToken(), iamToken, request);

            if (!responseObj.isSuccess()) {
                throw new BusinessException("GMC failed to create the product, error message:"
                        + responseObj.getErrorMessage());
            }
        }
    }

    /**
     * 更新商品模组
     *
     * @param request 更新模组请求
     * @param appInfo 各区域应用信息
     * @see com.digiwin.athena.service.application.impl.ApplicationServiceImpl#updateCommodityModule
     */
    protected void invokeUpdateCommodity(UpdateModulesRequest request,
                                         List<CommodityMultiEnvSyncDTO> appInfo) {

        for (CommodityMultiEnvSyncDTO response : appInfo) {
            AssemblyProp.EnvConfig envConfig = response.getEnvConfig();
            String token = response.getToken();

            String gmcUrl = envConfig.getGmcDomain() + GmcConstant.UPDATE_MODULES;

            GmcResultDto<Void> responseObj = commodityApi.updateCommodity(
                    gmcUrl, assemblyProp.getAppToken(), token, request);

            if (!responseObj.isSuccess()) {
                throw new BusinessException("更新商品基础信息失败!msg:" + responseObj.getErrorMessage());
            }
        }
    }


}
