package com.digiwin.athena.controller.asset;

import com.digiwin.athena.auth.annotation.FuncAuth;
import com.digiwin.athena.domain.pagedesign.TenantInfo;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.digiwin.athena.dto.ResultDto;
import com.digiwin.athena.dto.asset.*;
import com.digiwin.athena.mongodb.domain.application.AssetDefinition;
import com.digiwin.athena.service.asset.AssetCenter;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.constraints.NotEmpty;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

@RestController
@RequestMapping("/assetCenter")
@Tag(name = "资产中心", description = "a80fcf1b-8d00-4e23-be19-f0c0a88ded5d")
@Validated
public class AssetCenterController {

    @Autowired
    private AssetCenter assetCenter;

    @PostMapping("/assetCenterList/{type}")
    @Operation(summary = "资产中心列表",description = "type=all|DTD,all查询所有")
    public ResultDto<AssetPagination<AssetCenterInfoResDto>> assetCenterList(@RequestBody PageReqCondition<String> pageReqCondition,
                                                                             @PathVariable String type) {
        AssetPagination<AssetCenterInfoResDto> res =null;
        if ("all".equals(type)){
            res = assetCenter.getAssetCenterList(pageReqCondition,null);
        }else{
            res = assetCenter.getAssetCenterList(pageReqCondition, type);
        }
        return ResultDto.createSuccess(res);
    }

    @PostMapping("/designer/getDesignerAssetList")
    @Operation(summary = "设计器内查询资产列表")
    public ResultDto<AssetPagination<AssetDetailResDto>> getDesignerAssetList(@RequestBody PageReqCondition<DesignerAssetListReqDto> pageReqCondition){
        DesignerAssetListReqDto condition = pageReqCondition.getCondition();
        Assert.isTrue(!CollectionUtils.isEmpty(condition.getTypes()),"类型不能为空");

        AssetPagination<AssetDetailResDto> res = assetCenter.designerAssetList(pageReqCondition);
        return ResultDto.createSuccess(res);
    }

    @PostMapping("/saveAsset")
    @Operation(summary = "创建保存资产",description = "新增和保存，保存需要将assetId传递过来")
    public ResultDto<AssetDetailResDto> designerSaveAsset(@RequestBody CreateAssetReqDto createAssetReqDto){

        AssetDetailResDto assetDetailResDto = assetCenter.saveAsset(createAssetReqDto,null);
        return ResultDto.createSuccess(assetDetailResDto);
    }

    @PostMapping("/designerDeleteAsset")
    @Operation(summary = "删除资产")
    public ResultDto<Boolean> designerDeleteAsset(@RequestBody List<DeleteAssetReqDto> deleteAssetReqDtos){

        for (DeleteAssetReqDto deleteAssetReqDto : deleteAssetReqDtos) {
            assetCenter.deleteAsset(deleteAssetReqDto);
        }
        return ResultDto.createSuccess(true);
    }

    @PostMapping("/updateAssetBasicInfo")
    public ResultDto<Boolean> updateAssetBasicInfo(@RequestBody @Validated UpdateAssetBasicInfoReqDto updateAssetBasicInfoReqDto){
        assetCenter.updateAssetBasicInfo(updateAssetBasicInfoReqDto);
        return ResultDto.createSuccess(true);
    }

    @GetMapping("/assetDetail")
    @Operation(summary = "资产详情")
    public ResultDto<AssetDetailResDto> dtdAssetDetail(@Schema(description = "资产id") String assetId) {
        AssetDetailResDto res = assetCenter.getAsset(assetId);
        return ResultDto.createSuccess(res);
    }

    @GetMapping("/assetVersionList")
    @Operation(summary = "查询指定资产所有版本信息")
    public ResultDto<List<AssetInfoResDto>> dtdAssetVersionList(@Schema(description = "资产id") String assetId) {
        List<AssetInfoResDto> res = assetCenter.assetVersionList(assetId);
        return ResultDto.createSuccess(res);
    }

    @GetMapping("/application/removedAsset")
    @Operation(summary = "下架资产")
    @FuncAuth(resourceType = "application",resourceId = "1",actions = ":*")
    public ResultDto<Boolean> removedAsset(@NotEmpty(message = "资产id不能为空") String assetId,
                                           @NotEmpty(message = "应用code不能为空")String appCode) {
        return ResultDto.createSuccess(assetCenter.removedAsset(assetId));
    }


    @GetMapping("/removedAssetOpenTenant")
    @Operation(summary = "下架资产开发范围的指定租户")
    @FuncAuth(resourceType = "result",resourceId = "Var:$tenantId",actions = ":*")
    public ResultDto<Boolean> removedAssetOpenTenant(String assetId) {
        return ResultDto.createSuccess(assetCenter.removedAssetOpenTenant(assetId));
    }

    @GetMapping("/assetRelationAppInfo")
    @Operation(summary = "资产当前引用信息")
    public ResultDto<List<AssetRelationAppInfoResDto>> assetRelationAppInfo(@Schema(description = "资产id") String assetId,
                                                                            @Schema(description = "是否在资产中心") Boolean isAssetCenter) {
        List<AssetRelationAppInfoResDto> res = assetCenter.getAssetRefRelation(assetId,isAssetCenter);
        return ResultDto.createSuccess(res);
    }

    @GetMapping("/application/assetErrorCount")
    @Operation(summary = "应用内依赖开放异常数")
    public ResultDto<Integer> assetErrorCount(String appCode) {

        return ResultDto.createSuccess(assetCenter.assetErrorCount(appCode));
    }

    @PostMapping("/application/assetList/{appCode}/{type}")
    @Operation(summary = "应用内资产列表",description = "type=all|DTD,all查询所有")
    public ResultDto<Pagination<AssetCenterInfoResDto>> appAssetList(@RequestBody PageReqCondition<String> pageReqCondition,
                                                                        @PathVariable String appCode,
                                                                        @PathVariable String type) {
        Pagination<AssetCenterInfoResDto> res =null;
        if ("all".equals(type)){
            res = assetCenter.appAssetList(pageReqCondition,List.of(appCode), List.of());
        }else{
            res = assetCenter.appAssetList(pageReqCondition,List.of(appCode), List.of(type));
        }

        return ResultDto.createSuccess(res);
    }

    @PostMapping("/application/updateAssetDetail")
    @Operation(summary = "应用内更新资产详情")
    @FuncAuth(resourceType = "application",resourceId = "Json:0#$.application",actions = ":*")
    public ResultDto<Boolean> updateAssetDetail(@Validated @RequestBody AssetOpenReqDto assetOpenReqDto) {
        if (AssetDefinition.TENANT_RANGE.equals(assetOpenReqDto.getOpenRange())){
            List<TenantInfo> tenantInfos = assetOpenReqDto.getTenantInfos();
            Assert.isTrue(tenantInfos.size()>0,"需要指定开发的租户列表");

            for (TenantInfo tenantInfo : tenantInfos) {
                if (StringUtils.isEmpty(tenantInfo.getTenantId()) && StringUtils.isEmpty(tenantInfo.getTenantName())) {
                    throw new RuntimeException("开放指定的租户id 租户名称必填");
                }
            }
        }
        return ResultDto.createSuccess(assetCenter.updateAssetDefinition(assetOpenReqDto));
    }

    @PostMapping("/application/availableAsset")
    @Operation(summary = "应用内上架资产")
    @FuncAuth(resourceType = "application",resourceId = "Json:0#$.application",actions = ":*")
    public ResultDto<Boolean> availableAsset(@Validated @RequestBody AssetOpenReqDto assetOpenReqDto) {
        return ResultDto.createSuccess(assetCenter.availableAsset(assetOpenReqDto));
    }

    @PostMapping("/application/batchAvailableAsset")
    @Operation(summary = "应用内批量上架资产")
    @FuncAuth(resourceType = "application",resourceId = "Json:0#$.application",actions = ":*")
    public ResultDto<Boolean> batchAvailableAsset(@Validated @RequestBody AssetBatchOpenReqDto assetBatchOpenReqDto) {
        return ResultDto.createSuccess(assetCenter.batchAvailableAsset(assetBatchOpenReqDto));
    }

    @GetMapping("/application/assetInit")
    @Operation(summary = "应用内资产初始化",description = "type|ALL-初始化全部，DTD-初始化DTD")
    @FuncAuth(resourceType = "application",resourceId = "0",actions = ":*")
    public ResultDto<Boolean> dtdAssetInit(@NotEmpty(message = "应用code不能为空") String appCode,
                                           @NotEmpty(message = "type不能为空")String type) {
        assetCenter.assetInit(appCode,type);
        return ResultDto.createSuccess(true);
    }

    @PostMapping("/application/dependentAssetsInfo/{appCode}/{type}")
    @Operation(summary = "依赖引用的资产信息列表",description = "type=all|DTD,all查询所有")
    public ResultDto<Pagination<DependentAssetResDto>> dependentAssetsInfo(@RequestBody PageReqCondition<String> pageReqCondition,
                                                                           @PathVariable String appCode,
                                                                           @PathVariable String type) {
        Pagination<DependentAssetResDto> res = assetCenter.getDependAssetInfoList(pageReqCondition, appCode,"all".equals(type) ? null : type);
        return ResultDto.createSuccess(res);
    }



    @PostMapping("/application/dtdAssetAvailableList/{tabCategory}")
    @Operation(summary = "dtd使用列表", description = "tabCategory:team-团队|platform-平台")
    public ResultDto<Pagination<AssetDetailResDto>> dtdAssetAvailableList(@RequestBody PageReqCondition<String> pageReqCondition,
                                                                          @PathVariable String tabCategory) {

        Pagination<AssetDetailResDto> res;
        if ("team".equals(tabCategory)) {
            res = assetCenter.getSameTenantDtdAsset(pageReqCondition);

        } else {
            res = assetCenter.getPlatformDtdAsset(pageReqCondition);

        }
        return ResultDto.createSuccess(res);

    }

    @PostMapping("/application/batchSyncAsset")
    @Operation(summary = "批量更新资产数据")
    public ResultDto<Boolean> batchSyncAsset(@RequestBody List<String> assetIds) {
        if (CollectionUtils.isEmpty(assetIds)){
            throw new RuntimeException("选中资产不能为空");
        }
        return ResultDto.createSuccess(assetCenter.batchSyncAsset(assetIds));
    }

    @PostMapping("/application/appUseAssetList")
    @Operation(summary = "应用内使用资产列表")
    public ResultDto<Pagination<AssetCenterInfoResDto>> appUseAssetList(@RequestBody PageReqCondition<UseAssetListReqDto> pageReqCondition) {
        Pagination<AssetCenterInfoResDto> res = assetCenter.appUseAssetList(pageReqCondition);
        return ResultDto.createSuccess(res);
    }


    @GetMapping("/application/unbindingRelationship")
    @FuncAuth(resourceType = "application",resourceId = "2",actions = ":*")
    @Operation(summary = "解绑引用关系",description = "type=DTD")
    public ResultDto<Boolean> unbindingRelationship(@NotEmpty(message = "taskCode不能为空") String taskCodes,
                                                    @NotEmpty(message = "type不能为空") String type,
                                                    @NotEmpty(message = "appCode不能为空")String appCode){
        assetCenter.unbindingRelationship(taskCodes,type,appCode);
        return ResultDto.createSuccess(true);
    }

    @GetMapping("/assetCanCopyVersionList")
    @Operation(summary = "查询指定资产可复制版本信息")
    public ResultDto<List<AssetInfoResDto>> assetCanCopyVersionList(@Schema(description = "资产id") @NotEmpty(message = "资产id不能为空") String assetId) {
        List<AssetInfoResDto> res = assetCenter.assetCanCopyVersionList(assetId);
        return ResultDto.createSuccess(res);
    }

    @GetMapping("/getAssetDefinitionInfo")
    @Operation(summary = "根据业务code查询资产信息",description = "type=DTD")
    public ResultDto<AssetCenterInfoResDto> getAssetDefinitionInfo(@NotEmpty(message = "code不能为空") String code,
                                                                   @NotEmpty(message = "type不能为空") String type) {
        return ResultDto.createSuccess(assetCenter.getAssetDefinitionInfo(code,type));
    }

    @GetMapping("/getDepAssets")
    public ResultDto<List<DepAssetResDto>> getDepAssets(@NotEmpty(message = "资产id不能为空") String objId){
        List<DepAssetResDto> res = assetCenter.getDepAssets(objId);
        return ResultDto.createSuccess(res);
    }
}
