package com.digiwin.athena.datacollect.collector;

import cn.hutool.core.collection.CollUtil;
import com.digiwin.athena.datacollect.api.dto.AssetCollectApiResponse;
import com.digiwin.athena.datacollect.consumer.CollectDataConsumer;
import com.digiwin.athena.datacollect.context.CollectContext;
import com.digiwin.athena.datacollect.expression.ExpressionResolver;
import com.digiwin.athena.datacollect.model.CollectExecutorProp;
import com.digiwin.athena.datacollect.model.CollectResult;
import com.digiwin.athena.datacollect.model.PageResult;
import com.digiwin.athena.datacollect.script.ScriptExecutor;
import com.digiwin.athena.mongodb.domain.DataCollectConfig;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

/**
 * 资产数据采集器抽象基类
 * 实现步进式查询、分页查询、即时消费的通用逻辑
 */
@Slf4j
public abstract class BaseAssetDataCollector {

    protected ScriptExecutor scriptExecutor;
    protected ExpressionResolver expressionResolver;

    /**
     * 采集器类型标识
     */
    public abstract String getType();

    /**
     * 执行数据采集（核心方法）
     *
     * @param context  采集上下文（包含所有执行参数）
     * @param consumer 数据消费器（每页查询后立即消费）
     * @return 采集结果
     */
    public CollectResult collect(CollectContext context, CollectDataConsumer consumer) {
        LocalDateTime startTime = LocalDateTime.now();
        try {
            // 初始化配置（解析表达式）
            initContextExecutorParam(context);
            // 初始化查询参数
            context.begin();

            // 步进式时间循环（外层循环）
            while (context.hasNext()) {
                // TODO pzz performance 数据连续为空时，膨胀步进长度？
                log.info("开始步进查询: assetTypeKey={}, step={}, startTime={}, endTime={}",
                        context.getAssetType().getKey(),
                        context.getTotalSteps() + 1, context.getCurrentStepStartTime(), context.getCurrentStepEndTime());

                // loopParams循环（中层循环）
                List<Map<String, Object>> loopParams = context.getConfig().getExecutor().getLoopParams();
                if (CollUtil.isNotEmpty(loopParams)) {
                    // 从续查点开始循环
                    for (int loopIndex = context.getLoopIndex(); loopIndex < loopParams.size(); loopIndex++) {
                        Map<String, Object> loopParam = loopParams.get(loopIndex);
                        context.setLoopParam(loopParam);
                        context.setLoopIndex(loopIndex);

                        // 执行分页查询和消费
                        executePagingCollect(context, consumer);
                    }
                } else {
                    executePagingCollect(context, consumer);
                }

                context.step();
            }

            log.info("资产数据采集成功: assetTypeKey={}, steps={}, pages={}, totalCount={}, consumedCount={}",
                    context.getAssetType().getKey(), context.getTotalSteps(), context.getTotalPages(),
                    context.getTotalRecords(), context.getConsumedRecords());

            return CollectResult.buildSuccess(context, startTime);
        } catch (Exception e) {
            log.error("资产数据采集失败: assetTypeKey={}", context.getAssetType().getKey(), e);
            return CollectResult.buildFailed(e, context, startTime);
        }
    }


    /**
     * 执行分页查询和消费（内层循环）
     */
    protected void executePagingCollect(CollectContext context, CollectDataConsumer consumer) throws Exception {
        boolean hasMore = true;
        while (hasMore) {
            log.debug("查询第{}页: assetTypeKey={}, loopIndex={}",
                    context.getPageNo(), context.getAssetType().getKey(), context.getLoopIndex());

            // 调用子类实现的fetchPage方法
            PageResult<AssetCollectApiResponse.AssetDataItem> pageResult = fetchPage(context);

            // 如果有数据，立即消费
            if (!pageResult.isEmpty()) {
                // TODO pzz consume失败记录日志及数据信息，捕获异常，不打断采集流程
                int consumedCount = consumer.consume(context, pageResult.getData());
                context.addRecords(pageResult.getData().size(), consumedCount);

                log.debug("第{}页消费完成: fetchedCount={}, consumedCount={}",
                        context.getPageNo(), pageResult.getData().size(), consumedCount);
            }

            // 判断是否还有更多数据
            hasMore = pageResult.isHasMore();
            // 页码递增
            context.incrementPageNo();
        }
    }

    /**
     * 初始化Context（解析配置中的表达式）
     */
    protected void initContextExecutorParam(CollectContext context) {
        DataCollectConfig config = context.getConfig();
        if (config == null || config.getExecutor() == null) {
            return;
        }

        CollectExecutorProp executor = config.getExecutor();

        // 解析URL
        if (executor.getUrl() != null) {
            executor.setUrl(expressionResolver.resolve(executor.getUrl()));
        }

        // 解析Headers
        if (executor.getHeaders() != null) {
            executor.setHeaders(expressionResolver.resolve(executor.getHeaders()));
        }

        // 解析loopParams
        if (executor.getLoopParams() != null && !executor.getLoopParams().isEmpty()) {
            List<Map<String, Object>> mapList = expressionResolver.resolveList(executor.getLoopParams());
            executor.setLoopParams(mapList);
        }
    }


    /**
     * 抽象方法：分页查询
     * 子类实现具体的分页查询逻辑（如HTTP、数据库等）
     * 只负责查询一页数据，不负责分页循环
     *
     * @param context 采集上下文（包含当前页码、时间范围、loopParam等）
     * @return 分页结果（返回原始数据项，由上层负责转换）
     */
    protected abstract PageResult<AssetCollectApiResponse.AssetDataItem> fetchPage(CollectContext context) throws Exception;


    @Autowired
    public void setScriptExecutor(ScriptExecutor scriptExecutor) {
        this.scriptExecutor = scriptExecutor;
    }

    @Autowired
    public void setExpressionResolver(ExpressionResolver expressionResolver) {
        this.expressionResolver = expressionResolver;
    }
}
