package com.digiwin.athena.datacollect.expression;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.convert.Convert;
import cn.hutool.core.stream.CollectorUtil;
import cn.hutool.core.util.StrUtil;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.expression.MapAccessor;
import org.springframework.expression.Expression;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.common.TemplateParserContext;
import org.springframework.expression.spel.SpelCompilerMode;
import org.springframework.expression.spel.SpelParserConfiguration;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * Spring EL表达式解析器
 * 支持从配置文件中读取值，替换{@value #EXAMPLE}格式为实际变量值的表达式，
 */
@Slf4j
@Component
public class ExpressionResolver {
    public static final String EXPRESSION_PREFIX = "{{ ";
    public static final String EXPRESSION_SUFFIX = " }}";
    @SuppressWarnings("unused")
    private static final String EXAMPLE = EXPRESSION_PREFIX + "varName" + EXPRESSION_SUFFIX;

    protected final TemplateParserContext templateParserContext = new TemplateParserContext(EXPRESSION_PREFIX, EXPRESSION_SUFFIX);
    protected StandardEvaluationContext context;
    protected ExpressionParser parser;

    public String resolveToStr(String text, Object contextObj) {
        return StrUtil.toStringOrNull(resolveStr(text, contextObj));
    }

    /**
     * 解析字符串中的表达式
     * 支持格式: {@value #EXAMPLE}或直接的SpEL表达式
     *
     * @param text 包含表达式的文本
     * @return 解析后的文本
     */
    public Object resolveStr(String text, Object contextObj) {
        if (text == null || text.isEmpty()) {
            return text;
        }

        try {
            // 解析表达式
            Expression exp = getParser().parseExpression(text, templateParserContext);
            Object value = exp.getValue(context, contextObj);

            return value != null ? value : text;
        } catch (Exception e) {
            log.error("无法解析表达式: {}, 保持原值,msg:{}", text, e.getMessage());
            return text;
        }
    }

    public Map<String, Object> resolveMap(Map<String, Object> obj, Object contextObj, String... expressionKeys) {
        return resolve(obj, contextObj, null, expressionKeys);
    }

    public <T> List<T> resolveList(List<T> obj, Object contextObj, String... expressionKeys) {
        return resolve(obj, contextObj, null, expressionKeys);
    }

    public <T> T resolve(T obj, Object contextObj, BeanAccessor<T> beanAccessor, String... expressionKeys) {
        //noinspection unchecked
        Resolver<T> resolver = new Resolver<>(
                Convert.convert(Map.class, contextObj),
                beanAccessor,
                CollUtil.newHashSet(expressionKeys)
        );

        return resolver.resolve(obj);
    }

    public String wrapVarName(String varName) {
        if (varName == null) {
            return null;
        }

        varName = StrUtil.trim(varName);
        if (StrUtil.isBlank(varName)) {
            return varName;
        }

        return EXPRESSION_PREFIX + varName + EXPRESSION_SUFFIX;
    }

    protected ExpressionParser getParser() {
        if (parser != null) {
            return parser;
        }

        context = new StandardEvaluationContext();
        // 支持使用`.`访问map的entry
        context.addPropertyAccessor(new MapAccessor());


        SpelParserConfiguration config = new SpelParserConfiguration(SpelCompilerMode.MIXED,
                null, false, false, Integer.MAX_VALUE);
        parser = new SpelExpressionParser(config);

        return parser;
    }


    public static class BeanAccessor<T> {
        protected final List<FieldAccessor<T>> fieldAccessors = new ArrayList<>();

        /**
         * 创建新的字段映射配置
         */
        public static <T> BeanAccessor<T> create() {
            return new BeanAccessor<>();
        }

        public boolean accept(Object obj) {
            return fieldAccessors.stream()
                    .anyMatch(v -> v.accept(obj));
        }

        /**
         * 添加字段映射
         *
         * @param getter 获取表达式的getter方法引用
         * @param setter 设置结果的setter方法引用
         * @return this
         */
        public <R> BeanAccessor<T> addMapping(Class<T> entityClass, Function<T, String> getter, BiConsumer<T, R> setter) {
            return addMapping(entityClass, getter, setter, null);
        }

        /**
         * 添加字段映射
         *
         * @param getter 获取表达式的getter方法引用
         * @param setter 设置结果的setter方法引用
         * @return this
         */
        public <R> BeanAccessor<T> addMapping(Class<T> entityClass,
                                              Function<T, String> getter, BiConsumer<T, R> setter,
                                              Function<T, Object> childrenGetter) {
            fieldAccessors.add(new FieldAccessor<>(
                    entityClass,
                    getter,
                    (obj, value) -> {
                        //noinspection unchecked
                        setter.accept(obj, (R) value);
                    },
                    childrenGetter));
            return this;
        }

        /**
         * 字段映射内部类
         */
        @Getter
        public static class FieldAccessor<T> {
            protected final Function<T, String> expressionGetter;
            protected final BiConsumer<T, Object> resultSetter;
            protected final Function<T, Object> childrenGetter;
            protected final Class<T> entityClass;

            FieldAccessor(Class<T> entityClass, Function<T, String> expressionGetter, BiConsumer<T, Object> resultSetter, Function<T, Object> childrenGetter) {
                this.expressionGetter = expressionGetter;
                this.resultSetter = resultSetter;
                this.childrenGetter = childrenGetter;
                this.entityClass = entityClass;
            }

            public boolean accept(Object obj) {
                return entityClass.isAssignableFrom(obj.getClass());
            }

            public String getExpression(T obj) {
                return expressionGetter != null ? expressionGetter.apply(obj) : null;
            }

            public void consumeResult(T obj, Object value) {
                if (resultSetter != null) {
                    resultSetter.accept(obj, value);
                }
            }

            public Object getChildren(T obj) {
                return childrenGetter != null ? childrenGetter.apply(obj) : null;
            }
        }
    }

    @SuppressWarnings("unchecked")
    public class Resolver<T> {
        protected final Map<String, Object> contextObj;
        protected final BeanAccessor<T> beanAccessor;
        protected final Set<String> expressionKeys;

        public Resolver(Map<String, Object> contextObj, BeanAccessor<T> beanAccessor, Set<String> expressionKeys) {
            this.contextObj = contextObj;
            this.beanAccessor = beanAccessor;
            this.expressionKeys = expressionKeys;
        }

        /**
         * 解析Map中的表达式，递归解析list、map对象中的string
         *
         * @param map 包含表达式的Map
         * @return 解析后的Map
         */
        public Map<String, Object> resolveMap(Map<String, Object> map) {
            if (map == null || map.isEmpty()) {
                return map;
            }

            return map.entrySet().stream()
                    .filter(entry -> expressionKeys.contains(entry.getKey()) || CollUtil.isEmpty(expressionKeys))
                    .collect(CollectorUtil.toMap(
                            Map.Entry::getKey,
                            entry -> resolve(map, entry.getValue()),
                            (v1, v2) -> v1
                    ));
        }

        /**
         * 递归解析list、map对象中的string
         */
        public List<T> resolveList(List<T> list) {
            if (list == null || list.isEmpty()) {
                return list;
            }

            return list.stream()
                    .map(value -> resolve(list, value))
                    .map(v -> (T) v)
                    .collect(Collectors.toList());
        }

        public T resolve(T t) {
            if (t == null) {
                return null;
            }

            resolve(null, t);
            return t;
        }

        protected Object resolve(Object bean, Object value) {
            Object result;
            if (value instanceof String) {
                result = resolveStr((String) value, contextObj);
            } else if (value instanceof Map) {
                Map<String, Object> nestedMap = (Map<String, Object>) value;
                result = resolveMap(nestedMap);
            } else if (value instanceof List) {
                result = resolveList((List<T>) value);
            } else {
                result = value;
            }

            if (beanAccessor != null) {
                beanAccessor.fieldAccessors
                        .forEach(fieldAccessor -> {
                            if (!fieldAccessor.accept(value)) {
                                return;
                            }
                            String expression = fieldAccessor.getExpression((T) value);
                            if (expression != null) {
                                fieldAccessor.consumeResult((T) value, resolve(value, expression));
                            }

                            Object children = fieldAccessor.getChildren((T) value);
                            if (children != null) {
                                resolve(bean, children);
                            }
                        });
            }

            return result;
        }

    }
}
