package com.digiwin.athena.datacollect.job;

import cn.hutool.core.date.LocalDateTimeUtil;
import cn.hutool.core.thread.ThreadUtil;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.config.AssemblyProp;
import com.digiwin.athena.dao.mongodao.assetType.RAssetTypeMongoDao;
import com.digiwin.athena.dao.mongodao.datacollect.JobExecutionRecordMongoDao;
import com.digiwin.athena.datacollect.collector.BaseAssetDataCollector;
import com.digiwin.athena.datacollect.consumer.CollectDataConsumer;
import com.digiwin.athena.datacollect.context.CollectContext;
import com.digiwin.athena.datacollect.model.CollectResult;
import com.digiwin.athena.datacollect.model.JobExecData;
import com.digiwin.athena.mongodb.domain.DataCollectConfig;
import com.digiwin.athena.mongodb.domain.assetType.AssetType;
import com.digiwin.athena.mongodb.domain.datacollect.JobExecutionRecord;
import com.digiwin.athena.service.asset.AssetDataCollectService;
import jakarta.annotation.PostConstruct;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.quartz.Job;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.concurrent.*;
import java.util.stream.Collectors;

import static com.digiwin.athena.constants.AssemblyConstants.DataCollector.JOB_DETAIL_ASSET_TYPE_ID;

/**
 * TODO pzz 数据采集单独线程池，job异步执行，以防阻塞调度器
 * 资产数据采集Quartz任务
 * 负责任务调度、Context构建、结果记录
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class AssetDataCollectJob implements Job {
    private final RAssetTypeMongoDao rAssetTypeMongoDao;
    private final JobExecutionRecordMongoDao jobExecutionRecordMongoDao;
    private final AssetDataCollectService assetDataCollectService;
    private final AssemblyProp assemblyProp;
    private final List<BaseAssetDataCollector> collectors;

    private Map<String, BaseAssetDataCollector> collectorMap;
    private ThreadPoolExecutor executorService;
    private ScheduledExecutorService timeoutMonitorService;

    @PostConstruct
    public void init() {
        if (!Boolean.TRUE.equals(assemblyProp.getJob().getEnabled())) {
            log.debug("资产数据采集定时任务关闭！");
            return;
        }

        executorService = new ThreadPoolExecutor(
                0, assemblyProp.getJob().getMaxThreads(),
                60L, TimeUnit.SECONDS,
                new LinkedBlockingQueue<>(assemblyProp.getJob().getMaxRunJob()),
                ThreadUtil.createThreadFactoryBuilder()
                        .setNamePrefix("asset-dataCollector-")
                        .setUncaughtExceptionHandler((t, e) -> log.error("数据采集未捕获异常！msg:{}", e.getMessage(), e))
                        .build(),
                new ThreadPoolExecutor.AbortPolicy()
        );

        timeoutMonitorService = Executors.newScheduledThreadPool(
                1,
                ThreadUtil.createThreadFactoryBuilder()
                        .setNamePrefix("asset-timeout-monitor-")
                        .build()
        );
    }

    @Override
    public void execute(JobExecutionContext context) throws JobExecutionException {
        if (!Boolean.TRUE.equals(assemblyProp.getJob().getEnabled())) {
            return;
        }
        JobExecutor executor = new JobExecutor(context);

        try {
            // TODO pzz 修改任务成功后取消schedule定时
            Future<?> future = executorService.submit(executor);
            timeoutMonitorService.schedule(() -> {
                if (!future.isDone()) {
                    future.cancel(true);
                    log.warn("任务执行超时，打断任务: assetTypeKey={}", executor.getAssetType().getKey());
                }
            }, assemblyProp.getJob().getTimeout().toMinutes(), TimeUnit.MINUTES);
        } catch (RejectedExecutionException e) {
            log.error("任务执行被拒绝，线程池已满!最大值：{} 运行数：{}",
                    executorService.getQueue().size(), executorService.getTaskCount(), e);
        }
    }

    /**
     * 任务执行器内部类，负责具体执行逻辑
     */
    @Getter
    public class JobExecutor implements Runnable {
        private final String assetTypeId;
        private final String jobName;
        private final LocalDateTime fireTime;

        private AssetType assetType;
        private DataCollectConfig config;
        private JobExecutionRecord record;

        private CollectContext collectContext;
        private CollectResult result;

        public JobExecutor(JobExecutionContext context) {
            this.jobName = context.getJobDetail().getKey().getName();
            this.fireTime = LocalDateTimeUtil.of(context.getFireTime());
            this.assetTypeId = context.getJobDetail().getJobDataMap().getString(JOB_DETAIL_ASSET_TYPE_ID);
        }

        @Override
        public void run() {
            try {
                log.info("开始执行资产数据采集任务: jobName={}, assetTypeId={}", jobName, assetTypeId);
                init();

                executeDataCollection();

                record.complete(config, assetType, collectContext, result);
                log.info("资产数据采集任务执行完成: jobRecord：{}", record);
            } catch (Exception e) {
                log.error("资产数据采集任务执行失败: msg:{}, jobRecord{}", e.getMessage(), record, e);
                record.fail(collectContext, e.getMessage());
            } finally {
                jobExecutionRecordMongoDao.save(record);
            }
        }

        protected void init() {
            if (collectorMap == null) {
                collectorMap = collectors.stream()
                        .collect(Collectors.toMap(BaseAssetDataCollector::getType, c -> c));
            }
            loadAssetTypeAndConfig();

            this.record = JobExecutionRecord.initExecRecord(jobName, fireTime, assetType.getKey());
            buildContext();
        }

        protected void loadAssetTypeAndConfig() {
            assetType = rAssetTypeMongoDao.selectById(assetTypeId);
            if (assetType == null) {
                throw new BusinessException("资产类型不存在: " + assetTypeId);
            }

            config = DataCollectConfig.fromAssetType(assetType);
            if (config == null || config.getExecutor() == null) {
                throw new BusinessException("资产类型未配置数据采集: " + assetType.getType());
            }
        }

        protected void executeDataCollection() {
            String executorType = config.getExecutor().getType();
            BaseAssetDataCollector collector = collectorMap.get(executorType);
            if (collector == null) {
                throw new BusinessException("不支持的采集器类型: " + executorType);
            }

            CollectDataConsumer consumer = assetDataCollectService::batchProcessDataItems;
            // TODO pzz adjust 合并CollectResult到collectContext？
            result = collector.collect(collectContext, consumer);
        }

        protected void buildContext() {
            JobExecutionRecord lastExecRecord = jobExecutionRecordMongoDao.findLatestByBizKey(assetType.getKey());
            JobExecData presetExecData = lastExecRecord != null ? lastExecRecord.getNextExecData() : null;

            if (config.isConfigChanged(lastExecRecord)) {
                log.warn("检测到配置更新，结束当前任务并重新开始: assetTypeKey={}, oldConfigTime={}, newConfigTime={}",
                        assetType.getKey(), lastExecRecord != null ? lastExecRecord.obtainConfigUpdateTime() : null,
                        config.getConfigUpdateTime());
                presetExecData = null;
            }

            collectContext = CollectContext.buildCollectContext(config, assemblyProp.getCollector(),
                    assetType, presetExecData, fireTime);
        }
    }

}
