package com.digiwin.athena.service.assembly.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.convertor.BaseConverter;
import com.digiwin.athena.convertor.assembly.VariableConverter;
import com.digiwin.athena.datacollect.expression.ExpressionResolver;
import com.digiwin.athena.dto.assembly.variable.VarDefMapNode;
import com.digiwin.athena.dto.assembly.variable.VarDefNode;
import com.digiwin.athena.mongodb.domain.other.PresetData;
import com.digiwin.athena.service.assembly.DVariableService;
import com.digiwin.athena.sys.PresetDataServiceV2;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 设计器变量查询服务实现
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class DVariableServiceImpl implements DVariableService {
    public static final String PRESET_DATA_KEY_VAR_TREE = "Assembly_Var_Tree";

    protected final PresetDataServiceV2 presetDataServiceV2;
    protected final ExpressionResolver expressionResolver;

    @Override
    public List<VarDefNode> queryVariableTree(List<String> keys) {
        // 获取预设数据
        List<VarDefNode> varNodeList = getVarTreeList();
        // 如果keys为空，返回全部
        if (keys == null || keys.isEmpty()) {
            return varNodeList;
        }

        // 根据keys过滤
        List<VarDefNode> varDefList = varNodeList.stream()
                .filter(entry -> keys.contains(entry.getValue()))
                .collect(Collectors.toList());

        if (varNodeList.size() == 1) {
            // 如果只有一个配置，方便前端展示，则不展示第一个层级
            return varDefList.get(0).getChildren();
        }

        return varDefList;
    }

    protected List<VarDefNode> getVarTreeList() {
        // TODO pzz optimize 添加更新时间，查询时比对时间戳以此实现缓存，减少大对象查询
        PresetData presetData = presetDataServiceV2.getPresetDataObjectV2(PRESET_DATA_KEY_VAR_TREE);
        if (presetData == null || presetData.getObj() == null) {
            throw new BusinessException("未找到变量树配置数据！");
        }

        // 反序列化数据到Map
        List<VarDefNode> varNodeList = BaseConverter.INSTANCE.copyToList((Collection<?>) presetData.getObj(), VarDefNode.class);
        if (varNodeList == null || varNodeList.isEmpty()) {
            throw new BusinessException("未找到变量树配置数据！");
        }

        return varNodeList;
    }

    @Override
    public VarDefMapNode resolveVarDef(String key) {
        if (StrUtil.isBlank(key)) {
            return null;
        }

        List<VarDefNode> varTreeList = getVarTreeList();
        if (varTreeList == null || varTreeList.isEmpty()) {
            return null;
        }

        VarDefNode var = CollUtil.emptyIfNull(varTreeList).stream()
                .filter(v -> ObjectUtil.equal(v.getValue(), key))
                .findFirst().orElse(null);


        if (var == null) {
            return null;
        }

        VarDefMapNode mapNode = VariableConverter.INSTANCE.toMapNode(var);
        Map<String, VarDefMapNode> map = VariableConverter.INSTANCE.toMap(varTreeList);
        var beanAccessor = ExpressionResolver.BeanAccessor.<VarDefMapNode>create()
                .addMapping(
                        VarDefMapNode.class,
                        VarDefMapNode::getReference,
                        (node, v) -> {
                            if (v instanceof VarDefMapNode referenceNode) {
                                node.putReference(referenceNode);
                            } else {
                                log.error("变量树定义表达式解析后不为节点！nodeFullPath:{},expression:{},evalResult:{}", node.getFullPath(), node.getReference(), v);
                            }
                        },
                        VarDefMapNode::getChildren
                );

        return expressionResolver.resolve(mapNode, map, beanAccessor);
    }


    @Override
    public VarDefNode resolveVarDefList(String key) {
        return VariableConverter.INSTANCE.toDefNode(resolveVarDef(key));
    }

}
