package com.digiwin.athena.service.asset.impl;

import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.constants.enums.assembly.AssetCollectOperation;
import com.digiwin.athena.datacollect.api.dto.AssetCollectApiResponse;
import com.digiwin.athena.datacollect.context.CollectContext;
import com.digiwin.athena.dto.TenantInfo;
import com.digiwin.athena.dto.asset.CreateAssetReqDto;
import com.digiwin.athena.dto.asset.AssetReqDto;
import com.digiwin.athena.http.iam.dto.TenantInfoDto;
import com.digiwin.athena.http.iam.service.TenantInfoService;
import com.digiwin.athena.mongodb.domain.application.Asset;
import com.digiwin.athena.service.asset.AssetCenter;
import com.digiwin.athena.service.asset.AssetDataCollectService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 资产数据采集服务实现
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class AssetDataCollectServiceImpl implements AssetDataCollectService {

    private final AssetCenter assetCenterService;
    private final TenantInfoService tenantInfoService;


    @Override
    public int batchUpsertAssets(List<Asset> assets) {
        throw new UnsupportedOperationException();
    }

    @Override
    public int batchProcessDataItems(CollectContext context, List<AssetCollectApiResponse.AssetDataItem> dataItems) throws Exception {
        if (CollectionUtils.isEmpty(dataItems)) {
            return 0;
        }

        // 第二步：获取租户信息
        Map<String, TenantInfo> tenantInfoCache = new HashMap<>();

        int successCount = 0;
        for (AssetCollectApiResponse.AssetDataItem item : dataItems) {
            AssetCollectApiResponse.AssetData data = item.getData();
            if (data == null) {
                continue;
            }

            String opStr = item.obtainOp(context.getOpFromLoopParam());
            AssetCollectOperation operationType = AssetCollectOperation.of(opStr);
            if (operationType == null) {
                throw new BusinessException("资产采集操作类型不存在! op=" + opStr);
            }

            if (operationType.isDelete()) {
                processDeleteAssets(data);
            } else {
                saveAsset(tenantInfoCache, data);
            }

            successCount++;
        }

        return successCount;
    }

    private void saveAsset(Map<String, TenantInfo> tenantMap, AssetCollectApiResponse.AssetData data) {
        CreateAssetReqDto dto = null;
        try {
            // 非删除操作，转换为CreateAssetReqDto并提取租户信息
            dto = convertToCreateAssetReqDto(data);

            TenantInfo tenantInfo = tenantMap.computeIfAbsent(data.getAdpTenantId(),
                    k -> map(tenantInfoService.getTenantInfoWithCache(k)));

            assetCenterService.saveAsset(dto, tenantInfo);
        } catch (Exception e) {
            log.error("资产保存失败: asset={}, error={}", dto, e.getMessage());
            throw new RuntimeException(StrUtil.format("资产保存失败: assetCode={},assetType={}"
                    , data.getCode(), data.getType()), e);
        }
    }

    protected TenantInfo map(TenantInfoDto dto) {
        TenantInfo tenantInfo = new TenantInfo();

        tenantInfo.setTenantId(dto.getTenantId());
        tenantInfo.setTenantName(dto.getTenantName());
        tenantInfo.setSid(StrUtil.toStringOrNull(dto.getTenantSid()));

        return tenantInfo;
    }

    protected void processDeleteAssets(AssetCollectApiResponse.AssetData data) {
        try {
            assetCenterService.deleteAsset(List.of(buildDeleteDto(data)));
            log.info("资产删除成功: assetType Code={}", data.getKey());
        } catch (Exception e) {
            log.error("资产删除失败: assetType Code={}, error={}", data.getKey(), e.getMessage());
            throw new BusinessException("资产删除失败: assetType Code=" + data.getKey(), e);
        }
    }

    protected AssetReqDto buildDeleteDto(AssetCollectApiResponse.AssetData data) {
        AssetReqDto dto = new AssetReqDto();

        dto.setCode(data.getCode());
        dto.setType(data.getType());
        dto.setManageContext(data.getManageContext());

        return dto;
    }

    /**
     * 转换AssetData为CreateAssetReqDto
     */
    protected CreateAssetReqDto convertToCreateAssetReqDto(AssetCollectApiResponse.AssetData data) {
        CreateAssetReqDto dto = new CreateAssetReqDto();

        dto.setCode(data.getCode());
        dto.setType(data.getType());
        dto.setAssetName(data.getAssetName());
        dto.setAssetDesc(data.getAssetDesc());
        dto.setBusinessData(data.getBusinessData());
        dto.setManageContext(data.getManageContext());
        dto.setLang(data.getLang());
        return dto;
    }

}
