package com.digiwin.athena.mongodb.domain.application;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.http.Method;
import com.alibaba.fastjson.JSONObject;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.bo.assembly.SolutionPlan;
import com.digiwin.athena.constants.enums.assembly.OnlineState;
import com.digiwin.athena.dto.action.MultiLanguageDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanDetailDTO;
import com.digiwin.athena.mongodb.domain.BaseMongoEntity;
import io.swagger.v3.oas.annotations.media.Schema;
import lombok.Data;
import lombok.experimental.FieldNameConstants;
import org.bson.types.ObjectId;
import org.springframework.data.annotation.Id;
import org.springframework.data.mongodb.core.mapping.Document;
import org.springframework.data.mongodb.core.mapping.FieldName;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

/**
 * 解决方案定义
 */
@Data
@FieldNameConstants
@Document(collection = "applicationConfig")
public class ApplicationConfig implements BaseMongoEntity<String> {
    public static final String DEFAULT_ICON = "assets/img/home/enter5.png";
    public static final String CATEGORY_DEFAULT_ASSEMBLY = "6";
    public static final int NON_SORT_BASE = 1000;

    @Id
    private String objectId;

    /**
     *  解决方案编码，全局唯一
     */
    private String type;

    /**
     * app类型
     */
    private Integer appType;

    /**
     * 解决方案名称
     */
    private String name;

    private String version;                   // 版本号 service层暂时写死1.0
    private Integer versionSort;                  // 版本排序 version根据`.`分割，{第一部分}*1000+{第二部分}
    private String description;               // 描述
    private String assemblyType;              // 组装方式：dynamic(可组装)/static(不可组装)，枚举值
    /**
     * @see com.digiwin.athena.constants.enums.assembly.OnlineState#getCode
     */
    private String status;             // 解决方案状态：Draft(草稿)/Online(已上架)/Offline(未上架) 枚举值
    private String owner;                     // 归属方
    private List<SolutionPlanDetailDTO.DesignerBaseInfoDTO> designerList;  // 关联设计器列表
    private Map<String, Object> canvas;              // 布局配置
    private String layoutMode;                  // 布局方式
    private SolutionPlanDetailDTO.SsoConfigDTO ssoConfig;              // SSO配置(不可组装页面配置)

    /**
     * 全行业/数据分析/开发工具
     * 预留字段，支持path方式，对应前端的code
     */
    private String category;

    /**
     * 对应前端配置
     */
    private JSONObject page;

    /**
     * 前置条件组，一期只支持单一条件
     */
    private List<Precondition> precondition;

    /**
     * 预设模组
     */
    private IamConfig iamConfig;

    /**
     * 创建接口回调
     */
    private List<AppCallBack> appCreateCallback;

    /**
     * 更新接口回调
     */
    private List<AppCallBack> appUpdateCallback;

    /**
     * 删除接口回调
     */
    private List<AppCallBack> appDeleteCallback;

    /**
     * 收集接口配置(应用粒度)
     */
    private List<AppCallBack> collect;

    /**
     * 编辑器配置(应用粒度)
     */
    private List<AppCallBack> compile;

    /**
     * 发版接口配置(应用粒度)
     */
    private List<AppCallBack> publish;

    /**
     * 切版接口配置(应用粒度)
     */
    private List<AppCallBack> switched;

    /**
     * 是否为热门 0:热门卡片 1:非热门卡面
     */
    private Integer popular;

    /**
     * 首页热门卡片排序
     */
    private Integer popularSort;

    /**
     * 处理类
     */
    private String extraActionBeanName;

    /**
     * 模型初始化之前需要执行的脚本确定走哪个api生成的版本
     */
    private String beforeModelCreateScriptId;

    /**
     * 扩展配置信息
     */
    private JSONObject extProperties;

    /**
     * app创建url
     */
    private AppCallBack appCreate;

    /**
     * app更新url
     */
    private AppCallBack appUpdate;

    private Boolean isStdPublish;  // true走标准发布，对应的应用类型会在运维中心体现

    private Map<String, MultiLanguageDTO<String>> lang;              // 多语言配置

    /**
     * @see SolutionPlan#getMergeJson
     */
    private Map<String, Object> mergeJson;

    private LocalDateTime publishTime;         // 发布时间
    private String publishBy;                  // 发布人
    private LocalDateTime createTime;           // 创建时间
    private LocalDateTime editTime;             // 更新时间
    private String createBy;                  // 创建人
    private String editBy;                  // 更新人

    @Data
    public static class Precondition {

        /**
         * bean名称
         */
        private String beanName;

        /**
         * 方法名可不填，默认为judge
         */
        private String method = "judge";

        /**
         * 参数 对应OM商品id
         */
        private List<String> params;
    }

    @Data
    public static class IamConfig {

        /**
         * 模组相关配置
         */
        private List<Module> module;
    }

    @Data
    public static class AppCallBack {

        @Schema(description = "api类型，例如：http，groovy")
        private String type;

        @Schema(description = "url")
        private String url;

        @Schema(description = "当type是bean时，name是bean的name")
        private String name;

        @Schema(description = "请求类型：GET/POST")
        private Method method;

        @Schema(description = "头信息，json 字符串 k-v 格式")
        private String header;

        @Schema(description = "groovy脚本Id")
        private String scriptId;

        @Schema(description = "请求脚本")
        private String requestScript;

        @Schema(description = "响应脚本")
        private String responseScript;

        @Schema(description = "请求方法名称")
        private String methodName;

        private String createTaskScript;

        private String preScript;

        private Map<String,Object> headers;
        private String postScript;

        private Map<String,MultiLanguageDTO> label;
    }

    public void checkCanCreateApp() {
        if (!OnlineState.Online.getCode().equals(status)) {
            throw new BusinessException("解决方案未上架！");
        }
    }

    public void clearMergeJsonMap() {
        mergeJson.remove("id");
        mergeJson.remove(Fields.objectId);
        mergeJson.remove(com.digiwin.athena.mongodb.domain.application.ApplicationConfig.Fields.type);
        mergeJson.remove(com.digiwin.athena.mongodb.domain.application.ApplicationConfig.Fields.name);
        mergeJson.remove(com.digiwin.athena.mongodb.domain.application.ApplicationConfig.Fields.status);

        mergeJson.remove(com.digiwin.athena.mongodb.domain.application.ApplicationConfig.Fields.publishTime);
        mergeJson.remove(com.digiwin.athena.mongodb.domain.application.ApplicationConfig.Fields.publishBy);

        mergeJson.remove(com.digiwin.athena.mongodb.domain.application.ApplicationConfig.Fields.createTime);
        mergeJson.remove(com.digiwin.athena.mongodb.domain.application.ApplicationConfig.Fields.editTime);
        mergeJson.remove(com.digiwin.athena.mongodb.domain.application.ApplicationConfig.Fields.createBy);
        mergeJson.remove(com.digiwin.athena.mongodb.domain.application.ApplicationConfig.Fields.editBy);
    }

    public Map<String, Object> mergeToEntityMap() {
        Map<String, Object> entityMap = BeanUtil.beanToMap(this);

        clearMergeJsonMap();
        entityMap.putAll(mergeJson);
        mergeJson.remove("id");
        entityMap.remove(Fields.objectId);
        if (getId() != null) {
            entityMap.put(FieldName.ID.name(), new ObjectId(getId()));
        }

        return entityMap;
    }

    public String getId() {
        return objectId;
    }

    public void setId(String id) {
        this.objectId = id;
    }

}
