package com.digiwin.athena.service.assembly.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.convertor.BaseConverter;
import com.digiwin.athena.convertor.assembly.VariableConverter;
import com.digiwin.athena.datacollect.expression.ExpressionResolver;
import com.digiwin.athena.dto.assembly.variable.VarDefMapNode;
import com.digiwin.athena.dto.assembly.variable.VarDefNode;
import com.digiwin.athena.dto.assembly.variable.VarTreeNodeDTO;
import com.digiwin.athena.mongodb.domain.other.PresetData;
import com.digiwin.athena.service.assembly.DVariableService;
import com.digiwin.athena.sys.PresetDataServiceV2;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 设计器变量查询服务实现
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class DVariableServiceImpl implements DVariableService {
    public static final String PRESET_DATA_KEY_VAR_TREE = "Assembly_Var_Tree";

    protected final PresetDataServiceV2 presetDataServiceV2;
    protected final ExpressionResolver expressionResolver;

    @Override
    public List<VarTreeNodeDTO> queryVariableTree(List<String> keys) {
        if (CollUtil.isEmpty(keys)) {
            keys = Collections.singletonList(VarKey.Default.key());
        }

        List<VarDefMapNode> varDefMapNodes = resolveVarDef(keys);

        List<VarTreeNodeDTO> treeList = VariableConverter.INSTANCE.toDto(varDefMapNodes);
        if (treeList.size() == 1) {
            // 如果只有一个配置，方便前端展示，则不展示第一个层级
            return treeList.get(0).getChildren();
        }

        return treeList;
    }

    protected List<VarDefNode> getVarTreeList() {
        // TODO pzz optimize 添加更新时间，查询时比对时间戳以此实现缓存，减少大对象查询
        PresetData presetData = presetDataServiceV2.getPresetDataObjectV2(PRESET_DATA_KEY_VAR_TREE);
        if (presetData == null || presetData.getObj() == null) {
            throw new BusinessException("未找到变量树配置数据！");
        }

        // 反序列化数据到Map
        List<VarDefNode> varNodeList = BaseConverter.INSTANCE.copyToList((Collection<?>) presetData.getObj(), VarDefNode.class);
        if (varNodeList == null || varNodeList.isEmpty()) {
            throw new BusinessException("未找到变量树配置数据！");
        }

        return varNodeList;
    }

    @Override
    public VarDefMapNode resolveVarDef(String key) {
        if (StrUtil.isBlank(key)) {
            return null;
        }

        return CollUtil.getFirst(resolveVarDef(Collections.singletonList(key)));
    }

    @Override
    public List<VarDefMapNode> resolveVarDef(List<String> keys) {
        if (CollUtil.isEmpty(keys)) {
            return Collections.emptyList();
        }

        // TODO pzz optimize 存储为map，方便解析，之后转为list
        List<VarDefNode> varTreeList = getVarTreeList();
        Map<String, VarDefMapNode> varTreeMap = VariableConverter.INSTANCE.toMap(varTreeList);
        List<VarDefNode> varDefList = CollUtil.emptyIfNull(varTreeList).stream()
                .filter(v -> keys.contains(v.getValue()))
                .collect(Collectors.toList());

        if (CollUtil.isEmpty(varDefList)) {
            return Collections.emptyList();
        }

        return varDefList.stream()
                .map(var -> resolveVarDef(var, varTreeMap))
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    protected VarDefMapNode resolveVarDef(VarDefNode var, Map<String, VarDefMapNode> varTreeMap) {
        if (var == null) {
            return null;
        }

        VarDefMapNode mapNode = VariableConverter.INSTANCE.toMapNode(var);
        var beanAccessor = ExpressionResolver.BeanAccessor.<VarDefMapNode>create()
                .addMapping(
                        VarDefMapNode.class,
                        VarDefMapNode::getReference,
                        (node, v) -> {
                            if (v instanceof VarDefMapNode referenceNode) {
                                node.putReference(referenceNode);
                            } else {
                                log.error("变量树定义表达式解析后不为节点！nodeFullPath:{},expression:{},evalResult:{}", node.getFullPath(), node.getReference(), v);
                            }
                        },
                        VarDefMapNode::getChildren
                );

        return expressionResolver.resolve(mapNode, varTreeMap, beanAccessor);
    }


    @Override
    public VarDefNode resolveVarDefList(String key) {
        return VariableConverter.INSTANCE.toDefNode(resolveVarDef(key));
    }


    public enum VarKey {
        Default,
        SolutionPlan_Sso,
        ;

        public String key() {
            return this.name();
        }
    }
}
