package com.digiwin.athena.service.assembly.impl;

import cn.hutool.core.collection.CollUtil;
import com.digiwin.athena.base.BusinessException;
import com.digiwin.athena.bo.assembly.Designer;
import com.digiwin.athena.bo.assembly.SolutionPlan;
import com.digiwin.athena.config.AssemblyProp;
import com.digiwin.athena.constants.AssemblyErrorCode;
import com.digiwin.athena.convertor.assembly.DesignerConvertor;
import com.digiwin.athena.convertor.assembly.SolutionConvertor;
import com.digiwin.athena.dao.mongodao.assembly.DSolutionMongoDao;
import com.digiwin.athena.dao.mongodao.assembly.RSolutionMongoDao;
import com.digiwin.athena.dto.PageReqCondition;
import com.digiwin.athena.dto.Pagination;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanDetailDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanListDTO;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanPageQo;
import com.digiwin.athena.dto.assembly.solution.SolutionPlanSaveDTO;
import com.digiwin.athena.http.iam.service.AppService;
import com.digiwin.athena.service.SyncRuntime;
import com.digiwin.athena.service.assembly.CommodityService;
import com.digiwin.athena.service.assembly.DSolutionPlanService;
import com.digiwin.athena.service.assembly.RDesignerService;
import com.digiwin.athena.utils.user.UserHelper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * 解决方案管理服务实现
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class DSolutionPlanServiceImpl extends SyncRuntime<SolutionPlan> implements DSolutionPlanService {

    private final DSolutionMongoDao dSolutionMongoDao;
    private final RSolutionMongoDao rSolutionMongoDao;
    private final CommodityService commodityService;
    private final AssemblyProp assemblyProp;
    private final AppService appService;
    private RDesignerService rDesignerService;
    private DSolutionPlanServiceImpl proxyThis;


    @Override
    public Pagination<SolutionPlanListDTO> findList(PageReqCondition<SolutionPlanPageQo> pageReqCondition) {
        Pagination<SolutionPlanListDTO> page = dSolutionMongoDao.selectPage(pageReqCondition);
        fillNewestDesignerList(page.getData());
        return page;
    }

    @Override
    public Map<String, List<SolutionPlanListDTO>> findBaseListByDesignerId(Iterable<String> designerIdColl) {
        if (CollUtil.isEmpty(designerIdColl)) {
            return Collections.emptyMap();
        }

        Map<String, List<SolutionPlanListDTO>> map = dSolutionMongoDao.findListByDesignerId(designerIdColl);
        map.forEach((s, list) -> SolutionPlanListDTO.sort(list));
        return map;
    }

    @Transactional(rollbackFor = Throwable.class)
    @Override
    public SolutionPlan save(SolutionPlanSaveDTO designer) {
        SolutionPlan bo = SolutionConvertor.INSTANCE.toBo(designer);

        save(bo);

        return bo;
    }

    @Transactional(rollbackFor = Throwable.class)
    @Override
    public void save(SolutionPlan updated) {
        log.info("开始创建设计时态解决方案: id:{},updatedName:{}", updated.getId(), updated.getName());

        // 验证type字段格式
        updated.validateTypeFormat();

        // 验证唯一性
        checkUnique(updated);

        String objectId = updated.getId();
        if (objectId != null) {
            SolutionPlan existed = findExist(objectId);

            existed.overwriteUnmodifiable(updated);
        } else {
            Integer nextAppType = getNextAppType();

            updated.setAppType(nextAppType);
            updated.prepareNew();
        }

        dSolutionMongoDao.save(updated);

        log.info("成功创建设计时态解决方案: {}", updated.getId());
    }

    protected Integer getNextAppType() {
        // 修改为redis或者加锁？
        return dSolutionMongoDao.getNextAppType(SolutionPlan.ASSEMBLY_BASE_APP_TYPE);
    }

    /**
     * 单纯保存，不做任何业务处理
     */
    protected void pureSave(SolutionPlan bo) {
        dSolutionMongoDao.save(bo);
    }

    protected void checkUnique(SolutionPlan bo) {
        if (!validateUnique(bo.getType(), null, bo.getId())) {
            throw new BusinessException(AssemblyErrorCode.NAME_CODE_NOT_UNIQUE, "类型已存在！");
        }

        if (!validateUnique(null, bo.getName(), bo.getId())) {
            throw new BusinessException(AssemblyErrorCode.NAME_CODE_NOT_UNIQUE, "名称已存在！");
        }
    }

    protected SolutionPlan findExist(String objectId) {
        return Optional
                .ofNullable(dSolutionMongoDao.selectById(objectId))
                .orElseThrow(() -> new BusinessException(AssemblyErrorCode.SOLUTION_NOT_EXIST, "解决方案不存在！"));
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void deleteById(String objectId) {
        log.info("开始删除设计时态解决方案: {}", objectId);

        // 检查解决方案是否存在
        SolutionPlan bo = findExist(objectId);
        bo.checkDelete(bo);

        // 删除
        dSolutionMongoDao.deleteById(objectId);
        log.info("成功删除设计时态解决方案: {}", objectId);
    }

    @Override
    public SolutionPlan publish(String objId) {
        SolutionPlan solutionPlan = proxyThis.findById(objId);
        boolean firstPublish = solutionPlan.isFirstPublish();

        solutionPlan = super.publish(objId, solutionPlan);

        commodityService.syncSolutionPlan(firstPublish, solutionPlan);

        return solutionPlan;
    }


    @Override
    public SolutionPlanDetailDTO findDetailById(String objectId) {
        SolutionPlanDetailDTO detail = SolutionConvertor.INSTANCE.toDto(findById(objectId));

        // 填充writableDesignerList
        fillWritableDesignerList(detail);

        return detail;
    }

    /**
     * 填充可产生资产的关联设计器列表
     *
     * @param detail 解决方案详情DTO
     */
    public void fillWritableDesignerList(SolutionPlanDetailDTO detail) {
        // 提取设计器ID列表
        // 可能会出现设计器被删除查不到的情况
        List<Designer> designers = rDesignerService.getDesigners(detail.obtainDesignerIds());
        List<Designer> writeDesignerList = Designer.filterWritable(designers);

        detail.setDesignerList(DesignerConvertor.INSTANCE.toBaseInfoDTOList(designers));
        detail.setWritableDesignerList(DesignerConvertor.INSTANCE.toBaseInfoDTOList(writeDesignerList));
    }

    public void fillNewestDesignerList(List<SolutionPlanListDTO> list) {
        List<String> designerIds = SolutionPlanListDTO.obtainDesignerIds(list);
        var designerMap = DesignerConvertor.INSTANCE.toBaseInfoMap(rDesignerService.getDesignerMap(designerIds));

        list.forEach(detail -> detail.setDesignerList(
                CollUtil.map(
                        detail.getDesignerList(),
                        v -> designerMap.get(v.getId()), true)
        ));
    }

    @Override
    public SolutionPlan findById(String objectId) {
        log.debug("查询设计时态解决方案详情: {}", objectId);
        return dSolutionMongoDao.selectById(objectId);
    }


    @Override
    public boolean validateUnique(String code, String name, String excludeObjectId) {
        return !dSolutionMongoDao.existsByKey(code, name, excludeObjectId);
    }

    @Override
    public boolean checkIamExists(String code) {
        String userToken = UserHelper.getCurrentIamToken();
        boolean exists = appService.checkAppExist(assemblyProp.getAppToken(), userToken, code);

        log.info("校验解决方案设计器是否存在，type: {}, exists: {}", code, exists);
        return exists;
    }

    @Override
    protected void saveDesignerTimeData(SolutionPlan syncData) {
        pureSave(syncData);
    }

    @Override
    protected void insertRunTimeData(SolutionPlan syncData) {
        rSolutionMongoDao.save(syncData);
    }

    @Override
    protected void saveRunTimeData(SolutionPlan syncData) {
        rSolutionMongoDao.save(syncData);
    }

    @Override
    protected SolutionPlan getDesignerTimeData(String objId) {
        return proxyThis.findById(objId);
    }

    @Override
    protected SolutionPlan getRunTimeData(String objId) {
        return rSolutionMongoDao.selectById(objId);
    }

    @Override
    protected void deleteRuntimeData(String objId) {
        rSolutionMongoDao.deleteById(objId);
    }

    @Autowired
    @Lazy
    public void setProxyThis(DSolutionPlanServiceImpl proxyThis) {
        this.proxyThis = proxyThis;
    }

    @Autowired
    @Lazy
    public void setRDesignerService(RDesignerService rDesignerService) {
        this.rDesignerService = rDesignerService;
    }
}